import json
import pandas as pd
import matplotlib.pyplot as plt
from pathlib import Path

# Définition du répertoire de sortie
OUTPUT_DIR = Path("output_meteo")

def get_aqi_color(aqi_value):
    """
    Retourne la couleur de l'AQI selon les spécifications européennes.
    """
    if aqi_value < 20:
        return 'lightblue'
    elif aqi_value < 40:
        return 'green'
    elif aqi_value < 60:
        return 'yellow'
    elif aqi_value < 80:
        return 'red'
    elif aqi_value <= 100:
        return 'sienna' # Marron
    else:
        return 'purple'

def generate_aqi_reports(file_path):
    """
    Charge, traite les données AQI, génère des histogrammes colorés JPG 
    par localité dans 'output_meteo', et crée un fichier HTML de visualisation.
    """
    # Création du répertoire de sortie s'il n'existe pas
    OUTPUT_DIR.mkdir(exist_ok=True)
    
    try:
        with open(file_path, 'r') as f:
            data = json.load(f)
    except Exception as e:
        print(f"Erreur lors du chargement ou du décodage du fichier JSON: {e}")
        return

    # --- 1. Extraction et Calcul des Moyennes Journalières ---
    processed_data = {}

    for locality_name, locality_data in data.items():
        if locality_name == 'config':
            continue

        try:
            aqi_block = locality_data.get('aqi_data')
            if aqi_block and 'hourly' in aqi_block:
                hourly_data = aqi_block['hourly']
                
                df = pd.DataFrame({
                    'Time': pd.to_datetime(hourly_data['time']),
                    'AQI': hourly_data['european_aqi']
                }).dropna(subset=['AQI']).set_index('Time')
                
                daily_aqi_mean = df['AQI'].resample('D').mean().dropna()
                
                if not daily_aqi_mean.empty:
                    processed_data[locality_name] = daily_aqi_mean
                
        except Exception:
            continue

    if not processed_data:
        print("Aucune donnée AQI valide trouvée à traiter.")
        return

    # Liste pour stocker les chemins des images pour le HTML
    image_paths = []
    
    # --- 2. Génération et Sauvegarde des Histogrammes Verticaux ---
    for locality_name, daily_series in processed_data.items():
        plt.figure(figsize=(10, 6))
        
        # Déterminer la couleur de chaque barre
        bar_colors = [get_aqi_color(aqi) for aqi in daily_series.values]
        
        dates_str = daily_series.index.strftime('%Y-%m-%d')
        plt.bar(dates_str, daily_series.values, color=bar_colors, edgecolor='black', linewidth=0.5)

        # Titre et étiquettes
        title = (f"Qualité de l'Air (EAQI) Journalier - {locality_name}")
        plt.title(title, fontsize=14)
        plt.xlabel("Date", fontsize=12)
        plt.ylabel("AQI Européen Moyen", fontsize=12)
        plt.xticks(rotation=45, ha='right')
        
        plt.ylim(0, daily_series.max() * 1.1)
        plt.grid(axis='y', linestyle='--', alpha=0.6)
        plt.tight_layout()

        # Chemin et nom du fichier JPG
        image_name = f"aqi_histogramme_simplifie_{locality_name.replace(' ', '_')}.jpg"
        image_path = OUTPUT_DIR / image_name
        
        plt.savefig(image_path, format='jpg', dpi=300)
        plt.close()
        
        image_paths.append(image_name) # Stocker le nom pour le HTML

    # --- 3. Génération du Fichier HTML Responsive (Colonne Unique) ---
    html_content = f"""
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Rapport AQI Météo - {OUTPUT_DIR.name}</title>
    <style>
        body {{
            font-family: Arial, sans-serif;
            margin: 0;
            padding: 20px;
            background-color: #f4f4f4;
        }}
        .container {{
            max-width: 800px; /* Limite la largeur de la colonne */
            margin: 0 auto;
            padding: 20px;
            background-color: white;
            box-shadow: 0 0 10px rgba(0, 0, 0, 0.1);
        }}
        h1 {{
            text-align: center;
            color: #333;
        }}
        .report-section {{
            margin-bottom: 30px;
            padding-bottom: 20px;
            border-bottom: 1px solid #ddd;
        }}
        img {{
            width: 100%; /* Rend l'image responsive (s'adapte au conteneur) */
            height: auto;
            display: block;
            margin-top: 15px;
            border: 1px solid #ccc;
            border-radius: 5px;
        }}
        /* Légende des couleurs pour référence */
        .color-legend {{
            margin-top: 20px;
            padding: 10px;
            background-color: #eee;
            border-radius: 5px;
            font-size: 0.9em;
        }}
        .color-item {{
            display: flex;
            align-items: center;
            margin-bottom: 5px;
        }}
        .color-box {{
            width: 20px;
            height: 20px;
            margin-right: 10px;
            border: 1px solid #333;
        }}
    </style>
</head>
<body>
    <div class="container">
        <h1>Rapport d'Évolution Journalière de l'AQI</h1>
        <p><a href="index.html">⬅️ Retour à l'Index</a></p> 
        <p>Les graphiques ci-dessous montrent l'Indice de Qualité de l'Air (AQI) moyen journalier pour chaque localité. Les couleurs représentent les niveaux d'AQI européen (EAQI).</p>

        <div class="color-legend">
            <strong>Légende des Couleurs EAQI :</strong>
            <div class="color-item"><div class="color-box" style="background-color: lightblue;"></div> Bon (0-20)</div>
            <div class="color-item"><div class="color-box" style="background-color: green;"></div> Acceptable (20-40)</div>
            <div class="color-item"><div class="color-box" style="background-color: yellow;"></div> Modéré (40-60)</div>
            <div class="color-item"><div class="color-box" style="background-color: red;"></div> Mauvais (60-80)</div>
            <div class="color-item"><div class="color-box" style="background-color: sienna;"></div> Très Mauvais (80-100)</div>
            <div class="color-item"><div class="color-box" style="background-color: purple;"></div> Extrêmement Mauvais (>100)</div>
        </div>
"""
    
    # Ajout des images au contenu HTML
    for image_name in image_paths:
        # Extraire le nom de la localité à partir du nom du fichier
        # (ex: aqi_histogramme_simplifie_Rieulay.jpg -> Rieulay)
        locality_display_name = image_name.split('_')[-1].split('.')[0].replace('_', ' ')
        
        html_content += f"""
        <div class="report-section">
            <h2>{locality_display_name}</h2>
            <img src="{image_name}" alt="Histogramme AQI journalier pour {locality_display_name}">
        </div>
"""

    html_content += """
     <footer>
            <p><a href="index.html">⬅️ Retour à l'Index</a></p> 
        </footer>
    </div>
</body>
</html>
"""
    
    # Sauvegarde du fichier HTML
    html_file_path = OUTPUT_DIR / "aqi_reports.html"
    with open(html_file_path, 'w', encoding='utf-8') as f:
        f.write(html_content)
        
    print(f"\n--- TOUS LES RAPPORTS SONT PRÊTS ---")
    print(f"Le répertoire de sortie est: {OUTPUT_DIR.resolve()}")
    print(f"Le fichier HTML a été généré à: {html_file_path.name}")
    print(f"Images générées: {len(image_paths)}")


def main():
    generate_aqi_reports('all_data.json')

if __name__ == "__main__":
    main()
