import json
import os
from datetime import datetime
import numpy as np
import matplotlib.pyplot as plt
import matplotlib.dates as mdates
import locale

# Tente de définir la locale pour le français pour aider, mais la solution manuelle est plus fiable
try:
    locale.setlocale(locale.LC_ALL, 'fr_FR.utf8')
except locale.Error:
    try:
        locale.setlocale(locale.LC_ALL, 'fra')
    except locale.Error:
        pass # La traduction manuelle prend le relai si la locale n'est pas trouvée


# --- CONSTANTES DE FICHIER ET CONFIGURATION ---
DATA_FILE_PATH = "all_data.json" # Fichier de données de l'étape de scraping

# Dictionnaires de traduction pour les dates (Simplifié pour la robustesse)
# MONDAY=0, SUNDAY=6 (utilisé par date_obj.weekday())
FRENCH_DAYS_ABBREV = ['Lun', 'Mar', 'Mer', 'Jeu', 'Ven', 'Sam', 'Dim']

# Codes WMO pour la description météo (inchangé)
WMO_CODES_MAP = {
    0: "Ciel clair", 1: "Principalement dégagé", 2: "Partiellement nuageux", 3: "Couvert",
    45: "Brouillard", 48: "Brouillard givrant",
    51: "Bruine légère", 53: "Bruine modérée", 55: "Bruine dense",
    56: "Bruine verglaçante légère", 57: "Bruine verglaçante dense",
    61: "Pluie légère", 63: "Pluie modérée", 65: "Pluie forte",
    66: "Pluie verglaçante légère", 67: "Pluie verglaçante forte",
    71: "Neige légère", 73: "Neige modérée", 75: "Neige forte",
    77: "Neige en grains",
    80: "Averses de pluie légères", 81: "Averses de pluie modérées", 82: "Averses de pluie violentes",
    85: "Averses de neige légères", 86: "Averses de neige fortes",
    95: "Orage (faible à modéré)", 96: "Orage avec grêle légère", 99: "Orage avec grêle forte",
}

# (CSS_STYLE, INDEX_TEMPLATE, LOCALITY_TEMPLATE - inchangés pour ce besoin)

CSS_STYLE = """
body { font-family: sans-serif; margin: 0; background-color: #f4f4f9; color: #333; }
.container { max-width: 1000px; margin: auto; background: #fff; padding: 20px; border-radius: 0; box-shadow: none; min-height: 100vh; }
@media (min-width: 768px) {
    .container { margin: 20px auto; border-radius: 8px; box-shadow: 0 4px 8px rgba(0,0,0,0.1); min-height: auto; }
}

header { border-bottom: 2px solid #5d5dff; padding-bottom: 10px; margin-bottom: 20px; text-align: center; }
header p a { color: #5d5dff; text-decoration: none; font-weight: bold;}
header p a:hover, footer p a:hover { text-decoration: underline; }
h1, h2, h3, h4 { color: #5d5dff; }

/* Amélioration des onglets pour la responsivité */
.tab { overflow-x: auto; white-space: nowrap; border-bottom: 1px solid #ccc; padding-bottom: 5px; }
.tablinks { 
    background-color: #e6e6ff; border: none; outline: none; cursor: pointer; padding: 10px 14px; 
    transition: 0.3s; font-size: 15px; margin: 0 2px 5px 0; border-radius: 4px; display: inline-block; 
}
.tablinks:hover { background-color: #ccccff; }
.tablinks.active { background-color: #5d5dff; color: white; }
.forecast-day { padding: 10px 0; border-top: none; }

/* Responsivité et style des metadata (coordonnées/climatologie globale) */
.metadata { 
    display: flex; 
    justify-content: space-around; 
    flex-wrap: wrap; 
    margin-bottom: 20px;
}
.metadata div { 
    padding: 10px; border: 1px solid #eee; margin: 5px; border-radius: 4px; 
    background-color: #f9f9f9; 
    flex-grow: 1; /* Permet aux éléments de grandir pour remplir l'espace */
    min-width: 150px; /* Assure un minimum de lisibilité */
    text-align: center;
}
@media (max-width: 600px) {
    .metadata div { 
        min-width: 100%; 
        margin: 5px 0; 
        box-sizing: border-box; 
    }
}

/* Styles pour AQI */
.aqi-container {
    padding: 15px;
    border: 1px solid #ccc;
    border-radius: 8px;
    margin-bottom: 20px;
}
.aqi-summary {
    display: flex;
    flex-direction: column; /* Passage en colonne sur petit écran */
    align-items: center;
    justify-content: space-between;
    padding-bottom: 15px;
    border-bottom: 1px solid #ddd;
    text-align: center;
}
@media (min-width: 600px) {
    .aqi-summary {
        flex-direction: row; /* Retour en ligne sur grand écran */
    }
}
.aqi-value {
    font-weight: bold;
    font-size: 1.1em;
}
.aqi-good { background-color: #ccffcc; border-left: 5px solid #009900; }
.aqi-moderate { background-color: #ffffcc; border-left: 5px solid #ffcc00; }
.aqi-unhealthy-sensitive { background-color: #ffcccc; border-left: 5px solid #ff0000; }
.aqi-unhealthy { background-color: #ff99cc; border-left: 5px solid #990066; }
.aqi-hazardous { background-color: #ffcc99; border-left: 5px solid #663300; }

/* Styles pour les graphiques et images */
.chart-block img {
    max-width: 100%;
    height: auto; /* Garantit la responsivité des images */
    display: block;
    margin: 10px 0;
}
.chart-block {
    overflow-x: auto; /* Permet de scroller si le graphique est trop grand (rare avec max-width: 100% mais utile) */
}

/* Amélioration de la liste de liens d'index */
.locality-list {
    list-style: none;
    padding: 0;
}
.locality-list li {
    margin-bottom: 10px;
}
.locality-list a {
    display: block; /* Rendre toute la zone cliquable */
    padding: 15px;
    background-color: #e6e6ff;
    border-radius: 6px;
    text-decoration: none;
    color: #5d5dff;
    font-weight: bold;
    transition: background-color 0.3s;
    font-size: 1.1em; /* Augmentation de la taille pour améliorer la clarté */
    border-left: 5px solid #5d5dff;
}
.locality-list a:hover {
    background-color: #ccccff;
    border-left: 5px solid #3d3dcf;
}

footer { border-top: 1px solid #ccc; padding-top: 10px; margin-top: 20px; text-align: center; }
"""

INDEX_TEMPLATE = """
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0"> <title>Index Météo</title>
    <link rel="stylesheet" href="style.css">
</head>
<body>
    <div class="container">
        <header>
            <h1>Rapports Météo</h1>
            <p>Dernière mise à jour : {last_updated}</p>
            <p><a href="https://open-meteo.com/" target="_blank">Weather data by Open-Meteo.com</a></p>
        </header>
        <h2>Localités Disponibles</h2>
        <ul class="locality-list"> {locality_links}
        </ul>
        <hr>
        <ul class="locality-list"> <a href="aqi_reports.html">Qualité de l'air</a>
        </ul>
    </div>
</body>
</html>
"""

LOCALITY_TEMPLATE = """
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0"> <title>Météo pour {locality_name}</title>
    <link rel="stylesheet" href="style.css">
    <script>
        function openForecastDay(evt, dayId) {{
            var i, tabcontent, tablinks;
            tabcontent = document.getElementsByClassName("forecast-day");
            for (i = 0; i < tabcontent.length; i++) {{
                tabcontent[i].style.display = "none";
            }}
            tablinks = document.getElementsByClassName("tablinks");
            for (i = 0; i < tablinks.length; i++) {{
                tablinks[i].className = tablinks[i].className.replace(" active", "");
            }}
            document.getElementById(dayId).style.display = "block";
            evt.currentTarget.className += " active";
        }}
        window.onload = function() {{
            document.querySelector('.tablinks').click();
        }}
    </script>
</head>
<body>
    <div class="container">
        <header>
            <h1>Rapport Météo pour {locality_name}</h1>
            <div class="metadata"> <div>Coordonnées : Lat. {latitude}, Lon. {longitude}</div>
                <div>Fuseau : {timezone} ({timezone_abbreviation})</div>
            </div>
            <p><a href="index.html">⬅️ Retour à l'Index</a></p> 
            <p>Dernière mise à jour : {last_updated}</p>
            <p><a href="https://open-meteo.com/" target="_blank">Weather data by Open-Meteo.com</a></p>
        </header>

        {aqi_section}
        
        <hr>
        <h2>Prévisions sur {num_forecast_days} Jours</h2>
        
        <div class="chart-block" style="margin-bottom: 20px;">
            <h3>Température et Précipitations</h3>
            {overall_forecast_chart_tp}
        </div>
        
        <div class="chart-block" style="margin-bottom: 20px;">
            <h3>Pression</h3>
            {overall_forecast_chart_p}
        </div>
        
        <div class="chart-block" style="margin-bottom: 20px;">
            <h3>Vent Max</h3>
            {overall_forecast_chart_w}
        </div>


        <hr>
        <h2>Détail Horaire par Jour</h2>
        <div class="tab">
            {forecast_tabs}
        </div>
        <div style="margin-top: 10px;">
            {forecast_content}
        </div>
        
        <hr>
        {climatology_section}
        
        <footer>
            <p><a href="index.html">⬅️ Retour à l'Index</a></p> 
        </footer>
    </div>
</body>
</html>
"""

# --- FONCTIONS UTILITAIRES ET GRAPHIQUES ---

def apply_french_formatting(ax, fig, interval_type):
    """
    Applique le formatage des dates en français sur les axes Matplotlib.
    
    Correction: Utilise date_obj.weekday() pour garantir la traduction des noms 
    de jours abrégés en français, sans dépendre de la locale de Matplotlib.
    """
    
    def format_fr_date(x, pos=None):
        date_obj = mdates.num2date(x)
        day_index = date_obj.weekday() # 0=Lundi, 6=Dimanche
        
        # Le nom du jour abrégé en français est récupéré directement
        french_day_abbrev = FRENCH_DAYS_ABBREV[day_index]
        
        if interval_type == 'overall':
            # Format: 'Lun 02/11 12h'
            date_str_suffix = date_obj.strftime('%d/%m %Hh')
            translated_str = f"{french_day_abbrev} {date_str_suffix}"
            
        elif interval_type == 'climatology' or interval_type == 'aqi':
            # Format: 'Lun 02/11'
            date_str_suffix = date_obj.strftime('%d/%m')
            translated_str = f"{french_day_abbrev} {date_str_suffix}"
            
        else: # interval_type == 'daily'
            # Format: '12:00'
            return date_obj.strftime('%H:%M')

        return translated_str

    if interval_type == 'overall':
        # On définit un DateFormatter simple, mais on s'assure que format_data est utilisé
        date_formatter = mdates.DateFormatter('%d/%m %Hh') 
        date_formatter.format_data = format_fr_date 
        ax.xaxis.set_major_formatter(date_formatter)
        ax.xaxis.set_major_locator(mdates.DayLocator(interval=1)) 
        ax.xaxis.set_minor_locator(mdates.HourLocator(interval=6))
        
    elif interval_type == 'daily':
        ax.xaxis.set_major_formatter(mdates.DateFormatter('%H:%M'))
        ax.xaxis.set_major_locator(mdates.HourLocator(interval=3))
        
    elif interval_type == 'climatology' or interval_type == 'aqi':
        # On définit un DateFormatter simple, mais on s'assure que format_data est utilisé
        date_formatter = mdates.DateFormatter('%d/%m')
        date_formatter.format_data = format_fr_date 
        ax.xaxis.set_major_formatter(date_formatter)
        if interval_type == 'climatology':
            ax.xaxis.set_major_locator(mdates.DayLocator(interval=5))
        elif interval_type == 'aqi':
            ax.xaxis.set_major_locator(mdates.DayLocator(interval=1))
        
    fig.autofmt_xdate(rotation=45)


def generate_aqi_chart_png(locality_name, aqi_data, output_dir):
    """Génère un graphique de l'évolution quotidienne de l'AQI."""
    daily_data = aqi_data.get('daily', {})
    aqi_values = daily_data.get('european_aqi', [])
    times_str = daily_data.get('time', [])
    
    if not aqi_values or not times_str:
        return None

    valid_data = [(times_str[i], aqi) for i, aqi in enumerate(aqi_values) if aqi is not None]
    
    if not valid_data:
        return None
        
    dates = [datetime.fromisoformat(t).date() for t, _ in valid_data]
    aqi_values = [aqi for _, aqi in valid_data]
    
    filename_base = f"{locality_name.replace(' ', '_').lower()}_aqi_trend"
    chart_dir = os.path.join(output_dir, "charts")
    
    fig, ax = plt.subplots(figsize=(10, 5))
    
    # Style de tracé mis à jour : plein avec marqueur 'o', taille de ligne et marqueur ajustés
    ax.plot(dates, aqi_values, marker='o', linestyle='-', color='#8e44ad', label='AQI Européen', linewidth=2.5, markersize=6)
    
    # Remplissage de zones de couleur pour indiquer les niveaux 
    ax.axhspan(0, 50, color='green', alpha=0.1, label='Bon (0-50)')
    ax.axhspan(50, 100, color='yellow', alpha=0.1, label='Modéré (51-100)')
    ax.axhspan(100, 150, color='red', alpha=0.1, label='Sensible (101-150)')
    
    ax.set_ylabel("Indice AQI", color='#8e44ad')
    ax.tick_params(axis='y', labelcolor='#8e44ad')
    ax.set_xlabel("Jour", fontsize=12)
    ax.grid(True, linestyle='--', alpha=0.6)

    # Application du formatage français à l'axe X
    apply_french_formatting(ax, fig, 'aqi')
    
    ax.legend(loc='upper right', frameon=True)
    
    fig.tight_layout()
    path = os.path.join(chart_dir, f"{filename_base}.png")
    fig.savefig(path, bbox_inches='tight')
    plt.close(fig)
    
    return f"charts/{filename_base}.png"

def generate_overall_forecast_charts_png(locality_name, hourly_data, config, output_dir):
    """Génère les graphiques de prévisions pour toute la période."""
    
    num_forecast_hours = config.get("fore_days", 7) * 24
    
    times_str = hourly_data.get('time', [])[-num_forecast_hours:]
    times = [datetime.fromisoformat(t) for t in times_str]
    
    temps = np.array([v if v is not None else np.nan for v in hourly_data.get('temperature_2m', [])[-num_forecast_hours:]])
    precips = np.array([v if v is not None else np.nan for v in hourly_data.get('precipitation', [])[-num_forecast_hours:]])
    pressures = np.array([v if v is not None else np.nan for v in hourly_data.get('pressure_msl', [])[-num_forecast_hours:]])
    wind_gusts = np.array([v if v is not None else np.nan for v in hourly_data.get('wind_gusts_10m', [])[-num_forecast_hours:]])
    
    if not times_str:
        return {}

    filename_base = f"{locality_name.replace(' ', '_').lower()}_overall_forecast"
    chart_dir = os.path.join(output_dir, "charts")
    
    chart_paths = {}

    # ----------------------------------------------------
    # Graphique 1 : Température et Précipitations (T/P) 
    # ----------------------------------------------------
    fig1, ax1_temp = plt.subplots(figsize=(10, 5)) 

    color_temp = '#e74c3c'  
    color_precip = '#3498db' 
    
    # Style de tracé mis à jour : plein avec marqueur 'o', taille de ligne et marqueur ajustés
    line_temp = ax1_temp.plot(times, temps, marker='o', linestyle='-', color=color_temp, label='Température (°C)', linewidth=2.5, markersize=6)
    ax1_temp.set_ylabel("Température (°C)", color=color_temp)
    ax1_temp.tick_params(axis='y', labelcolor=color_temp)
    ax1_temp.spines["left"].set_color(color_temp)
    ax1_temp.grid(True, linestyle='--', alpha=0.6, which='major', axis='x')

    ax1_precip = ax1_temp.twinx()
    bar_precip = ax1_precip.bar(times, precips, width=1/24, color=color_precip, alpha=0.5, label='Précipitations (mm)')
    ax1_precip.set_ylabel("Précipitations (mm)", color=color_precip)
    ax1_precip.tick_params(axis='y', labelcolor=color_precip)
    ax1_precip.spines["right"].set_color(color_precip)
    ax1_precip.set_ylim(bottom=0)
    
    ax1_temp.set_xlabel("Date et Heure", fontsize=12)
    
    apply_french_formatting(ax1_temp, fig1, 'overall')

    handles1, labels1 = ax1_temp.get_legend_handles_labels()
    handles2, labels2 = ax1_precip.get_legend_handles_labels()
    ax1_temp.legend(handles1 + handles2, labels1 + labels2, loc='upper left', bbox_to_anchor=(0, 1.15), ncol=2, frameon=True)

    fig1.tight_layout()
    path_tp = os.path.join(chart_dir, f"{filename_base}_temp_precip.png")
    fig1.savefig(path_tp, bbox_inches='tight')
    plt.close(fig1)
    chart_paths['temp_precip'] = f"charts/{filename_base}_temp_precip.png"

    # ----------------------------------------------------
    # Graphique 2 : Pression (P) 
    # ----------------------------------------------------
    fig2, ax2_pressure = plt.subplots(figsize=(10, 5)) 

    color_pressure = '#f1c40f' 
    
    # Style de tracé mis à jour : plein avec marqueur 'o', taille de ligne et marqueur ajustés
    ax2_pressure.plot(times, pressures, marker='o', linestyle='-', color=color_pressure, label='Pression (hPa)', linewidth=2.5, markersize=6)
    ax2_pressure.set_ylabel("Pression (hPa)", color=color_pressure)
    ax2_pressure.tick_params(axis='y', labelcolor=color_pressure)
    ax2_pressure.spines["left"].set_color(color_pressure)
    ax2_pressure.grid(True, linestyle='--', alpha=0.6, which='major', axis='x')
    
    ax2_pressure.set_xlabel("Date et Heure", fontsize=12)
    
    apply_french_formatting(ax2_pressure, fig2, 'overall')

    ax2_pressure.legend(loc='upper left', bbox_to_anchor=(0, 1.08), frameon=True)

    fig2.tight_layout()
    path_p = os.path.join(chart_dir, f"{filename_base}_pressure.png")
    fig2.savefig(path_p, bbox_inches='tight')
    plt.close(fig2)
    chart_paths['pressure'] = f"charts/{filename_base}_pressure.png"
    
    # ----------------------------------------------------
    # Graphique 3 : Vent Max (W) 
    # ----------------------------------------------------
    fig3, ax3_wind = plt.subplots(figsize=(10, 5)) 

    color_wind = '#2ecc71' 
    
    # Style de tracé mis à jour : plein avec marqueur 'o', taille de ligne et marqueur ajustés
    ax3_wind.plot(times, wind_gusts, marker='o', linestyle='-', color=color_wind, label='Vent Max (km/h)', linewidth=2.5, markersize=6)
    ax3_wind.set_ylabel("Vent Max (km/h)", color=color_wind)
    ax3_wind.tick_params(axis='y', labelcolor=color_wind)
    ax3_wind.spines["left"].set_color(color_wind)
    ax3_wind.grid(True, linestyle='--', alpha=0.6, which='major', axis='x')
    ax3_wind.set_ylim(bottom=0)
    
    ax3_wind.set_xlabel("Date et Heure", fontsize=12)
    
    apply_french_formatting(ax3_wind, fig3, 'overall')

    ax3_wind.legend(loc='upper left', bbox_to_anchor=(0, 1.08), frameon=True)

    fig3.tight_layout()
    path_w = os.path.join(chart_dir, f"{filename_base}_wind.png")
    fig3.savefig(path_w, bbox_inches='tight')
    plt.close(fig3)
    chart_paths['wind_max'] = f"charts/{filename_base}_wind.png"
    
    return chart_paths


def generate_daily_forecast_charts_png(locality_name, day_key, hourly_data_for_day, output_dir):
    """Génère les graphiques de prévisions pour la journée (Temp/Pluie, Pression, Vent Max)."""
    chart_dir = os.path.join(output_dir, "charts")
    filename_base = f"{locality_name.replace(' ', '_').lower()}_{day_key}"
    
    times_str = hourly_data_for_day['hourly']['time']
    times = [datetime.fromisoformat(t) for t in times_str] 
    
    temps = np.array([v if v is not None else np.nan for v in hourly_data_for_day['hourly']['temperature_2m']])
    precips = np.array([v if v is not None else np.nan for v in hourly_data_for_day['hourly']['precipitation']])
    pressures = np.array([v if v is not None else np.nan for v in hourly_data_for_day['hourly']['pressure_msl']])
    wind_gusts = np.array([v if v is not None else np.nan for v in hourly_data_for_day['hourly']['wind_gusts_10m']])
    
    chart_paths = {}
    
    # ----------------------------------------------------
    # Graphique 1 : Température et Précipitations (T/P) 
    # ----------------------------------------------------
    fig1, ax1_temp = plt.subplots(figsize=(10, 5))

    color_temp = '#e74c3c'
    color_precip = '#3498db' 
    
    # Style de tracé mis à jour : plein avec marqueur 'o', taille de ligne et marqueur ajustés
    line_temp = ax1_temp.plot(times, temps, marker='o', linestyle='-', color=color_temp, label='Température (°C)', linewidth=2.5, markersize=6)
    ax1_temp.set_ylabel("Température (°C)", color=color_temp)
    ax1_temp.tick_params(axis='y', labelcolor=color_temp)
    ax1_temp.spines["left"].set_color(color_temp)
    ax1_temp.grid(True, linestyle='--', alpha=0.6, which='major', axis='x')

    ax1_precip = ax1_temp.twinx()
    bar_precip = ax1_precip.bar(times, precips, width=1/24, color=color_precip, alpha=0.5, label='Précipitations (mm)')
    ax1_precip.set_ylabel("Précipitations (mm)", color=color_precip)
    ax1_precip.tick_params(axis='y', labelcolor=color_precip)
    ax1_precip.spines["right"].set_color(color_precip)
    ax1_precip.set_ylim(bottom=0)
    
    ax1_temp.set_xlabel("Heure", fontsize=12)
    
    apply_french_formatting(ax1_temp, fig1, 'daily')

    handles1, labels1 = ax1_temp.get_legend_handles_labels()
    handles2, labels2 = ax1_precip.get_legend_handles_labels()
    ax1_temp.legend(handles1 + handles2, labels1 + labels2, loc='upper left', bbox_to_anchor=(0, 1.15), ncol=2, frameon=True)


    fig1.tight_layout()
    path_tp = os.path.join(chart_dir, f"{filename_base}_daily_temp_precip.png")
    fig1.savefig(path_tp, bbox_inches='tight')
    plt.close(fig1)
    chart_paths['temp_precip'] = f"charts/{filename_base}_daily_temp_precip.png"
    
    # ----------------------------------------------------
    # Graphique 2 : Pression (P) 
    # ----------------------------------------------------
    fig2, ax2_pressure = plt.subplots(figsize=(10, 5))

    color_pressure = '#f1c40f'
    
    # Style de tracé mis à jour : plein avec marqueur 'o', taille de ligne et marqueur ajustés
    ax2_pressure.plot(times, pressures, marker='o', linestyle='-', color=color_pressure, label='Pression (hPa)', linewidth=2.5, markersize=6)
    ax2_pressure.set_ylabel("Pression (hPa)", color=color_pressure)
    ax2_pressure.tick_params(axis='y', labelcolor=color_pressure)
    ax2_pressure.spines["left"].set_color(color_pressure)
    ax2_pressure.grid(True, linestyle='--', alpha=0.6, which='major', axis='x')

    ax2_pressure.set_xlabel("Heure", fontsize=12)
    
    apply_french_formatting(ax2_pressure, fig2, 'daily')

    ax2_pressure.legend(loc='upper left', bbox_to_anchor=(0, 1.08), frameon=True)

    fig2.tight_layout()
    path_p = os.path.join(chart_dir, f"{filename_base}_daily_pressure.png")
    fig2.savefig(path_p, bbox_inches='tight')
    plt.close(fig2)
    chart_paths['pressure'] = f"charts/{filename_base}_daily_pressure.png"
    
    # ----------------------------------------------------
    # Graphique 3 : Vent Max (W) 
    # -----------------------------------------------
    fig3, ax3_wind = plt.subplots(figsize=(10, 5))

    color_wind = '#2ecc71'
    
    # Style de tracé mis à jour : plein avec marqueur 'o', taille de ligne et marqueur ajustés
    ax3_wind.plot(times, wind_gusts, marker='o', linestyle='-', color=color_wind, label='Vent Max (km/h)', linewidth=2.5, markersize=6)
    ax3_wind.set_ylabel("Vent Max (km/h)", color=color_wind)
    ax3_wind.tick_params(axis='y', labelcolor=color_wind)
    ax3_wind.spines["left"].set_color(color_wind)
    ax3_wind.grid(True, linestyle='--', alpha=0.6, which='major', axis='x')
    ax3_wind.set_ylim(bottom=0)
    
    apply_french_formatting(ax3_wind, fig3, 'daily')

    ax3_wind.legend(loc='upper left', bbox_to_anchor=(0, 1.08), frameon=True)

    fig3.tight_layout()
    path_w = os.path.join(chart_dir, f"{filename_base}_daily_wind.png")
    fig3.savefig(path_w, bbox_inches='tight')
    plt.close(fig3)
    chart_paths['wind_max'] = f"charts/{filename_base}_daily_wind.png"
    
    return chart_paths


def generate_climatology_charts_png(locality_name, climatology_stats, output_dir):
    """Génère trois graphiques climatologiques (31 jours passés)."""
    daily_stats = climatology_stats['daily']
    if not daily_stats:
        return {} 

    daily_stats_sorted = sorted(daily_stats, key=lambda x: x['date'])

    dates_str = [d['date'] for d in daily_stats_sorted]
    dates = [datetime.strptime(d, "%Y-%m-%d").date() for d in dates_str]
    
    tmax = np.array([d['tmax'] if d['tmax'] != 'N/D' else np.nan for d in daily_stats_sorted])
    precip = np.array([d['precip'] if d['precip'] != 'N/D' else np.nan for d in daily_stats_sorted])
    
    tmin = np.array([d['tmin'] if d['tmin'] != 'N/D' else np.nan for d in daily_stats_sorted])
    pmin = np.array([d['pmin'] if d['pmin'] != 'N/D' else np.nan for d in daily_stats_sorted])
    pmax = np.array([d['pmax'] if d['pmax'] != 'N/D' else np.nan for d in daily_stats_sorted])
    wmax = np.array([d['wmax'] if d['wmax'] != 'N/D' else np.nan for d in daily_stats_sorted])

    filename_base = f"{locality_name.replace(' ', '_').lower()}_climatology"
    chart_dir = os.path.join(output_dir, "charts")
    chart_paths = {}
    
    
    # ----------------------------------------------------
    # Graphique 1 : Tmax, Tmin et Précipitations (MODIFIÉ)
    # ----------------------------------------------------
    fig_tp, ax_temp = plt.subplots(figsize=(10, 5))
    
    color_tmax = '#e74c3c'
    color_tmin = '#3498db'
    color_precip = '#7f8c8d'

    # Tmax - Tracé plein avec marqueur 'o', taille de ligne et marqueur ajustés
    ax_temp.plot(dates, tmax, marker='o', linestyle='-', color=color_tmax, label='Temp Max (°C)', linewidth=2.5, markersize=6)
    # Tmin - Tracé plein avec marqueur 'o', taille de ligne et marqueur ajustés (AJOUTÉ)
    ax_temp.plot(dates, tmin, marker='o', linestyle='-', color=color_tmin, label='Temp Min (°C)', linewidth=2.5, markersize=6)
    
    ax_temp.set_ylabel("Température (°C)")
    ax_temp.tick_params(axis='y')
    ax_temp.grid(True, linestyle='--', alpha=0.6, which='major', axis='x')

    ax_precip = ax_temp.twinx()
    bar_precip = ax_precip.bar(dates, precip, color=color_precip, alpha=0.6, label='Précipitations (mm)')
    ax_precip.set_ylabel("Précipitations (mm)", color=color_precip)
    ax_precip.tick_params(axis='y', labelcolor=color_precip)
    ax_precip.spines["right"].set_color(color_precip)
    ax_precip.set_ylim(bottom=0)
    
    apply_french_formatting(ax_temp, fig_tp, 'climatology') 

    handles1, labels1 = ax_temp.get_legend_handles_labels()
    handles2, labels2 = ax_precip.get_legend_handles_labels()
    # Mise à jour du nombre de colonnes dans la légende
    ax_temp.legend(handles1 + handles2, labels1 + labels2, loc='upper left', bbox_to_anchor=(0, 1.15), ncol=3, frameon=True)

    # Mise à jour du nom du fichier pour refléter Tmin
    path_tp = os.path.join(chart_dir, f"{filename_base}_tmax_tmin_precip.png")
    fig_tp.savefig(path_tp, bbox_inches='tight')
    plt.close(fig_tp)
    chart_paths['tmax_tmin_precip'] = f"charts/{filename_base}_tmax_tmin_precip.png"
    
    
    # ----------------------------------------------------
    # Graphique 2 : Pression Min et Max (Style mis à jour)
    # ----------------------------------------------------
    fig_pminmax, ax_press = plt.subplots(figsize=(10, 5))
    
    color_pmax = '#f1c40f'
    color_pmin = '#e67e22' 

    # Style de tracé mis à jour : plein avec marqueur 'o', taille de ligne et marqueur ajustés
    ax_press.plot(dates, pmin, marker='o', linestyle='-', color=color_pmin, label='Pression Min (hPa)', linewidth=2.5, markersize=6)
    ax_press.plot(dates, pmax, marker='o', linestyle='-', color=color_pmax, label='Pression Max (hPa)', linewidth=2.5, markersize=6)
    
    ax_press.set_ylabel("Pression (hPa)", color=color_pmax)
    ax_press.tick_params(axis='y', labelcolor=color_pmax)
    ax_press.spines["left"].set_color(color_pmax)
    ax_press.grid(True, linestyle='--', alpha=0.6, which='major', axis='x')

    apply_french_formatting(ax_press, fig_pminmax, 'climatology') 

    ax_press.legend(loc='upper left', bbox_to_anchor=(0, 1.15), ncol=2, frameon=True)

    path_pminmax = os.path.join(chart_dir, f"{filename_base}_pminmax.png")
    fig_pminmax.savefig(path_pminmax, bbox_inches='tight')
    plt.close(fig_pminmax)
    chart_paths['pminmax'] = f"charts/{filename_base}_pminmax.png"
    
    
    # ----------------------------------------------------
    # Graphique 3 : Vent Max (Wmax) (Style mis à jour)
    # ----------------------------------------------------
    fig_w, ax_wind = plt.subplots(figsize=(10, 5))
    
    color_wmax = '#2ecc71'

    # Style de tracé mis à jour : plein avec marqueur 'o', taille de ligne et marqueur ajustés
    ax_wind.plot(dates, wmax, marker='o', linestyle='-', color=color_wmax, label='Vent Max (km/h)', linewidth=2.5, markersize=6)
    ax_wind.set_ylabel("Vent Max (km/h)", color=color_wmax)
    ax_wind.tick_params(axis='y', labelcolor=color_wmax)
    ax_wind.spines["left"].set_color(color_wmax)
    ax_wind.grid(True, linestyle='--', alpha=0.6, which='major', axis='x')
    ax_wind.set_ylim(bottom=0)
    
    apply_french_formatting(ax_wind, fig_w, 'climatology')
    
    ax_wind.legend(loc='upper left', bbox_to_anchor=(0, 1.08), frameon=True)

    path_w = os.path.join(chart_dir, f"{filename_base}_wind.png")
    fig_w.savefig(path_w, bbox_inches='tight')
    plt.close(fig_w)
    chart_paths['wind_max'] = f"charts/{filename_base}_wind.png"

    return chart_paths

def calculate_climatology(hourly_data, config):
    """Calcule les statistiques climatologiques quotidiennes sur les jours passés."""
    
    past_days = config.get("past_days", 31)
    
    hourly = hourly_data
    if not hourly or not hourly.get('time'):
        return {'daily': [], 'global': {}}

    num_past_hours = past_days * 24
    
    times_str = hourly.get('time', [])[-num_past_hours:]
    temps = hourly.get('temperature_2m', [])[-num_past_hours:]
    precips = hourly.get('precipitation', [])[-num_past_hours:]
    pressures = hourly.get('pressure_msl', [])[-num_past_hours:]
    wind_gusts = hourly.get('wind_gusts_10m', [])[-num_past_hours:]
    
    daily_stats = {}

    for i, t_str in enumerate(times_str):
        try:
            time_obj = datetime.fromisoformat(t_str)
            day_key = time_obj.date().isoformat()
        except ValueError:
            continue

        if day_key not in daily_stats:
            daily_stats[day_key] = {
                'temps': [], 'precips': [], 'pressures': [], 'winds': [], 
                'tmin': np.inf, 'tmax': -np.inf, 'precip': 0, 'pmin': np.inf, 'pmax': -np.inf, 'wmax': -np.inf, 'date': day_key
            }

        t = temps[i] if temps[i] is not None else np.nan
        p = precips[i] if precips[i] is not None else np.nan
        pr = pressures[i] if pressures[i] is not None else np.nan
        w = wind_gusts[i] if wind_gusts[i] is not None else np.nan

        if not np.isnan(t):
            daily_stats[day_key]['temps'].append(t)
            daily_stats[day_key]['tmin'] = min(daily_stats[day_key]['tmin'], t)
            daily_stats[day_key]['tmax'] = max(daily_stats[day_key]['tmax'], t)

        if not np.isnan(p):
            daily_stats[day_key]['precip'] += p

        if not np.isnan(pr):
            daily_stats[day_key]['pmin'] = min(daily_stats[day_key]['pmin'], pr)
            daily_stats[day_key]['pmax'] = max(daily_stats[day_key]['pmax'], pr)

        if not np.isnan(w):
            daily_stats[day_key]['wmax'] = max(daily_stats[day_key]['wmax'], w)
            
    # Finalisation des stats quotidiennes
    final_daily_stats = []
    
    all_temps = []
    all_precips = []
    
    for day_key, stats in daily_stats.items():
        stats['tmin'] = round(stats['tmin'], 1) if stats['tmin'] != np.inf else 'N/D'
        stats['tmax'] = round(stats['tmax'], 1) if stats['tmax'] != -np.inf else 'N/D'
        stats['pmin'] = round(stats['pmin'], 1) if stats['pmin'] != np.inf else 'N/D'
        stats['pmax'] = round(stats['pmax'], 1) if stats['pmax'] != -np.inf else 'N/D'
        stats['wmax'] = round(stats['wmax'], 1) if stats['wmax'] != -np.inf else 'N/D'
        stats['precip'] = round(stats['precip'], 1)
        
        all_temps.extend([t for t in stats['temps'] if not np.isnan(t)])
        all_precips.append(stats['precip'])
        
        final_daily_stats.append({k: v for k, v in stats.items() if k not in ['temps', 'pressures', 'winds', 'precips']})
        
    # Stats Globales
    avg_temp = round(np.mean(all_temps), 1) if all_temps else 'N/D'
    min_temp = round(np.min(all_temps), 1) if all_temps else 'N/D'
    max_temp = round(np.max(all_temps), 1) if all_temps else 'N/D'
    total_precip = round(np.sum(all_precips), 1) if all_precips else 'N/D'
    
    global_stats = {
        'avg_temp': avg_temp,
        'min_temp': min_temp,
        'max_temp': max_temp,
        'total_precip': total_precip
    }

    return {'daily': final_daily_stats, 'global': global_stats}

def generate_aqi_section(locality_name, aqi_data, output_dir):
    """Génère la section qualité de l'air (AQI) incluant l'évolution graphique."""
    def get_aqi_class(aqi_value):
        if aqi_value is None or aqi_value < 0: return ""
        if aqi_value <= 50: return "aqi-good"
        elif aqi_value <= 100: return "aqi-moderate"
        elif aqi_value <= 150: return "aqi-unhealthy-sensitive"
        elif aqi_value <= 200: return "aqi-unhealthy"
        else: return "aqi-hazardous"

    def get_aqi_description(aqi_value):
        if aqi_value is None or aqi_value < 0: return "Données AQI non disponibles"
        if aqi_value <= 50: return "Bonne"
        elif aqi_value <= 100: return "Modérée"
        elif aqi_value <= 150: return "Mauvaise pour les groupes sensibles"
        elif aqi_value <= 200: return "Mauvaise"
        else: return "Très mauvaise/Dangereuse"
        
    if not aqi_data or not aqi_data.get('daily'):
        return ""
    
    daily_aqi_values = aqi_data['daily'].get('european_aqi', [])
    current_aqi = daily_aqi_values[0] if daily_aqi_values else None
    
    # 1. Génération du graphique d'évolution
    aqi_chart_path = generate_aqi_chart_png(locality_name, aqi_data, output_dir)
    
    # 2. Section de résumé (current AQI)
    if current_aqi is None:
        summary_html = "<p>Données AQI actuelles non disponibles.</p>"
        aqi_class = ""
    else:
        aqi_class = get_aqi_class(current_aqi)
        aqi_desc = get_aqi_description(current_aqi)
        
        summary_html = f"""
        <div style="font-size: 1.2em;">
            Indice AQI Actuel: 
            <span class="aqi-value">{current_aqi}</span> 
            (<strong>{aqi_desc}</strong>)
        </div>
        <p style="margin: 0; font-size: 0.9em; color: #555;">
            *Basé sur l'indice européen
        </p>
        """
        
    # 3. Assemblage
    chart_html = ""
    if aqi_chart_path:
        chart_html = f"""
        <div style="margin-top: 15px;">
            <h4>Évolution des prévisions d'AQI (Quotidiens)</h4>
            <div class="chart-block">
                <img src="{aqi_chart_path}" alt="Graphique d'évolution de l'AQI pour {locality_name}">
            </div>
        </div>
        """

    if not summary_html and not chart_html:
        return ""

    return f"""
    <hr>
    <h2>Qualité de l'Air (AQI Européen)</h2>
    <div class="aqi-container {aqi_class}">
        <div class="aqi-summary">
            {summary_html}
        </div>
        {chart_html}
    </div>
    """

def generate_forecast_content(locality_name, day_key, hourly_data_for_day, output_dir):
    """Génère les trois graphiques de détail pour une journée et le HTML de synthèse avec icônes."""
    
    daily_chart_paths = generate_daily_forecast_charts_png(locality_name, day_key, hourly_data_for_day, output_dir)
    
    daily_chart_tp_path = daily_chart_paths.get('temp_precip')
    daily_chart_p_path = daily_chart_paths.get('pressure')
    daily_chart_w_path = daily_chart_paths.get('wind_max')
    
    conditions_summary = ""
    hourly_times = hourly_data_for_day['hourly']['time']
    
    def wmo_description(code):
        return WMO_CODES_MAP.get(code, "Inconnu")
    
    # Création d'une grille pour les conditions horaires, rendue scrollable horizontalement sur petits écrans
    conditions_summary += '<div style="display: flex; overflow-x: auto; white-space: nowrap; padding-bottom: 10px; margin-bottom: 20px; border-bottom: 1px solid #ccc;">'

    for i in range(0, len(hourly_times), 3):
        if i >= len(hourly_data_for_day['hourly']['weather_code']) or hourly_data_for_day['hourly']['weather_code'][i] is None:
             continue

        time_obj = datetime.fromisoformat(hourly_times[i])
        time_formatted = time_obj.strftime("%H:%M")
        wmo_code = hourly_data_for_day['hourly']['weather_code'][i]
        wmo_desc = wmo_description(wmo_code)
        
        wmo_icon = f'<img src="icons/{wmo_code}.svg" alt="{wmo_desc}" style="width:96px;height:96px;vertical-align:middle; display:block; margin: 0 auto;">'
        
        conditions_summary += f"""
        <div style="text-align:center; flex: 0 0 auto; margin: 0 10px;">
            <p style="margin:0; font-weight:bold;">{time_formatted}</p>
            {wmo_icon}
            <p style="margin:5px 0 0 0; font-size:0.8em; white-space: normal; max-width: 80px;">{wmo_desc}</p>
        </div>
        """
    conditions_summary += '</div>'


    # Rendu du HTML en colonne unique
    return f"""
    <h3>Conditions Météo</h3>
    {conditions_summary}
    
    <div class="chart-block" style="margin-bottom: 20px;">
        <h4>Température et Précipitations</h4>
        <img src="{daily_chart_tp_path}" alt="Graphique horaire Température et Pluie pour {locality_name} le {day_key}">
    </div>
    <div class="chart-block" style="margin-bottom: 20px;">
        <h4>Pression Atmosphérique</h4>
        <img src="{daily_chart_p_path}" alt="Graphique horaire Pression pour {locality_name} le {day_key}">
    </div>
    <div class="chart-block" style="margin-bottom: 20px;">
        <h4>Vent Max</h4>
        <img src="{daily_chart_w_path}" alt="Graphique horaire Vent Max pour {locality_name} le {day_key}">
    </div>
    """

def generate_climatology_section(locality_name, climatology_stats, output_dir):
    """Génère la section climatologie avec trois graphiques séparés en colonne unique."""
    global_stats = climatology_stats['global']
    
    chart_paths = generate_climatology_charts_png(locality_name, climatology_stats, output_dir)
    
    # Mise à jour du nom du fichier
    chart_tp_path = chart_paths.get('tmax_tmin_precip')
    chart_pminmax_path = chart_paths.get('pminmax')
    chart_wmax_path = chart_paths.get('wind_max')

    if not chart_tp_path:
        return "" 

    # 1. Stats Globales - Utilisation de la classe 'metadata' mise à jour
    global_html = f"""
    <div class="metadata">
        <div>Temp. Moyenne (31j): <strong>{global_stats['avg_temp']} °C</strong></div>
        <div>Temp. Min Absolue (31j): <strong>{global_stats['min_temp']} °C</strong></div>
        <div>Temp. Max Absolue (31j): <strong>{global_stats['max_temp']} °C</strong></div>
        <div>Précipitations Totales (31j): <strong>{global_stats['total_precip']} mm</strong></div>
    </div>
    """

    # 2. Graphiques en colonne unique - Utilisation de la classe 'chart-block' mise à jour
    charts_html = f"""
    <div class="chart-block" style="margin-bottom: 20px;">
        <h3>Évolution Température (Max et Min) et Précipitations</h3>
        <img src="{chart_tp_path}" alt="Graphique de Tmax, Tmin et précipitations historique">
    </div>
    <div class="chart-block" style="margin-bottom: 20px;">
        <h3>Évolution Pression</h3>
        <img src="{chart_pminmax_path}" alt="Graphique de pression min/max historique">
    </div>
    <div class="chart-block" style="margin-bottom: 20px;">
        <h3>Évolution Vent Max</h3>
        <img src="{chart_wmax_path}" alt="Graphique de vent max historique">
    </div>
    """
    
    return f"<h2>Climatologie (Historique sur 31 Jours)</h2>{global_html}{charts_html}"

def load_data(file_path):
    """Charge les données du fichier JSON."""
    try:
        with open(file_path, 'r', encoding='utf-8') as f:
            data = json.load(f)
        return data
    except FileNotFoundError:
        print(f"Erreur: Le fichier de données '{file_path}' n'a pas été trouvé. Veuillez lancer l'étape de scraping.")
        return None
    except json.JSONDecodeError:
        print(f"Erreur: Le fichier de données '{file_path}' n'est pas un JSON valide.")
        return None


def generate_static_site(data):
    """Fonction principale de génération du site statique."""

    if not data:
        print("Génération annulée en raison d'une erreur de chargement des données.")
        return
        
    output_dir = "output_meteo"
    current_year = datetime.now().year
    
    # Création des répertoires nécessaires
    os.makedirs(os.path.join(output_dir, "charts"), exist_ok=True)
    os.makedirs(os.path.join(output_dir, "icons"), exist_ok=True) 
    
    print(f"Création du répertoire de sortie: {output_dir}/")

    # Création du fichier CSS
    with open(os.path.join(output_dir, "style.css"), "w", encoding="utf-8") as f:
        f.write(CSS_STYLE)
    print("-> style.css généré.")
    
    config = data.get('config', {})
    localities = {name: data[name] for name in data if name != 'config'}
    locality_links = ""
    
    last_updated = datetime.now().strftime("%Y-%m-%d %H:%M:%S")

    # 1. Génération des pages de localité
    for locality_name, locality_data in localities.items():
        filename = locality_name.replace(' ', '_').replace('-', '_').lower() + ".html"
        print(f"-> Génération de la page pour {locality_name} dans {filename}...")

        meteo_data = locality_data.get('meteo_data', {})
        hourly = meteo_data.get('hourly', {})
        hourly_units = meteo_data.get('hourly_units', {})
        aqi_data = locality_data.get('aqi_data', {})

        # 1.1 Préparation des données horaires par jour
        hourly_by_day = {}
        for i, t_str in enumerate(hourly.get('time', [])):
            try:
                time_obj = datetime.fromisoformat(t_str)
                day_key = time_obj.date().isoformat()
                
                if day_key not in hourly_by_day:
                    hourly_by_day[day_key] = {
                        'time': [], 'temperature_2m': [], 'precipitation': [], 
                        'weather_code': [], 'pressure_msl': [], 'wind_gusts_10m': []
                    }
                
                for key in hourly_by_day[day_key].keys():
                    if key == 'time':
                        hourly_by_day[day_key][key].append(t_str)
                    else:
                        data_list = hourly.get(key, [])
                        hourly_by_day[day_key][key].append(data_list[i] if i < len(data_list) else None)
            except ValueError:
                continue
        
        all_days = sorted(hourly_by_day.keys())
        forecast_days = all_days[-config.get('fore_days', 7):]
        
        # 1.2 Génération des Graphiques Généraux (Vue d'ensemble)
        overall_chart_paths = generate_overall_forecast_charts_png(locality_name, hourly, config, output_dir)
        
        overall_forecast_chart_tp = f'<div class="chart-block"><img src="{overall_chart_paths.get("temp_precip")}" alt="Graphique Température et Précipitations"></div>' if overall_chart_paths.get("temp_precip") else "<p>Données T/P manquantes.</p>"
        overall_forecast_chart_p = f'<div class="chart-block"><img src="{overall_chart_paths.get("pressure")}" alt="Graphique Pression"></div>' if overall_chart_paths.get("pressure") else "<p>Données Pression manquantes.</p>"
        overall_forecast_chart_w = f'<div class="chart-block"><img src="{overall_chart_paths.get("wind_max")}" alt="Graphique Vent Max"></div>' if overall_chart_paths.get("wind_max") else "<p>Données Vent Max manquantes.</p>"


        # 1.3 Génération du contenu jour par jour (Onglets)
        forecast_tabs = ""
        forecast_content = ""
        
        for i, day in enumerate(forecast_days):
            # Traduction robuste du label de l'onglet 
            if i == 0:
                day_label = "Aujourd'hui"
            else:
                date_obj = datetime.fromisoformat(day)
                day_index = date_obj.weekday()
                french_day_abbrev = FRENCH_DAYS_ABBREV[day_index]
                date_suffix = date_obj.strftime("%d/%m")
                day_label = f"{french_day_abbrev}. {date_suffix}" 
            
            content_html = generate_forecast_content(
                locality_name, 
                day, 
                {'hourly': hourly_by_day[day], 'hourly_units': hourly_units}, 
                output_dir
            )
            
            forecast_tabs += f"""
            <button class="tablinks" onclick="openForecastDay(event, 'day-{i}')">{day_label}</button>
            """
            forecast_content += f"""
            <div id="day-{i}" class="forecast-day" style="display:none;">
                {content_html}
            </div>
            """

        # 1.4 Climatologie 
        climatology_stats = calculate_climatology(hourly, config)
        climatology_section = generate_climatology_section(locality_name, climatology_stats, output_dir)
        
        # 1.5 Section AQI
        aqi_section = generate_aqi_section(locality_name, aqi_data, output_dir)
        
        # 1.6 Remplissage du template 
        html_content = LOCALITY_TEMPLATE.format(
            locality_name=locality_name,
            latitude=locality_data.get('latitude', 'N/D'),
            longitude=locality_data.get('longitude', 'N/D'),
            timezone=meteo_data.get('timezone', 'N/D'),
            timezone_abbreviation=meteo_data.get('timezone_abbreviation', 'N/D'),
            last_updated=last_updated,
            num_forecast_days=config.get('fore_days', 7),
            aqi_section=aqi_section,
            overall_forecast_chart_tp=overall_forecast_chart_tp, 
            overall_forecast_chart_p=overall_forecast_chart_p,
            overall_forecast_chart_w=overall_forecast_chart_w,
            forecast_tabs=forecast_tabs,
            forecast_content=forecast_content,
            climatology_section=climatology_section,
            current_year=current_year 
        )

        with open(os.path.join(output_dir, filename), "w", encoding="utf-8") as f:
            f.write(html_content)
            
        locality_links += f"""
        <li><a href="{filename}">{locality_name} (Lat: {locality_data.get('latitude')}, Lon: {locality_data.get('longitude')})</a></li>
        """
        

    # 2. Génération de la page d'index
    index_content = INDEX_TEMPLATE.format(
        locality_links=locality_links,
        last_updated=last_updated,
        current_year=current_year 
    )

    with open(os.path.join(output_dir, "index.html"), "w", encoding="utf-8") as f:
        f.write(index_content)
    print("-> index.html généré.")
    
    print("\n✅ Génération du site statique terminée.")

# --- POINT D'ENTRÉE DU SCRIPT ---
def main():
    """Point d’entrée réutilisable pour la génération HTML."""
    try:
        import matplotlib.pyplot as plt
    except ImportError:
        print("\nERREUR: La bibliothèque Matplotlib n'est pas installée.")
        print("Veuillez l'installer en utilisant : pip install matplotlib")
        return

    all_data = load_data(DATA_FILE_PATH)
    if all_data:
        generate_static_site(all_data)


if __name__ == "__main__":
    main()
