import configparser
import requests
import json
import os
import sys

# --- Configuration des noms de fichiers ---
CONFIG_FILE = 'config.ini'
LOCALITES_FILE = 'localites.ini'
OUTPUT_FILE = 'all_data.json'  # Fichier de sortie unique

# --- Fonctions utilitaires ---

def load_config(filename):
    """Charge un fichier INI et retourne l'objet ConfigParser."""
    config = configparser.ConfigParser()
    try:
        read_files = config.read(filename)
        if not read_files:
            print(f"Erreur: Impossible de lire le fichier {filename}. Assurez-vous qu'il existe.")
            sys.exit(1)
        return config
    except Exception as e:
        print(f"Erreur lors de la lecture du fichier {filename}: {e}")
        sys.exit(1)

def download_data(url):
    """Télécharge les données à partir de l'URL donnée et les retourne au format JSON."""
    # L'URL passée ici est désormais nettoyée des guillemets.
    #print(f"-> Téléchargement des données de: {url}")
    try:
        response = requests.get(url)
        response.raise_for_status() 
        return response.json()
    except requests.exceptions.RequestException as e:
        # Afficher une erreur moins verbeuse pour une connexion non établie
        if 'No connection adapters were found' in str(e):
            print(f"Erreur de connexion (URL mal formatée ou problème réseau). URL: {url}")
        else:
            print(f"Erreur lors du téléchargement de {url}: {e}")
        return None

def save_single_json_file(data, filepath):
    """Sauvegarde toutes les données agrégées dans un seul fichier JSON."""
    try:
        dir_name = os.path.dirname(filepath)
        if dir_name:
            os.makedirs(dir_name, exist_ok=True)
            
        with open(filepath, 'w', encoding='utf-8') as f:
            json.dump(data, f, indent=4)
        print(f"\n✨ Succès: Toutes les données ont été sauvegardées dans {filepath}")
    except Exception as e:
        print(f"Erreur lors de la sauvegarde du fichier {filepath}: {e}")

# --- Fonction Principale ---

def main():
    """Fonction principale pour lire la configuration, les localités, télécharger les données et les agréger."""
    
    all_results = {}
    
    # 1. Charger les configurations
    config_config = load_config(CONFIG_FILE)
    localites_config = load_config(LOCALITES_FILE)
    
    # 2. Récupérer les paramètres globaux
    try:
        past_days = config_config.getint('config', 'past_days')
        fore_days = config_config.getint('config', 'fore_days')
        
        # --- CORRECTION APPLIQUÉE ICI : Utiliser .strip('"') pour retirer les guillemets ---
        url_meteo_template = config_config.get('config', 'url_meteo').strip('"')
        url_aqi_template = config_config.get('config', 'url_aqi').strip('"')
        
        all_results['config'] = {'past_days': past_days, 'fore_days': fore_days}
        
    except configparser.Error as e:
        print(f"Erreur de lecture des options dans [{CONFIG_FILE}]: {e}")
        return

    print(f"Paramètres globaux: past_days={past_days}, fore_days={fore_days}")
    print("-" * 50)

    # 3. Parcourir toutes les localités et collecter les données
    for locality in localites_config.sections():
        print(f"Traitement de la localité: **{locality}**")
        
        all_results[locality] = {}
        
        try:
            latitude = localites_config.getfloat(locality, 'latitude')
            longitude = localites_config.getfloat(locality, 'longitude')
            
            all_results[locality]['latitude'] = latitude
            all_results[locality]['longitude'] = longitude

        except configparser.Error as e:
            print(f"Erreur de lecture des coordonnées pour {locality}: {e}")
            continue

        # 4. Construire les URL (sans les guillemets grâce à .strip())
        url_meteo = url_meteo_template.format(latitude=latitude, longitude=longitude, past_days=past_days, fore_days=fore_days)
        url_aqi = url_aqi_template.format(latitude=latitude, longitude=longitude, past_days=past_days, fore_days=fore_days)
        
        # 5. Télécharger et agréger les données
        data_meteo = download_data(url_meteo)
        if data_meteo:
            all_results[locality]['meteo_data'] = data_meteo
            print(f"   -> Données Météo pour {locality} collectées.")
        
        data_aqi = download_data(url_aqi)
        if data_aqi:
            all_results[locality]['aqi_data'] = data_aqi
            print(f"   -> Données AQI pour {locality} collectées.")
            
        print("-" * 50)

    # 6. Sauvegarder toutes les données dans un fichier unique
    save_single_json_file(all_results, OUTPUT_FILE)

if __name__ == "__main__":
    try:
        main()
    except NameError:
        print("Erreur: La bibliothèque 'requests' n'est pas installée. Veuillez l'installer avec 'pip install requests'.")
