import time
import socket
import os
import traceback
from datetime import datetime, timedelta
import get_data_1 as get_data
import gen_html_7 as gen_html
import put_grimeco_1 as put_grimeco
import aqi_3 as aqi
import logging
from logging.handlers import RotatingFileHandler

# --- Configuration générale ---
HEURES_CIBLES = ["05:00", "11:00", "17:00", "23:00"]
LOG_FILE = "/home/pi/logs/script_auto.log"
os.makedirs(os.path.dirname(LOG_FILE), exist_ok=True)

# --- Configuration du logger avec rotation automatique ---
logger = logging.getLogger("scheduler")
logger.setLevel(logging.INFO)

handler = RotatingFileHandler(
    LOG_FILE, maxBytes=1_000_000, backupCount=5, encoding="utf-8"
)
formatter = logging.Formatter("[%(asctime)s] %(message)s", datefmt="%Y-%m-%d %H:%M:%S")
handler.setFormatter(formatter)
logger.addHandler(handler)

def log(msg):
    """Raccourci pour le logger"""
    print(msg)
    logger.info(msg)

# --- Vérification du réseau ---
def attendre_reseau(timeout=300):
    """Attend que le réseau soit disponible (max 5 min)."""
    log("🌐 Vérification du réseau...")
    start = time.time()
    while time.time() - start < timeout:
        try:
            socket.create_connection(("8.8.8.8", 53), 2)
            log("✅ Réseau disponible.")
            return True
        except OSError:
            log("⏳ Attente du réseau...")
            time.sleep(5)
    log("⚠️ Réseau toujours indisponible après 5 minutes.")
    return False

# --- Gestion du temps ---
def heure_to_datetime(hstr, date):
    h, m = map(int, hstr.split(":"))
    return datetime(date.year, date.month, date.day, h, m)

def prochaine_heure(now):
    """Retourne la prochaine heure programmée et son texte."""
    for hstr in HEURES_CIBLES:
        heure_dt = heure_to_datetime(hstr, now.date())
        if heure_dt > now:
            return heure_dt, hstr
    demain = now.date() + timedelta(days=1)
    return heure_to_datetime(HEURES_CIBLES[0], demain), HEURES_CIBLES[0]

def attendre(attente):
    """Attente fractionnée (par minutes) pour rester réactif."""
    while attente > 0:
        time.sleep(min(60, attente))
        attente -= 60

# --- Fonction principale exécutée à chaque horaire ---
def ma_fonction(heure):
    log(f"🚀 Début de l’exécution pour {heure}")
    try:
        for module, nom in [
            (get_data, "get_data_1"),
            (gen_html, "gen_html_7"),
            (aqi, "aqi_3.py"),
            (put_grimeco, "put_grimeco_1"),
        ]:
            t0 = time.time()
            module.main()
            duree = time.time() - t0
            log(f"✅ {nom}.main() terminé en {duree:.1f} s")
    except Exception as e:
        log(f"❌ Erreur pendant l’exécution de ma_fonction({heure}): {e}")
        logger.error(traceback.format_exc())
    log(f"🏁 Fin de l’exécution pour {heure}\n{'-'*60}")

# --- Scheduler principal ---
def main():
    log("=== Démarrage du scheduler ===")
    attendre_reseau()

    maintenant = datetime.now()
    aujourd_hui = maintenant.date()

    # Exécution initiale (heure passée la plus proche)
    heures_passees = [h for h in HEURES_CIBLES if heure_to_datetime(h, aujourd_hui) <= maintenant]
    if heures_passees:
        derniere = max(heures_passees, key=lambda h: heure_to_datetime(h, aujourd_hui))
        log(f"▶ Exécution initiale pour la dernière heure passée : {derniere}")
        ma_fonction(derniere)
    else:
        log("⏩ Aucune heure passée aujourd’hui — attente de la première exécution.")

    # Boucle continue
    while True:
        prochain_dt, prochaine_str = prochaine_heure(datetime.now())
        attente = (prochain_dt - datetime.now()).total_seconds()
        log(f"🕓 Prochaine exécution prévue à {prochain_dt.strftime('%Y-%m-%d %H:%M')} (dans {int(attente/60)} min)")
        attendre(attente)
        ma_fonction(prochaine_str)

if __name__ == "__main__":
    main()
