# ==============================================================
# Classe ArduinoController
# ==============================================================

import time,serial,threading
from typing import Optional, Tuple

class ArduinoController:
    def __init__(self, port="/dev/ttyUSB0", baud=115200, timeout=1.0, debug=False):
        self.port = port
        self.baud = baud
        self.timeout = timeout
        self.debug = debug
        self.lock = threading.Lock()
        self.msg_id = 0
        try:
            self.ser = serial.Serial(self.port, self.baud, timeout=0.1)
            time.sleep(2)
            print(f"✅ Port série ouvert sur {self.port}")
        except Exception as e:
            print(f"❌ Erreur ouverture du port série : {e}")
            raise

    def _debug_print_snd(self, full):
        if self.debug:
            print(f"S -> {full.strip()}")

    def _debug_print_rcv(self, line):
        if self.debug:
            print(f"R <- {line}")

    def send_and_wait(self, cmd: str, timeout: Optional[float] = None) -> Optional[Tuple[int, str, int]]:
        timeout = timeout or self.timeout
        with self.lock:
            self.msg_id += 1
            this_id = self.msg_id
            full = f"{this_id},{cmd}\n"
            try:
                self.ser.reset_input_buffer()
                self.ser.write(full.encode())
                self._debug_print_snd(full)
            except Exception as e:
                print(f"⚠️ Erreur écriture série: {e}")
                return None

            start = time.time()
            while time.time() - start < timeout:
                try:
                    raw = self.ser.readline().decode(errors='ignore').strip()
                except Exception as e:
                    print(f"⚠️ Erreur lecture série: {e}")
                    return None
                if not raw:
                    continue
                self._debug_print_rcv(raw)
                parts = [p.strip() for p in raw.split(",")]
                if len(parts) >= 3:
                    try:
                        num = int(parts[0])
                    except:
                        num = parts[0]
                    bok = parts[1]
                    try:
                        val = int(parts[2])
                    except:
                        try:
                            val = int(float(parts[2]))
                        except:
                            val = parts[2]
                    return (num, bok, val)
            if self.debug:
                print("⏱️ Timeout attente réponse Arduino.")
            return None

    def send_no_wait(self, cmd: str):
        with self.lock:
            self.msg_id += 1
            full = f"{self.msg_id},{cmd}\n"
            try:
                self.ser.write(full.encode())
                self._debug_print_snd(full)
            except Exception as e:
                print(f"⚠️ Erreur écriture série (no_wait): {e}")

    def close(self):
        try:
            if hasattr(self, "ser") and self.ser and self.ser.is_open:
                self.ser.close()
        except Exception:
            pass
