# audiocontroller.py
# ==============================================================
# CLASSE AUDIOPLAYER (Gère la lecture des ambiances sonores)
# ==============================================================
import os
import random
import time
import subprocess
import threading
from typing import Optional

class AudioPlayer:
    def __init__(self, mp3_dir: str = "mp3/", ipc_socket: str = "/tmp/mpv-audio-ipc"):
        """
        Initialise l'AudioPlayer.
        :param mp3_dir: Répertoire contenant les fichiers MP3.
        :param ipc_socket: Chemin du socket IPC pour le contrôle de mpv.
        """
        self.mp3_dir = mp3_dir
        self.ipc_socket = ipc_socket
        self.audio_process: Optional[subprocess.Popen] = None
        
        self.playlist: list[str] = []  # Liste des MP3 pour la lecture aléatoire sans répétition
        self.played_files: list[str] = [] # Liste des MP3 déjà joués
        self._load_playlist()
        
        self.fade_thread: Optional[threading.Thread] = None
        self.stop_fade_event = threading.Event()
        
        # S'assurer que le répertoire de travail pour mpv est propre (pour IPC)
        if os.path.exists(self.ipc_socket):
            os.remove(self.ipc_socket)


    def _load_playlist(self):
        """Charge et initialise la liste des fichiers MP3 disponibles."""
        try:
            all_files = [f for f in os.listdir(self.mp3_dir) if f.lower().endswith('.mp3')]
            if not all_files:
                print(f"⚠️ Aucun fichier MP3 trouvé dans le répertoire {self.mp3_dir}")
                return
            
            self.playlist = all_files
            random.shuffle(self.playlist)
            print(f"🎧 Playlist audio chargée avec {len(self.playlist)} fichiers.")

        except FileNotFoundError:
            print(f"❌ Erreur: Le répertoire MP3 '{self.mp3_dir}' n'existe pas.")

    def _get_next_audio_file(self) -> Optional[str]:
        """Sélectionne un fichier audio aléatoire sans répétition."""
        if not self.playlist:
            if not self.played_files:
                print("🛑 La playlist MP3 est vide et aucun MP3 n'a été joué. Arrêt.")
                return None
                
            # Réinitialisation de la liste de lecture (Logique de non-répétition)
            self.playlist = self.played_files
            random.shuffle(self.playlist)
            self.played_files = []
            print("🔄 Playlist MP3 réinitialisée.")

        filename = self.playlist.pop(0)
        return os.path.join(self.mp3_dir, filename)

    def _send_ipc_command(self, command: dict):
        """Envoie une commande JSON à mpv via son socket IPC."""
        if not self.audio_process or self.audio_process.poll() is not None:
            return
            
        try:
            with open(self.ipc_socket, 'w') as f:
                f.write(json.dumps(command) + '\n')
        except FileNotFoundError:
            # Le socket n'est pas encore prêt ou a été supprimé
            pass
        except Exception as e:
            print(f"⚠️ Erreur envoi commande IPC: {e}")

    def is_playing(self) -> bool:
        """Vérifie si un fichier audio est en cours de lecture."""
        return self.audio_process is not None and self.audio_process.poll() is None

    def play_next_ambient(self):
        """Arrête l'audio en cours et démarre un nouvel audio aléatoire."""
        if self.is_playing():
            self.stop() # Arrêt immédiat si une nouvelle ambiance est demandée
        
        next_file_path = self._get_next_audio_file()
        
        if next_file_path:
            filename = os.path.basename(next_file_path)
            print(f"▶️ Démarrage ambiance sonore : {filename}")
            
            # Utilisation de mpv pour l'audio en boucle avec IPC pour le contrôle du volume
            cmd = [
                "mpv",
                "--no-video",
                "--loop=inf",  # Lecture en boucle de l'ambiance
                "--input-ipc-server=" + self.ipc_socket,
                "--input-terminal=no",
                "--volume=100", # Démarrer au volume maximal
                next_file_path
            ]
            
            try:
                self.audio_process = subprocess.Popen(cmd, stdin=subprocess.DEVNULL, stdout=subprocess.DEVNULL, stderr=subprocess.DEVNULL)
                self.played_files.append(filename)
            except Exception as e:
                print(f"❌ Erreur de lancement de mpv pour l'audio : {e}")
                self.audio_process = None
                
    
    def _run_fade_out(self, duration: float = 1.0):
        """Thread pour effectuer le fondu (fade-out) du volume."""
        self.stop_fade_event.clear()
        start_time = time.time()
        
        # S'assurer que le volume est à 100% avant de commencer (au cas où)
        self._send_ipc_command({"command": ["set_property", "volume", 100]})
        
        while time.time() - start_time < duration and not self.stop_fade_event.is_set():
            if not self.is_playing():
                break # Le processus s'est arrêté naturellement
                
            elapsed = time.time() - start_time
            # Calcul du volume décroissant (de 100 à 0)
            current_volume = max(0, 100 - (elapsed / duration) * 100)
            self._send_ipc_command({"command": ["set_property", "volume", int(current_volume)]})
            time.sleep(0.05) # Petite pause pour une transition fluide
            
        self.stop_fade_event.set() # Terminé ou interrompu

        # Arrêt final après le fondu
        if self.is_playing():
             self.stop(wait_for_fade=False)


    def fade_out_and_stop(self, duration: float = 1.0):
        """Démarre le fondu audio dans un thread séparé."""
        if not self.is_playing():
            return
            
        print(f"🔇 Démarrage du fondu audio ({duration}s)...")
        if self.fade_thread and self.fade_thread.is_alive():
            # Arrêter un fondu précédent s'il y en a un
            self.stop_fade_event.set()
            self.fade_thread.join(timeout=0.1)
            
        self.fade_thread = threading.Thread(target=self._run_fade_out, args=(duration,), daemon=True)
        self.fade_thread.start()


    def stop(self, wait_for_fade: bool = True):
        """Arrête l'audio immédiatement, en attendant la fin du fondu si nécessaire."""
        if wait_for_fade and self.fade_thread and self.fade_thread.is_alive():
             self.fade_thread.join(timeout=1.1) # Attendre que le fondu se termine
             
        # Assurer l'arrêt du processus mpv
        if self.audio_process and self.audio_process.poll() is None:
            print("⏹️ Arrêt du processus audio MPV.")
            self.audio_process.terminate()
            self.audio_process.wait(timeout=1)
            
        self.audio_process = None
        # Nettoyer le socket IPC
        if os.path.exists(self.ipc_socket):
             os.remove(self.ipc_socket)
