# main.py (MODIFIÉ)
# ==============================================================
# Programme Principal : CHEF D'ORCHESTRE
# ==============================================================
import os
import time
import random
import datetime
from typing import List, Dict, Any, Tuple, Optional
from arduinocontroller import ArduinoController
from multiplayermanager import MultiPlayerManager
from audiocontroller import AudioPlayer
#from qr_overlay import QROverlayManager # NOUVEL IMPORT

# --- Structure pour stocker les événements programmés ---
class Event:
    def __init__(self, time_str: str, json_file: str, sound_mode: str, screen_id: int):
        self.time: datetime.time = datetime.datetime.strptime(time_str, "%H:%M").time()
        self.json_file: str = json_file
        self.sound_mode: str = sound_mode
        self.screen_id: int = screen_id
        self.has_played_today: bool = False # Pour éviter de rejouer le même événement

    def __lt__(self, other):
        return self.time < other.time

# --- Classe Orchestrator ---
class Orchestrator:
    
    # --- Chemins des répertoires ---
    JSON_SC0_DIR = "json_screen0/"
    JSON_SC1_DIR = "json_screen1/"
    JSON_EVENTS_DIR = "json_evenements/"
    MP3_DIR = "mp3/"
    TIMELINE_FILE = "chronologie.txt"
    
    def __init__(self, port="/dev/ttyUSB0", baud=115200, debug=False):
        self.arduino: ArduinoController = ArduinoController(port=port, baud=baud, debug=debug)
        self.manager: MultiPlayerManager = MultiPlayerManager(arduino=self.arduino)
        self.audio_player: AudioPlayer = AudioPlayer(mp3_dir=self.MP3_DIR)
        #self.qr_manager: QROverlayManager = QROverlayManager() # INSTANCIATION DU MANAGER QR
        
        self.events: List[Event] = self._load_timeline()
        
        # Logique de non-répétition pour le mode veille vidéo
        self.video_playlists: Dict[int, List[str]] = {0: [], 1: []}
        self.video_played: Dict[int, List[str]] = {0: [], 1: []}
        
        # Statut de lecture
        self.current_video_json: Dict[int, str] = {0: "", 1: ""} # Le JSON en cours de lecture
        self.is_event_running: Dict[int, bool] = {0: False, 1: False} # Est-ce un événement ?
        
        self._init_playlists()
        
        print("🎶 Orchestrateur prêt.")

    # --- Initialisation et chargement des listes de lecture ---

    def _load_files_from_dir(self, directory: str) -> List[str]:
        """Charge la liste des fichiers JSON d'un répertoire."""
        try:
            files = [f for f in os.listdir(directory) if f.lower().endswith('.json')]
            if not files:
                 print(f"⚠️ Aucun fichier JSON trouvé dans {directory}")
            return files
        except FileNotFoundError:
            print(f"❌ Erreur: Le répertoire '{directory}' n'existe pas.")
            return []

    def _init_playlists(self):
        """Initialise les listes de lecture vidéo aléatoires sans répétition."""
        self.video_playlists[0] = self._load_files_from_dir(self.JSON_SC0_DIR)
        random.shuffle(self.video_playlists[0])
        self.video_playlists[1] = self._load_files_from_dir(self.JSON_SC1_DIR)
        random.shuffle(self.video_playlists[1])

    def _get_next_video_json(self, screen_id: int) -> Optional[str]:
        """Sélectionne un fichier JSON vidéo aléatoire sans répétition."""
        playlist = self.video_playlists[screen_id]
        played = self.video_played[screen_id]
        directory = self.JSON_SC0_DIR if screen_id == 0 else self.JSON_SC1_DIR
        
        if not playlist:
            if not played:
                print(f"🛑 Playlist vidéo SC{screen_id} vide. Impossible de démarrer la veille.")
                return None
            
            # Réinitialisation de la liste de lecture (Logique de non-répétition)
            self.video_playlists[screen_id] = played
            random.shuffle(self.video_playlists[screen_id])
            self.video_played[screen_id] = []
            print(f"🔄 Playlist vidéo SC{screen_id} réinitialisée.")
            playlist = self.video_playlists[screen_id] # Mise à jour de la playlist après réinitialisation

        filename = playlist.pop(0)
        self.video_played[screen_id].append(filename)
        return os.path.join(directory, filename)

    def _load_timeline(self) -> List[Event]:
        """Parse le fichier de chronologie et charge les événements."""
        events = []
        try:
            with open(self.TIMELINE_FILE, 'r', encoding="utf-8") as f:
                for line in f:
                    line = line.strip()
                    if not line or line.startswith("#"):
                        continue
                    
                    parts = line.split()
                    if len(parts) == 4 and parts[3] in ['0', '1']:
                        time_str, json_file, sound_mode, screen_id = parts
                        if sound_mode not in ['sound', 'nosound']:
                            print(f"⚠️ Mode audio invalide dans {self.TIMELINE_FILE} : {sound_mode}")
                            continue
                            
                        events.append(Event(time_str, json_file, sound_mode, int(screen_id)))
                    else:
                        print(f"⚠️ Ligne de chronologie ignorée (format incorrect) : {line}")

        except FileNotFoundError:
            print(f"❌ Fichier de chronologie '{self.TIMELINE_FILE}' introuvable.")
            return []
            
        events.sort()
        print(f"📅 {len(events)} événements chargés et triés.")
        return events

    # --- Logique de lecture ---

    def _start_default_mode(self, screen_id: int):
        """Démarre le mode veille (vidéo + ambiance audio) sur un écran."""
        
        # 1. Sélectionner et démarrer la vidéo de veille
        next_json = self._get_next_video_json(screen_id)
        if next_json:
            print(f"👀 [SC{screen_id}] Démarrage mode veille avec vidéo : {next_json}")
            self.current_video_json[screen_id] = next_json
            self.is_event_running[screen_id] = False
            # Le mode veille est SANS le son de la vidéo pour laisser l'ambiance MP3 jouer
            self.manager.play_video(screen_id, next_json, audio_enabled=True) 
        else:
            print(f"🛑 [SC{screen_id}] Impossible de trouver un fichier pour le mode veille. Arrêt.")
            return

        # 2. Démarrer l'ambiance sonore si elle n'est pas déjà en cours
        if not self.audio_player.is_playing():
            self.audio_player.play_next_ambient()
        
    def _start_event(self, event: Event):
        """Démarre une vidéo événementielle, préempte l'audio et la veille."""
        
        screen_id = event.screen_id
        full_json_path = os.path.join(self.JSON_EVENTS_DIR, event.json_file)
        
        print(f"\n🔔🔔 [SC{screen_id}] --- Démarrage ÉVÉNEMENT programmé : {event.json_file} ---")
        
        # 1. Commande le fondu et l'arrêt de l'ambiance sonore MP3 (MODIFICATION CDC)
        self.audio_player.fade_out_and_stop(duration=2.0)
        
        # 2. Arrêt propre de la vidéo de veille en cours
        self.manager.stop_video(screen_id)
        
        # 3. Démarrage de la vidéo événementielle (respect du paramètre sound/nosound)
        audio_enabled = (event.sound_mode == 'sound')
        self.manager.play_video(screen_id, full_json_path, audio_enabled=audio_enabled)
        
        self.current_video_json[screen_id] = full_json_path
        self.is_event_running[screen_id] = True
        event.has_played_today = True # Marquer l'événement comme joué
        
    def _is_video_done(self, screen_id: int) -> bool:
        """Vérifie si la vidéo actuelle (veille ou événement) est terminée.
        
        Basé sur la détection de fin dans VideoPlayer._run_sync_loop (processus mpv terminé).
        Comme VideoPlayer ne fournit pas de propriété 'is_done', nous nous basons
        sur l'état du thread de synchronisation.
        """
        player = self.manager.players.get(screen_id)
        if not player:
            return True # Considéré comme terminé si le lecteur n'existe pas

        # Si le thread de synchro est terminé ET que l'Event playback_done est marqué
        if player.playback_done.is_set() and (player.sync_thread is None or not player.sync_thread.is_alive()):
            return True
        return False
        
    def _check_for_events(self):
        """Vérifie l'heure système par rapport aux événements programmés."""
        now = datetime.datetime.now().time()
        
        for event in self.events:
            if not event.has_played_today and now >= event.time:
                # Vérifie si la vidéo actuelle n'est PAS un événement ou si l'écran est libre
                if not self.is_event_running[event.screen_id] or self._is_video_done(event.screen_id):
                    self._start_event(event)

    def _reset_daily_events(self):
        """Réinitialise le statut des événements joués à minuit."""
        for event in self.events:
            event.has_played_today = False
        print("☀️ Statut des événements quotidiens réinitialisé.")


    # --- Boucle principale ---

    def run(self):
        """Boucle principale de l'orchestrateur."""
        
        # 0. Démarrage du QR Code en surcouche (NOUVEAU)
        #self.qr_manager.start_overlay()
        
        # 1. Démarrage initial du mode veille
        self._start_default_mode(0)
        self._start_default_mode(1)

        last_check_date = datetime.date.today()
        
        try:
            while True:
                
                # Vérification quotidienne pour réinitialiser le statut 'has_played_today'
                if datetime.date.today() > last_check_date:
                    self._reset_daily_events()
                    last_check_date = datetime.date.today()

                # 2. Vérification des événements programmés (Préemption)
                self._check_for_events()

                # 3. Surveillance de la fin de lecture pour chaque écran
                for screen_id in range(2):
                    if self._is_video_done(screen_id):
                        if self.is_event_running[screen_id]:
                             print(f"\n🔚 [SC{screen_id}] Fin de l'événement. Retour au mode veille.")
                        else:
                            print(f"\n🔄 [SC{screen_id}] Fin de la vidéo de veille. Démarrage de la suivante.")
                            
                        # Retour au mode veille (Logique de fin d'événement/veille)
                        self._start_default_mode(screen_id)

                time.sleep(1) # Faible consommation de CPU (Contrainte 4. Performance)

        except KeyboardInterrupt:
            print("\nInterruption par l'utilisateur.")
        except Exception as e:
            print(f"\n❌ Erreur critique dans la boucle principale : {e}")
        finally:
            self.stop()

    def stop(self):
        """Arrêt propre de tous les composants."""
        self.manager.stop_all()
        self.audio_player.stop(wait_for_fade=False)
        #self.qr_manager.stop_overlay() # ARRÊT ET NETTOYAGE DU QR CODE
        self.arduino.close()
        print("Orchestrateur arrêté.")

# --- Point d'entrée principal ---
if __name__ == "__main__":
    '''
    # --- Création de répertoires factices (pour le test) ---
    for d in ["json_screen0", "json_screen1", "json_evenements", "mp3"]:
        os.makedirs(d, exist_ok=True)
        # Créer des fichiers de test si les répertoires sont vides
        if d != "mp3" and not os.listdir(d):
             # Simuler un fichier JSON de veille
            with open(os.path.join(d, f"veille_{d.split('_')[-1]}_1.json"), "w") as f:
                f.write('{"video_path": "video_1.mp4", "effects": []}')
            if d == "json_evenements":
                 # Simuler un fichier JSON d'événement
                with open(os.path.join(d, "event_test.json"), "w") as f:
                    f.write('{"video_path": "event_video.mp4", "effects": [{"name": "Lumiere", "start_time": 0, "end_time": 5000, "description": "code_start=1\\ncode_end=2\\ncolor=rouge"}]}')
        elif d == "mp3" and not os.listdir(d):
            # Simuler un fichier MP3
             with open(os.path.join(d, "ambiance_test.mp3"), "w") as f:
                f.write("DUMMY MP3 CONTENT")

    # --- Création d'un fichier de chronologie factice (pour le test) ---
    with open("chronologie.txt", "w") as f:
        # Événement pour dans 1 minute (pour tester la préemption)
        next_minute = (datetime.datetime.now() + datetime.timedelta(minutes=1)).strftime("%H:%M")
        f.write(f"{next_minute} event_test.json sound 0\n")
    '''    
    # --- Lancement de l'orchestrateur ---
    try:
        # NOTE : Adaptez 'port' si votre Arduino n'est pas sur /dev/ttyUSB0
        orchestrator = Orchestrator(port="/dev/ttyUSB0", debug=True)
        orchestrator.run()
    except Exception as e:
        print(f"\nLe programme n'a pas pu démarrer: {e}")
        print("Assurez-vous que l'Arduino est branché et que le port est correct.")
