# ==============================================================
# Classe EffectController
# ==============================================================
from arduinocontroller import ArduinoController

class EffectController:
    def __init__(self, arduino: ArduinoController):
        self.active_effects = {}   # id interne → code_end
        self.id_effect = 0
        self.arduino = arduino
        self.colors = self._init_colors()

    # --- Parsing description ---
    def _parse_description(self, effect):
        params = {}
        for line in effect.get("description", "").splitlines():
            line = line.strip()
            if not line or line.startswith("#"):
                continue
            if "=" in line:
                k, v = line.split("=", 1)
                params[k.strip()] = v.strip()
        return params

    # --- Couleurs prédéfinies ---
    def _init_colors(self):
        return {
            "noir": (0, 0, 0), "blanc": (255, 255, 255), "rouge": (255, 0, 0),
            "vert": (0, 255, 0), "bleu": (0, 0, 255), "jaune": (255, 255, 0),
            "cyan": (0, 255, 255), "magenta": (255, 0, 255), "orange": (255, 128, 0),
            "vert_lime": (128, 255, 0), "bleu_ciel": (0, 128, 255), "rose": (255, 0, 128),
            "violet": (128, 0, 255), "turquoise": (0, 255, 128), "gris": (128, 128, 128),
            "gris_clair": (192, 192, 192), "beige": (245, 245, 220)
        }

    def cv_color(self, name):
        return self.colors.get(name.lower(), (0, 0, 0))

    # --- Démarrage effet ---
    def start_effect(self, effect):
        params = self._parse_description(effect)
        cmd = ""
        code_end = "0"

        for cle, valeur in params.items():
            if cle.startswith("couleur"):
                r, g, b = self.cv_color(valeur)
                cmd += f"{r},{g},{b},"
            elif cle.startswith("code fin"):
                code_end = valeur
            else:
                cmd += valeur + ","

        cmd = cmd.rstrip(",")
        print(f"→ CMD START: {cmd}")
        
        resp = self.arduino.send_and_wait(cmd, timeout=1.0)
        if resp:
            _, bok, val = resp
            print(f"✅ start {effect['name']} : bok={bok}, arduino_id={val}")
            effect["arduino_id"] = val
            self.id_effect += 1
            self.active_effects[self.id_effect] = code_end
            return self.id_effect
        else:
            print(f"⚠️ Pas de réponse pour {effect['name']}")
            return None
        """
        self.arduino.send_no_wait(cmd)
        self.id_effect += 1
        self.active_effects[self.id_effect] = code_end
        return self.id_effect
        """
    # --- Arrêt effet ---
    def stop_effect(self, id):
        if id not in self.active_effects:
            print(f"⚠️ id {id} inconnu, effet déjà arrêté ?")
            return

        code_end = self.active_effects[id]
        cmd = ""

        if code_end == "0":
            cmd = f"0,{id}"
        else:
            cmd = f"{code_end},0"

        print(f"→ CMD STOP ({id}) : {cmd}")
        resp = self.arduino.send_and_wait(cmd, timeout=1.0)
        print("Réponse:", resp)

        # Nettoyage
        del self.active_effects[id]
