#include <Arduino.h>
#include <Adafruit_NeoPixel.h>

// ================================
// CONSTANTES ET DÉFINITIONS
// ================================

// Codes de retour
#define ER 0
#define OK 1

// Codes d'erreur
#define MSG_LENGTH_3 0
#define STRIP_NUMBER 1
#define PARAM_COUNT 2
#define UNKNOWN_CMD 3

#define BUFFER_SIZE 64  // Taille maximale d'une ligne de commande
#define MAX_VALUES 16   // Nombre maximal de valeurs dans une commande
#define MAX_EFFECTS 50  // Nombre maximal d'effets simultanés
#define MAX_MOTORS 4    // Nombre maximal de moteurs pas à pas
#define STRIPCOUNT 2    // Nombre de rubans de LEDs

// AJOUT: Pas pour une révolution complète du moteur 28BYJ-48
// Cette valeur doit être ajustée si le pas est différent pour votre moteur (ex: 4096)
#define STEPS_PER_REVOLUTION 2048 

// ================================
// CONFIGURATION DES RUBANS LED
// ================================

// Ruban 1
#define PIN1 2
#define NUMPIXELS1 300
Adafruit_NeoPixel strip1(NUMPIXELS1, PIN1, NEO_GRB + NEO_KHZ800);
// Ruban 2
#define PIN2 3
#define NUMPIXELS2 144
Adafruit_NeoPixel strip2(NUMPIXELS2, PIN2, NEO_GRB + NEO_KHZ800);
// Tableau des rubans pour un accès simplifié
Adafruit_NeoPixel *strips[STRIPCOUNT] = { &strip1, &strip2 };
// ================================
// DÉFINITIONS DES EFFETS LED
// ================================

// Types d'effets disponibles
enum EffectType {
  STOP,     // Arrêt d'effet
  FIXED,    // Couleur fixe
  BLINK,    // Clignotement
  CHASE,    // Effet de poursuite
  FADE,     // Fondu entre couleurs
  RAINBOW,  // Arc-en-ciel
  FIRE,     // Effet feu
  SNOW,     // Effet neige
  SMOKE     // Effet fumée
};
// Nombre de paramètres requis pour chaque effet
int nbParam[] = { 3, 8, 12, 12, 12, 6, 6, 7, 7 };
// Structure représentant un effet en cours
struct Effect {
  int id;                    // Identifiant unique de l'effet
  Adafruit_NeoPixel *strip;  // Ruban concerné
  int startIndex;            // Index de début
  int endIndex;              // Index de fin
  EffectType type;           // Type d'effet
  uint32_t color1;           // Première couleur
  uint32_t color2;           // Seconde couleur
  int interval;              // Intervalle entre mises à jour (ms)
  unsigned long lastUpdate;  // Dernière mise à jour
  bool toggle;               // État de basculement
  int step;                  // Étape courante de l'animation
  bool active;               // Statut d'activation
  uint8_t brightnessStep;    // Pas de luminosité (pour FADE) ou densité
};

// Gestion des effets
Effect effects[MAX_EFFECTS];  // Tableau des effets actifs
int effectCount = 0;          // Nombre d'effets actifs
int nextEffectId = 1;         // Prochain ID d'effet disponible
boolean bex = false;          // Verrou pour éviter les modifications pendant la mise à jour

// ================================
// GESTION DE LA COMMUNICATION
// ================================

char inputBuffer[BUFFER_SIZE];  // Buffer de réception des commandes
uint8_t bufIndex = 0;           // Index courant dans le buffer
bool lineReady = false;         // Indicateur de ligne complète reçue

// ================================
// CLASSES
// ================================

/**
 * Gestion d'un moteur DC
 */
class Motor {
private:
  int pinENA;  // PWM (ENA)
  int pinIN1;  // IN1
  int pinIN2;  // IN2

public:
  // Constructeur : initialise les broches
  Motor(int ena, int in1, int in2) {
    pinENA = ena;
    pinIN1 = in1;
    pinIN2 = in2;
    pinMode(pinENA, OUTPUT);
    pinMode(pinIN1, OUTPUT);
    pinMode(pinIN2, OUTPUT);
  }

  // Arrêter le moteur
  void stop() {
    digitalWrite(pinIN1, LOW);
    digitalWrite(pinIN2, LOW);
    analogWrite(pinENA, 0);
  }

  // Tourner dans le sens horaire (vitesse 0-255)
  void forward(int speed) {
    digitalWrite(pinIN1, HIGH);
    digitalWrite(pinIN2, LOW);
    analogWrite(pinENA, constrain(speed, 0, 255));
  }

  // Tourner dans le sens antihoraire (vitesse 0-255)
  void backward(int speed) {
    digitalWrite(pinIN1, LOW);
    digitalWrite(pinIN2, HIGH);
    analogWrite(pinENA, constrain(speed, 0, 255));
  }
};

/**
 * Gestion d'un moteur pas à pas 28BYJ-48
 * La position 'home' est la position initiale (stepCounter = 0).
 */
class Stepper28BYJ48 {
private:
  int pin1, pin2, pin3, pin4;
  int sequence[8][4] = {
    { 1, 0, 0, 0 }, { 1, 1, 0, 0 }, { 0, 1, 0, 0 }, { 0, 1, 1, 0 }, { 0, 0, 1, 0 }, { 0, 0, 1, 1 }, { 0, 0, 0, 1 }, { 1, 0, 0, 1 }
  };
  int currentStep;    // Pas courant dans la séquence
  long lastStepTime;  // Dernier temps de pas
  int stepDelay;      // Vitesse en ms entre chaque pas
  int stepDirection;  // 1 = horaire, -1 = antihoraire
  bool moving;        // Vrai si le moteur tourne
  
  // AJOUTS POUR LA GESTION HOME
  long stepCounter;   // Compteur de pas depuis la position 'home'
  bool homing;        // Vrai si le moteur est en cours de retour 'home'
  
  // Appliquer un pas spécifique de la séquence
  void setStep(int step) {
    digitalWrite(pin1, sequence[step][0]);
    digitalWrite(pin2, sequence[step][1]);
    digitalWrite(pin3, sequence[step][2]);
    digitalWrite(pin4, sequence[step][3]);
  }

public:
  Stepper28BYJ48(int p1, int p2, int p3, int p4, int delayMs = 2) {
    pin1 = p1;
    pin2 = p2;
    pin3 = p3;
    pin4 = p4;
    stepDelay = delayMs;
    currentStep = 0;
    lastStepTime = 0;
    stepDirection = 1;
    moving = false;
    pinMode(pin1, OUTPUT);
    pinMode(pin2, OUTPUT);
    pinMode(pin3, OUTPUT);
    pinMode(pin4, OUTPUT);
    
    // Initialisation des nouveaux membres
    homing = false;
    stepCounter = 0; // La position de départ est la position 'home'
  }

  // Démarrer la rotation continue
  void start(int direction = 1) {
    stepDirection = (direction >= 0) ? 1 : -1;
    moving = true;
    homing = false; // Sortir du mode 'home' si une nouvelle rotation est demandée
  }

  // Arrêter le moteur
  void stop() {
    moving = false;
    homing = false;
    release();
  }
  
  /**
   * Démarrer la procédure de retour à la position 'home' (0 pas).
   * Utilise le chemin de rotation le plus court.
   */
  void startHoming() {
    // Si le compteur est déjà un multiple de STEPS_PER_REVOLUTION (position home ou un tour complet)
    if ((stepCounter % STEPS_PER_REVOLUTION) == 0) {
        stepCounter = 0; // S'assurer que le compteur est à 0
        return; 
    }
    
    // Calculer la position actuelle dans le tour (entre 1 et 2047)
    long currentPos = stepCounter % STEPS_PER_REVOLUTION;
    if (currentPos < 0) currentPos += STEPS_PER_REVOLUTION; // Gestion des pas négatifs

    // Déterminer la direction de rotation la plus courte pour le retour
    // Si la position est dans la seconde moitié du tour (par exemple 1025 à 2047), 
    // on tourne dans la direction opposée (-1) pour faire le chemin le plus court
    if (currentPos > STEPS_PER_REVOLUTION / 2) {
      stepDirection = -1; // Sens Anti-Horaire
    } else {
      stepDirection = 1; // Sens Horaire
    }
    
    moving = true;
    homing = true;
  }

  // Changer la vitesse à la volée
  void setSpeed(int delayMs) {
    stepDelay = delayMs;
  }

  // Mise à jour à appeler dans loop()
  void update() {
    if (!moving) return;
    long now = millis();
    if (now - lastStepTime >= stepDelay) {
      // Logique pour avancer dans la séquence de pas
      currentStep = (currentStep + stepDirection + 8) % 8;
      setStep(currentStep);
      lastStepTime = now;
      
      // Logique de comptage des pas
      stepCounter += stepDirection;
      
      // Logique de retour 'home'
      if (homing && (stepCounter % STEPS_PER_REVOLUTION) == 0) {
        // Position 'home' atteinte
        stop(); 
        stepCounter = 0; // Confirmer la position 'home'
        Serial.print("PLATEAU ");
        Serial.print(pin1); 
        Serial.println(" HOME ATTEINT");
      }
    }
  }

  // Relâcher les bobines pour économiser de l'énergie
  void release() {
    digitalWrite(pin1, LOW);
    digitalWrite(pin2, LOW);
    digitalWrite(pin3, LOW);
    digitalWrite(pin4, LOW);
  }

  bool isMoving() {
    return moving;
  }
};

/**
 * Gestionnaire de moteurs pas à pas
 */
class StepperManager {
private:
  Stepper28BYJ48 *motors[MAX_MOTORS];
  int motorCount;
public:
  StepperManager() {
    motorCount = 0;
    for (int i = 0; i < MAX_MOTORS; i++) motors[i] = nullptr;
  }

  // Ajouter un moteur au gestionnaire
  bool addMotor(Stepper28BYJ48 *motor) {
    if (motorCount >= MAX_MOTORS) return false;
    motors[motorCount++] = motor;
    return true;
  }

  // Démarrer un moteur spécifique
  void startMotor(int index, int direction = 1) {
    if (index < motorCount) motors[index]->start(direction);
  }

  // Arrêter un moteur spécifique
  void stopMotor(int index) {
    if (index < motorCount) motors[index]->stop();
  }
  
  // Demander au moteur de revenir à la position 'home'
  void homeMotor(int index) {
    if (index < motorCount) motors[index]->startHoming();
  }

  // Régler la vitesse d'un moteur
  void setMotorSpeed(int index, int delayMs) {
    if (index < motorCount) motors[index]->setSpeed(delayMs);
  }

  // Mettre à jour tous les moteurs
  void updateAll() {
    for (int i = 0; i < motorCount; i++) motors[i]->update();
  }

  // Arrêter tous les moteurs
  void stopAll() {
    for (int i = 0; i < motorCount; i++) motors[i]->stop();
  }

  // Vérifier si au moins un moteur tourne
  bool anyMoving() {
    for (int i = 0; i < motorCount; i++) {
      if (motors[i]->isMoving()) return true;
    }
    return false;
  }
};

/**
 * Gestion d'une lampe LED
 */
class Lamp {
private:
  int pinPWM;  // Broche PWM
  bool invert;  // false = actif HIGH, true = actif LOW

public:
  // Constructeur
  Lamp(int pwm, bool inv = false) {
    pinPWM = pwm;
    invert = inv;
    pinMode(pinPWM, OUTPUT);
    off();
  }

  // Allumer à une intensité donnée (0 à 255)
  void set(int value) {
    value = constrain(value, 0, 255);
    if (invert) {
      analogWrite(pinPWM, 255 - value);
    } else {
      analogWrite(pinPWM, value);
    }
  }

  // Allumer à 100 %
  void on() {
    set(255);
  }

  // Éteindre
  void off() {
    set(0);
  }

  // Allumer à un pourcentage (0 à 100 %)
  void setPercent(int percent) {
    if (percent < 0) percent = 0;
    if (percent > 100) percent = 100;
    int value = map(percent, 0, 100, 0, 255);
    set(value);
  }
};
// ================================
// DÉCLARATIONS DES PÉRIPHÉRIQUES
// ================================

// Moteurs pas à pas pour les plateaux
Stepper28BYJ48 plateau1(28, 26, 24, 22, 3);
Stepper28BYJ48 plateau2(36, 34, 32, 30, 3);
StepperManager manager;

// Moteur DC pour le tapis roulant
Motor tapis = Motor(4, 5, 6);
// Lampes LED
Lamp lampe1(8);
Lamp lampe2(9);
Lamp lampe3(10);
Lamp lampe4(11);
// ================================
// VARIABLES GLOBALES POUR LES EFFETS
// ================================

int fireId, smokeId, rainbowId, blinkId;
int fixedId, chaseId, snowId;
// ================================
// GESTION DES EFFETS LAMPES 
// ================================

struct LampEffect {
  bool active;      // Vrai si l'effet est actif
  Lamp *lamp;              // Pointeur vers la lampe concernée
  int intensity;           // Intensité maximale
  int interval;            // Intervalle de clignotement (ms)
  unsigned long lastUpdate;  // Dernière mise à jour
  bool state;              // État actuel (true=allumé, false=éteint)
};
// Tableau des effets de lampes (un par lampe)
LampEffect lampEffects[4] = {
    {false, &lampe1, 0, 0, 0, false},
    {false, &lampe2, 0, 0, 0, false},
    {false, &lampe3, 0, 0, 0, false},
    {false, &lampe4, 0, 0, 0, false}
};
/**
 * Mettre à jour tous les effets de lampes en cours 
 */
void updateLampEffects() {
  unsigned long now = millis();
  for (int i = 0; i < 4; i++) {
    if (!lampEffects[i].active) continue;
    if (now - lampEffects[i].lastUpdate >= lampEffects[i].interval) {
      lampEffects[i].lastUpdate = now;
      lampEffects[i].state = !lampEffects[i].state;
      if (lampEffects[i].state) {
        // Allumer à l'intensité définie
        lampEffects[i].lamp->set(lampEffects[i].intensity);
      } else {
        // Éteindre
        lampEffects[i].lamp->set(0);
      }
    }
  }
}
// ================================
// FONCTIONS DE BASE POUR LES RUBANS LED
// ================================

/**
 * Mettre à jour l'affichage d'un ruban
 */
void updateStrip(Adafruit_NeoPixel &s) {
  while (!s.canShow())
    ;
  s.show();
}

/**
 * Initialiser un ruban
 */
void initStrip(Adafruit_NeoPixel &s, uint32_t color = 0) {
  s.begin();
  s.fill(color);
  updateStrip(s);
}

/**
 * Régler la luminosité d'un ruban
 */
void setBrightness(Adafruit_NeoPixel &s, uint8_t b) {
  s.setBrightness(b);
  updateStrip(s);
}

// ================================
// GESTION DES EFFETS LED
// ================================

/**
 * Ajouter un nouvel effet
 */
int addEffect(Adafruit_NeoPixel &s, int startIndex, int endIndex, EffectType type,
              uint32_t color1 = 0, uint32_t color2 = 0, int interval = 0, uint8_t density = 0) {
  if (effectCount >= MAX_EFFECTS) return -1;
  bex = true;
  effects[effectCount] = { nextEffectId, &s, startIndex, endIndex, type, color1, color2, interval,
                           millis(), false, 0, true, density };
  int id = nextEffectId++;
  effectCount++;
  bex = false;
  return id;
}

/**
 * Supprimer un effet par son ID
 */
void removeEffectById(int id) {
  bex = true;
  for (int i = 0; i < effectCount; i++) {
    if (effects[i].id == id) {
      // Éteindre les LEDs de l'effet
      effects[i].active = false;
      for (int j = effects[i].startIndex; j <= effects[i].endIndex; j++)
        effects[i].strip->setPixelColor(j, 0);
      updateStrip(*effects[i].strip);
      // Décaler les effets restants
      for (int j = i; j < effectCount - 1; j++)
        effects[j] = effects[j + 1];
      effectCount--;
      break;
    }
  }
  bex = false;
}

// ================================
// FONCTIONS DE DÉMARRAGE DES EFFETS
// ================================

/**
 * Démarrer un effet de couleur fixe
 */
int startFixed(Adafruit_NeoPixel &s, int start, int end, int r, int g, int b) {
  uint32_t color = s.Color(r, g, b);
  for (int i = start; i <= end; i++) s.setPixelColor(i, color);
  updateStrip(s);
  return addEffect(s, start, end, FIXED, color);
}

/**
 * Démarrer un effet de clignotement
 */
int startBlink(Adafruit_NeoPixel &s, int start, int end, int r1, int g1, int b1, int r2, int g2, int b2, int interval) {
  uint32_t c1 = s.Color(r1, g1, b1);
  uint32_t c2 = s.Color(r2, g2, b2);
  return addEffect(s, start, end, BLINK, c1, c2, interval);
}

/**
 * Démarrer un effet de poursuite
 */
int startChase(Adafruit_NeoPixel &s, int start, int end, int r, int g, int b, int rb, int gb, int bb, int interval) {
  uint32_t color = s.Color(r, g, b);
  int32_t cf = s.Color(rb, gb, bb);
  return addEffect(s, start, end, CHASE, color, cf, interval);
}

/**
 * Démarrer un effet de fondu
 */
int startFade(Adafruit_NeoPixel &s, int start, int end, int r1, int g1, int b1, int r2, int g2, int b2, int interval) {
  uint32_t c1 = s.Color(r1, g1, b1);
  uint32_t c2 = s.Color(r2, g2, b2);
  return addEffect(s, start, end, FADE, c1, c2, interval, 5);
}

/**
 * Démarrer un effet arc-en-ciel
 */
int startRainbow(Adafruit_NeoPixel &s, int start, int end, int interval) {
  return addEffect(s, start, end, RAINBOW, 0, 0, interval);
}

/**
 * Démarrer un effet feu
 */
int startFire(Adafruit_NeoPixel &s, int start, int end, int interval) {
  return addEffect(s, start, end, FIRE, 0, 0, interval);
}

/**
 * Démarrer un effet neige
 */
int startSnow(Adafruit_NeoPixel &s, int start, int end, int interval, int density) {
  return addEffect(s, start, end, SNOW, 0, 0, interval, uint8_t(density));
}

/**
 * Démarrer un effet fumée
 */
int startSmoke(Adafruit_NeoPixel &s, int start, int end, int interval, int density) {
  return addEffect(s, start, end, SMOKE, 0, 0, interval, uint8_t(density));
}

// ================================
// FONCTIONS UTILITAIRES
// ================================

/**
 * Générer une couleur à partir d'une position dans la roue chromatique
 */
uint32_t Wheel(byte WheelPos, Adafruit_NeoPixel &strip) {
  WheelPos = 255 - WheelPos;
  if (WheelPos < 85) return strip.Color(255 - WheelPos * 3, 0, WheelPos * 3);
  if (WheelPos < 170) {
    WheelPos -= 85;
    return strip.Color(0, WheelPos * 3, 255 - WheelPos * 3);
  }
  WheelPos -= 170;
  return strip.Color(WheelPos * 3, 255 - WheelPos * 3, 0);
}

/**
 * Envoyer un message de réponse
 */
void sendMsg(int a, int b, int c) {
  Serial.print(a);
  Serial.print(',');
  Serial.print(b);
  Serial.print(',');
  Serial.println(c);
}

// ================================
// MISE À JOUR GÉNÉRALE DES EFFETS
// ================================

/**
 * Mettre à jour tous les effets en cours
 */
void update() {
  if (bex) return;
  unsigned long now = millis();
  for (int i = 0; i < effectCount; i++) {
    if (!effects[i].active) continue;
    switch (effects[i].type) {
      case FIXED:
        // Aucune mise à jour nécessaire pour l'effet fixe
        break;
      case BLINK:
        if (now - effects[i].lastUpdate >= effects[i].interval) {
          effects[i].lastUpdate = now;
          effects[i].toggle = !effects[i].toggle;
          uint32_t color = effects[i].toggle ? effects[i].color1 : effects[i].color2;
          for (int j = effects[i].startIndex; j <= effects[i].endIndex; j++)
            effects[i].strip->setPixelColor(j, color);
          updateStrip(*effects[i].strip);
        }
        break;
      case CHASE:
        if (now - effects[i].lastUpdate >= effects[i].interval) {
          effects[i].lastUpdate = now;
          int len = effects[i].endIndex - effects[i].startIndex + 1;
          for (int j = 0; j < len; j++) {
            int idx = effects[i].startIndex + j;
            effects[i].strip->setPixelColor(idx, (j == effects[i].step) ? effects[i].color1 : effects[i].color2);
          }
          updateStrip(*effects[i].strip);
          effects[i].step = (effects[i].step + 1) % len;
        }
        break;
      case FADE:
        if (now - effects[i].lastUpdate >= effects[i].interval) {
          effects[i].lastUpdate = now;
          uint8_t r1 = (effects[i].color1 >> 16) & 0xFF;
          uint8_t g1 = (effects[i].color1 >> 8) & 0xFF;
          uint8_t b1 = effects[i].color1 & 0xFF;
          int step = effects[i].toggle ? effects[i].brightnessStep : -effects[i].brightnessStep;
          effects[i].toggle = !effects[i].toggle;
          for (int j = effects[i].startIndex; j <= effects[i].endIndex; j++) {
            uint8_t r = constrain(r1 + step, 0, 255);
            uint8_t g = constrain(g1 + step, 0, 255);
            uint8_t b = constrain(b1 + step, 0, 255);
            effects[i].strip->setPixelColor(j, effects[i].strip->Color(r, g, b));
          }
          updateStrip(*effects[i].strip);
        }
        break;
      case RAINBOW:
        if (now - effects[i].lastUpdate >= effects[i].interval) {
          effects[i].lastUpdate = now;
          int len = effects[i].endIndex - effects[i].startIndex + 1;
          for (int j = 0; j < len; j++) {
            int idx = effects[i].startIndex + j;
            effects[i].strip->setPixelColor(idx, Wheel((j + effects[i].step) & 255, *effects[i].strip));
          }
          updateStrip(*effects[i].strip);
          effects[i].step = (effects[i].step + 1) % 256;
        }
        break;
      case FIRE:
        if (now - effects[i].lastUpdate >= effects[i].interval) {
          effects[i].lastUpdate = now;
          int len = effects[i].endIndex - effects[i].startIndex + 1;
          for (int j = 0; j < len; j++) {
            int flicker = random(160, 255);
            uint8_t r = flicker;
            uint8_t g = flicker / 2;
            uint8_t b = 0;
            effects[i].strip->setPixelColor(effects[i].startIndex + j, effects[i].strip->Color(r, g, b));
          }
          updateStrip(*effects[i].strip);
        }
        break;
      case SNOW:
        if (now - effects[i].lastUpdate >= effects[i].interval) {
          effects[i].lastUpdate = now;
          int len = effects[i].endIndex - effects[i].startIndex + 1;
          for (int j = len - 1; j > 0; j--) {
            uint32_t c = effects[i].strip->getPixelColor(effects[i].startIndex + j - 1);
            effects[i].strip->setPixelColor(effects[i].startIndex + j, c);
          }
          if (random(0, 255) < effects[i].brightnessStep)
            effects[i].strip->setPixelColor(effects[i].startIndex, effects[i].strip->Color(255, 255, 255));
          else
            effects[i].strip->setPixelColor(effects[i].startIndex, 0);
          updateStrip(*effects[i].strip);
        }
        break;

      case SMOKE:
        if (now - effects[i].lastUpdate >= effects[i].interval) {
          effects[i].lastUpdate = now;
          int len = effects[i].endIndex - effects[i].startIndex + 1;
          for (int j = len - 1; j > 0; j--) {
            uint32_t c = effects[i].strip->getPixelColor(effects[i].startIndex + j - 1);
            uint8_t r = ((c >> 16) & 0xFF) * 0.8;
            uint8_t g = ((c >> 8) & 0xFF) * 0.8;
            uint8_t b = (c & 0xFF) * 0.8;
            effects[i].strip->setPixelColor(effects[i].startIndex + j, effects[i].strip->Color(r, g, b));
          }
          //if (random(0, 255) < effects[i].brightnessStep) {
          if (random(0, 255) < 150) {
            uint8_t gray = random(100, 200);
            effects[i].strip->setPixelColor(effects[i].startIndex, effects[i].strip->Color(gray, gray, gray));
          } else
            effects[i].strip->setPixelColor(effects[i].startIndex, 0);
          updateStrip(*effects[i].strip);
        }
        break;
    }
  }
}

// ================================
// GESTION DES COMMANDES
// ================================

/**
 * Gérer les commandes des moteurs (MODIFIÉ pour Home)
 */
int gerer_moteurs(int cmd, int speed) {
  switch (cmd) {
    case 100:  // Arrêt tapis
      tapis.stop();
      return OK;
    case 101:  // Tapis avant
      tapis.forward(speed);
      return OK;
    case 102:  // Tapis arrière
      tapis.backward(speed);
      return OK;
    case 110:  // Plateau1 stop
      manager.stopMotor(0);
      return OK;
    case 111:  // Plateau1 sens horaire
      manager.startMotor(0, 1);
      manager.setMotorSpeed(0, speed);
      return OK;
    case 112:  // Plateau1 sens antihoraire
      manager.startMotor(0, -1);
      manager.setMotorSpeed(0, speed);
      return OK;
    case 113:  // Plateau1 Home (AJOUT)
      manager.homeMotor(0);
      return OK;
    case 120:  // Plateau2 stop
      manager.stopMotor(1);
      return OK;
    case 121:  // Plateau2 sens horaire
      manager.startMotor(1, 1);
      manager.setMotorSpeed(1, speed);
      return OK;
    case 122:  // Plateau2 sens antihoraire
      manager.startMotor(1, -1);
      manager.setMotorSpeed(1, speed);
      return OK;
    case 123:  // Plateau2 Home (AJOUT)
      manager.homeMotor(1);
      return OK;
    default:
      return ER;
  }
}

/**
 * Gérer les commandes des lampes
 */
int gerer_lampes(int cmd, int param1, int param2) {
  int index = (cmd / 10) % 10;
  
  if (index >= 4) return ER;

  // Arrêter tout clignotement avant une commande fixe
  if (cmd % 10 != 2) {
    lampEffects[index].active = false;
  }

  switch (cmd) {
    // Commandes d'arrêt
    case 200: 
    case 210: 
    case 220: 
    case 230: 
      lampEffects[index].lamp->set(0);
      return OK;

    // Commandes d'intensité fixe
    case 201: 
    case 211: 
    case 221: 
    case 231: 
      lampEffects[index].lamp->set(param1);
      return OK;

    // Commandes de clignotement (BLINK)
    case 202: 
    case 212: 
    case 222: 
    case 232: 
      lampEffects[index].intensity = constrain(param1, 0, 255);
      lampEffects[index].interval = constrain(param2, 10, 5000);
      lampEffects[index].lastUpdate = millis();
      lampEffects[index].state = false; 
      lampEffects[index].active = true;
      return OK;
      
    // Commandes d'arrêt clignotement (STOP BLINK)
    case 203: 
    case 213: 
    case 223: 
    case 233: 
      lampEffects[index].lamp->set(0);
      return OK;
      
    default:
      return ER;
  }
}

/**
 * Traiter une ligne de commande reçue
 */
void processLine(char *line) {
  int t[MAX_VALUES];
  int retCode;
  uint8_t count = 0;
  // Découpage de la ligne en valeurs
  char *token = strtok(line, ",");
  while (token != NULL && count < MAX_VALUES) {
    t[count] = atoi(token);
    count++;
    token = strtok(NULL, ",");
  }
  // Vérification de la longueur du message
  if (count < 3) {
    sendMsg(0, ER, MSG_LENGTH_3);
    return;
  }
  int num = t[0];  // Numéro de séquence
  int cmd = t[1];  // Commande

  // Routage vers le gestionnaire approprié
  if (cmd >= 100 && cmd < 200) {
    // Moteurs
    retCode = gerer_moteurs(cmd, t[2]);
    sendMsg(num, retCode, 0);
    return;
  }
  if (cmd >= 200 && cmd < 299) {
    // Lampes
    int param1 = t[2];
    int param2 = (count > 3) ? t[3] : 0;
    
    // Vérification du nombre de paramètres pour le clignotement (BLINK: 4 params)
    if ((cmd == 202 || cmd == 212 || cmd == 222 || cmd == 232) && count < 4) {
      sendMsg(num, ER, PARAM_COUNT);
      return;
    }
    
    retCode = gerer_lampes(cmd, param1, param2);
    sendMsg(num, retCode, 0);
    return;
  }
  
  // Gestion des effets LED
  int ruban = t[2];
  int id;
  if (cmd != STOP)
    if (ruban < 0 || ruban > STRIPCOUNT) {
      sendMsg(num, ER, STRIP_NUMBER);
      return;
    }
  if (count != nbParam[cmd]) {
    sendMsg(num, ER, PARAM_COUNT);
    return;
  }
  switch (cmd) {
    case STOP:
      removeEffectById(ruban);
      sendMsg(num, OK, 0);
      return;
    case FIXED:
      id = startFixed(*strips[ruban], t[3], t[4], t[5], t[6], t[7]);
      break;
    case BLINK:
      id = startBlink(*strips[ruban], t[3], t[4], t[5], t[6], t[7], t[8], t[9], t[10], t[11]);
      break;
    case CHASE:
      id = startChase(*strips[ruban], t[3], t[4], t[5], t[6], t[7], t[8], t[9], t[10], t[11]);
      break;
    case FADE:
      id = startFade(*strips[ruban], t[3], t[4], t[5], t[6], t[7], t[8], t[9], t[10], t[11]);
      break;
    case RAINBOW:
      id = startRainbow(*strips[ruban], t[3], t[4], t[5]);
      break;
    case FIRE:
      id = startFire(*strips[ruban], t[3], t[4], t[5]);
      break;
    case SNOW:
      id = startSnow(*strips[ruban], t[3], t[4], t[5], t[6]);
      break;
    case SMOKE:
      id = startSmoke(*strips[ruban], t[3], t[4], t[5], t[6]);
      break;
    default:
      sendMsg(num, ER, UNKNOWN_CMD);
      return;
  }
  sendMsg(num, OK, id);
}

// ================================
// FONCTIONS PRINCIPALES ARDUINO
// ================================

/**
 * Initialisation
 */
void setup() {
  int i;
  Serial.begin(115200);
  // Initialisation des rubans LED
  for (i = 0; i < STRIPCOUNT; i++) {
    initStrip(*strips[i]);
    setBrightness(*strips[i], 32);
  }
  // Configuration des moteurs pas à pas
  manager.addMotor(&plateau1);
  manager.addMotor(&plateau2);
}

/**
 * Boucle principale
 */
void loop() {
  // Réception des commandes série
  while (Serial.available() > 0) {
    char c = Serial.read();
    if (c == '\n') {
      inputBuffer[bufIndex] = '\0';
      lineReady = true;
      bufIndex = 0;
    } else if (c != '\r') {
      if (bufIndex < BUFFER_SIZE - 1) {
        inputBuffer[bufIndex++] = c;
      } else {
        bufIndex = 0;
      }
    }
  }
  // Traitement des commandes reçues
  if (lineReady) {
    processLine(inputBuffer);
    lineReady = false;
  }
  // Mise à jour des périphériques
  manager.updateAll();  // Moteurs pas à pas
  update();             // Effets LED
  updateLampEffects();  // Mises à jour des clignotements de lampe non-bloquantes
}
