import json
import time
import threading
import sys
import subprocess      
from screeninfo import get_monitors 
from effectcontroller import EffectController
from arduinocontroller import ArduinoController
# ==============================================================
# CLASSE VIDEO PLAYER (Gère un seul écran)
# ==============================================================
class VideoPlayer:
    """
    Gère la lecture d'une vidéo et la synchronisation des effets sur UN seul écran.
    """
    def __init__(self, screen_id: int, arduino: ArduinoController):
        self.screen_id = screen_id
        self.arduino = arduino
        self.effect_controller = EffectController(arduino=arduino)
        
        # --- LOGIQUE DE GÉOMÉTRIE (Calcul une seule fois) ---
        monitors = get_monitors()
        self.geometry_arg = ""
        
        if self.screen_id < len(monitors):
            monitor = monitors[self.screen_id]
            # Calcul de la géométrie WxH+X+Y
            self.geometry_arg = f"{monitor.width}x{monitor.height}+{monitor.x}+{monitor.y}"
            
        print(f"🖥️ [Écran {screen_id}] Géométrie calculée : {self.geometry_arg}")

        # Remplacement de l'objet mpv par le processus vidéo
        self.video_process = None 
        
        # Variables de synchronisation par horloge système
        self.start_time_monotonic = 0 
        self.audio_enabled_flag = False 

        self.json_path = ""
        self.video_path = ""
        self.effects = []
        self.effects_in_progress = []
        self.next_effect_index = 0
        
        self.sync_thread = None
        self.playback_done = threading.Event()

    # NOTE: L'observer 'eof-reached' n'est plus nécessaire/utilisable car mpv
    # est lancé en processus séparé. La détection de fin est gérée dans _run_sync_loop.

    def _load_json(self):
        try:
            with open(self.json_path, "r", encoding="utf-8") as f:
                data = json.load(f)
            self.video_path = data.get("video_path", "placeholder.mp4")
            print(f"🎞️ [Écran {self.screen_id}] Vidéo chargée : {self.video_path}")
            return data["effects"]
        except (FileNotFoundError, json.JSONDecodeError) as e:
            print(f"❌ Erreur de chargement du JSON à {self.json_path}: {e}")
            return []

    # ... _parse_description, _start_effect_and_get_id, _stop_effect restent inchangés ...
    def _parse_description(self, effect):
        params = {}
        for line in effect.get("description", "").splitlines():
            if "=" in line:
                k, v = line.split("=", 1)
                params[k.strip()] = v.strip()
        if "type" in params:
            params["type"] = params["type"].lower()
        return params

    def _start_effect_and_get_id(self, effect, params):
        effect_id = self.effect_controller.start_effect(effect)
        print(f"🟢 [Écran {self.screen_id}] Start {effect['name']} (id={effect_id})")
        return effect_id

    def _stop_effect(self, e_data):
        effect = e_data["effect"]
        effect_id = e_data["id"]
        print(f"🔴 [Écran {self.screen_id}] Stop {effect['name']} (id={effect_id})")
        self.effect_controller.stop_effect(effect_id)


    def play(self, json_path: str, audio_enabled: bool):
        """
        Démarre la lecture de la vidéo spécifiée par le JSON.
        """
        self.json_path = json_path
        self.effects = self._load_json()
        
        # AMÉLIORATION : Tri des effets par start_time pour s'assurer que
        # la boucle de synchro démarre les effets dans le bon ordre chronologique.
        self.effects.sort(key=lambda e: e['start_time'])
        
        self.audio_enabled_flag = audio_enabled # Stocke l'état audio
        
        #if not self.effects:
        #    print(f"🛑 [Écran {self.screen_id}] Lecture annulée : Aucun effet trouvé ou erreur de chargement.")
        #    return

        # Réinitialisation de l'état
        self.playback_done.clear()
        self.next_effect_index = 0
        self.effects_in_progress = []

        # Affichage de l'état audio
        sound_status = "activé" if audio_enabled else "désactivé"
        print(f"🔊 [Écran {self.screen_id}] Son {sound_status}.")

        # Démarrage de la boucle de synchro dans un thread
        self.sync_thread = threading.Thread(target=self._run_sync_loop, daemon=True)
        self.sync_thread.start()

    def is_done(self) -> bool: # <-- AJOUTER CETTE MÉTHODE
        """
        Vérifie si la lecture de la vidéo actuelle est terminée
        en regardant l'état de l'événement de synchronisation.
        """
        return self.playback_done.is_set()
        
    def stop(self):
        """
        Arrête la lecture en cours et nettoie les effets.
        """
        if not self.playback_done.is_set():
            print(f"⏹️ [Écran {self.screen_id}] Arrêt manuel demandé.")
            self.playback_done.set()
            
        # Arrêt du processus MPV s'il est en cours
        if self.video_process:
            print(f"🔪 [Écran {self.screen_id}] Arrêt du processus MPV.")
            self.video_process.terminate()
            self.video_process.wait(timeout=1)
            self.video_process = None
            
        # Attendre que la boucle de synchro se termine
        if self.sync_thread and self.sync_thread.is_alive():
            self.sync_thread.join(timeout=1)

        self._cleanup_effects()
        print(f"✅ [Écran {self.screen_id}] Lecteur arrêté et nettoyé.")

    def _run_sync_loop(self):
        """
        La boucle principale qui synchronise les effets.
        Lance la vidéo via subprocess et utilise l'horloge système.
        """
        print(f"▶️ [Écran {self.screen_id}] Démarrage de la lecture via subprocess...")
        
        # --- Lancement de la vidéo via subprocess.Popen ---
        cmd = ["mpv", 
               "--vo=x11",
               #"--vo=gpu", 
               "--ao=pulse",
               "--no-border", 
               #"--loop=inf", # option pour rejouer la vidéo en boucle (peut être enlevée)
               "--input-terminal=no",
               f"--geometry={self.geometry_arg}", 
               self.video_path]
        
        if not self.audio_enabled_flag:
             cmd.append("--mute")
        
        self.video_process = subprocess.Popen(cmd,stdin=subprocess.DEVNULL)

        # Enregistrement du temps de démarrage réel pour la synchro
        #GG A MODIFIER : -1.0 ajouté , peut être mettre la valeur en paramètre du constructeur
        self.start_time_monotonic = time.monotonic()-1.0
        # Boucle de synchronisation des effets basée sur le temps écoulé
        while not self.playback_done.is_set():
            
            # Vérifie si le processus mpv s'est terminé (par exemple, si la vidéo n'est pas en boucle)
            if self.video_process and self.video_process.poll() is not None:
                 print(f"🎬 [Écran {self.screen_id}] Processus MPV terminé (fin naturelle ou erreur).")
                 self.playback_done.set()
                 break
            
            # Temps écoulé depuis le début de la lecture (en ms)
            current_time_ms = int((time.monotonic() - self.start_time_monotonic) * 1000)
            
            # --- Démarrage des effets planifiés ---
            while self.next_effect_index < len(self.effects) and current_time_ms >= self.effects[self.next_effect_index]["start_time"]:
                e = self.effects[self.next_effect_index]
                params = self._parse_description(e)
                effect_id = self._start_effect_and_get_id(e, params)
                self.effects_in_progress.append({"effect": e, "id": effect_id, "params": params})
                self.next_effect_index += 1

            # --- Arrêt des effets expirés ---
            for e_data in self.effects_in_progress[:]:
                if current_time_ms >= e_data["effect"]["end_time"]:
                    self._stop_effect(e_data)
                    self.effects_in_progress.remove(e_data)

            time.sleep(0.02)
        
    def _cleanup_effects(self):
        if self.effects_in_progress:
            print(f"🧹 [Écran {self.screen_id}] Nettoyage des effets restants...")
            for e_data in self.effects_in_progress[:]:
                self._stop_effect(e_data)
                self.effects_in_progress.remove(e_data)
