import os
import shutil
import hashlib
import json
import tkinter as tk
from tkinter import filedialog, messagebox, scrolledtext, ttk
import threading
import queue

# --- LOGIQUE DE CALCUL ---

def file_hash(path, algo="sha256", chunk_size=8192):
    """Calcule le hash d'un fichier pour vérifier l'intégrité."""
    h = hashlib.new(algo)
    try:
        with open(path, "rb") as f:
            while chunk := f.read(chunk_size):
                h.update(chunk)
        return h.hexdigest()
    except Exception:
        return None


# --- APPLICATION ---

class SyncApp:
    def __init__(self, root):
        self.root = root
        self.root.title("SyncVision Pro - Mode Miroir Intégral")
        self.config_file = "config.json"

        self.ui_queue = queue.Queue()

        self.src_path = tk.StringVar()
        self.dst_path = tk.StringVar()
        self.mirror_mode = tk.BooleanVar(value=False)

        self.count_total = tk.IntVar(value=0)
        self.count_copied = tk.IntVar(value=0)
        self.count_skipped = tk.IntVar(value=0)
        self.count_deleted = tk.IntVar(value=0)

        self.progress_var = tk.DoubleVar()
        self.is_running = False

        self.load_config()
        self.setup_ui()

        self.root.protocol("WM_DELETE_WINDOW", self.on_close)
        self.root.after(100, self.process_ui_queue)

    def setup_ui(self):
        style = ttk.Style()
        style.configure("TProgressbar", thickness=20)

        frame_top = tk.LabelFrame(self.root, text=" Paramètres de synchronisation ", padx=10, pady=10)
        frame_top.pack(fill='x', padx=15, pady=10)

        tk.Label(frame_top, text="Source :").grid(row=0, column=0, sticky='w')
        tk.Entry(frame_top, textvariable=self.src_path).grid(row=0, column=1, sticky='ew', padx=5)
        tk.Button(frame_top, text="Parcourir", command=lambda: self.browse_folder(self.src_path)).grid(row=0, column=2)

        tk.Label(frame_top, text="Destination :").grid(row=1, column=0, sticky='w')
        tk.Entry(frame_top, textvariable=self.dst_path).grid(row=1, column=1, sticky='ew', padx=5)
        tk.Button(frame_top, text="Parcourir", command=lambda: self.browse_folder(self.dst_path)).grid(row=1, column=2)

        tk.Checkbutton(
            frame_top,
            text="Activer le mode Miroir (supprime fichiers ET dossiers orphelins)",
            variable=self.mirror_mode
        ).grid(row=2, column=0, columnspan=3, sticky='w', pady=5)

        frame_top.columnconfigure(1, weight=1)

        frame_stats = tk.Frame(self.root)
        frame_stats.pack(fill='x', padx=15)

        stats = [
            ("Total :", self.count_total, "blue"),
            ("Copiés :", self.count_copied, "green"),
            ("Identiques :", self.count_skipped, "gray"),
            ("Supprimés :", self.count_deleted, "red")
        ]

        for label, var, color in stats:
            tk.Label(frame_stats, text=label).pack(side='left')
            tk.Label(frame_stats, textvariable=var, fg=color, font=('Segoe UI', 10, 'bold')).pack(side='left', padx=(0, 15))

        self.progress_bar = ttk.Progressbar(self.root, variable=self.progress_var, maximum=100)
        self.progress_bar.pack(fill='x', padx=15, pady=10)

        self.log_area = scrolledtext.ScrolledText(
            self.root, height=12, state='disabled', bg="#1e1e1e", fg="#00ff00", font=('Consolas', 9)
        )
        self.log_area.pack(fill='both', expand=True, padx=15, pady=5)

        self.btn_sync = tk.Button(
            self.root, text="Lancer la synchronisation", bg="#2ecc71", fg="white",
            font=('Segoe UI', 10, 'bold'), height=2, command=self.start_sync
        )
        self.btn_sync.pack(fill='x', padx=15, pady=15)

    def log(self, message):
        self.log_area.configure(state='normal')
        self.log_area.insert(tk.END, message + "\n")
        self.log_area.see(tk.END)
        self.log_area.configure(state='disabled')

    def browse_folder(self, var):
        folder = filedialog.askdirectory()
        if folder:
            var.set(folder)

    def load_config(self):
        if os.path.exists(self.config_file):
            try:
                with open(self.config_file, "r") as f:
                    cfg = json.load(f)
                self.src_path.set(cfg.get("src", ""))
                self.dst_path.set(cfg.get("dst", ""))
                self.mirror_mode.set(cfg.get("mirror", False))
                self.root.geometry(cfg.get("geometry", "850x650"))
            except Exception:
                self.root.geometry("850x650")
        else:
            self.root.geometry("850x650")

    def on_close(self):
        cfg = {
            "src": self.src_path.get(),
            "dst": self.dst_path.get(),
            "mirror": self.mirror_mode.get(),
            "geometry": self.root.winfo_geometry()
        }
        with open(self.config_file, "w") as f:
            json.dump(cfg, f)
        self.root.destroy()

    # ---------- LOGIQUE DE SYNCHRONISATION ----------

    def start_sync(self):
        if not self.src_path.get() or not self.dst_path.get():
            messagebox.showwarning("Erreur", "Veuillez sélectionner les dossiers.")
            return

        if self.is_running:
            return

        self.is_running = True
        self.count_total.set(0)
        self.count_copied.set(0)
        self.count_skipped.set(0)
        self.count_deleted.set(0)
        self.progress_var.set(0)

        self.btn_sync.config(state='disabled', text="Opération en cours...", bg="#95a5a6")
        self.log("--- Début de l'analyse ---")

        threading.Thread(target=self.run_sync_worker, daemon=True).start()

    def run_sync_worker(self):
        src = self.src_path.get()
        dst = self.dst_path.get()

        try:
            # 1. Scanner la source
            src_files = {}
            src_dirs = set()
            for root_dir, dirs, files in os.walk(src):
                dirs[:] = [d for d in dirs if not d.startswith('.')]
                for d in dirs:
                    rel_d = os.path.relpath(os.path.join(root_dir, d), src)
                    src_dirs.add(rel_d)
                for f in files:
                    if not f.startswith('.'):
                        abs_p = os.path.join(root_dir, f)
                        rel_p = os.path.relpath(abs_p, src)
                        src_files[rel_p] = abs_p

            self.ui_queue.put(("total", len(src_files)))

            # 2. Copier/Mettre à jour
            copied = skipped = 0
            for i, (rel, src_file) in enumerate(src_files.items()):
                dst_file = os.path.join(dst, rel)
                os.makedirs(os.path.dirname(dst_file), exist_ok=True)

                copy_needed = False
                if not os.path.exists(dst_file):
                    copy_needed = True
                elif os.path.getsize(src_file) != os.path.getsize(dst_file):
                    copy_needed = True
                elif file_hash(src_file) != file_hash(dst_file):
                    copy_needed = True

                if copy_needed:
                    shutil.copy2(src_file, dst_file)
                    copied += 1
                    self.ui_queue.put(("log", f"[COPIE] {rel}"))
                    self.ui_queue.put(("copied", copied))
                else:
                    skipped += 1
                    self.ui_queue.put(("skipped", skipped))

                self.ui_queue.put(("progress", ((i + 1) / max(len(src_files), 1)) * 100))

            # 3. Mode Miroir (Détection Orphelins)
            if self.mirror_mode.get():
                self.ui_queue.put(("log", "Recherche d'éléments à supprimer..."))
                orphans = [] # Fichiers et Dossiers
                
                # topdown=False permet de lister les sous-dossiers avant leurs parents
                for root_dir, dirs, files in os.walk(dst, topdown=False):
                    for f in files:
                        p = os.path.join(root_dir, f)
                        rel = os.path.relpath(p, dst)
                        if rel not in src_files:
                            orphans.append(('file', p))
                    
                    for d in dirs:
                        p = os.path.join(root_dir, d)
                        rel = os.path.relpath(p, dst)
                        if rel not in src_dirs:
                            orphans.append(('dir', p))

                if orphans:
                    self.ui_queue.put(("confirm_mirror", orphans, dst))
                else:
                    self.ui_queue.put(("done", "Sync terminée (aucun orphelin)."))
            else:
                self.ui_queue.put(("done", "Synchronisation terminée."))

        except Exception as e:
            self.ui_queue.put(("error", str(e)))

    def delete_worker(self, orphans, dst):
        """Supprime fichiers et dossiers un par un avec log."""
        deleted_count = 0
        for type_item, path in orphans:
            try:
                rel_path = os.path.relpath(path, dst)
                if type_item == 'file':
                    os.remove(path)
                else:
                    # On utilise rmdir car topdown=False garantit que le dossier est vide
                    os.rmdir(path)
                
                deleted_count += 1
                self.ui_queue.put(("log", f"[SUPPR] {rel_path}"))
                self.ui_queue.put(("deleted", deleted_count))
            except Exception as e:
                self.ui_queue.put(("log", f"[ERREUR] Impossible de supprimer {path}: {e}"))
        
        self.ui_queue.put(("done", f"Miroir terminé ({deleted_count} éléments supprimés)."))

    # ---------- GESTION UI ----------

    def process_ui_queue(self):
        try:
            while True:
                item = self.ui_queue.get_nowait()
                action = item[0]

                if action == "log": self.log(item[1])
                elif action == "total": self.count_total.set(item[1])
                elif action == "copied": self.count_copied.set(item[1])
                elif action == "skipped": self.count_skipped.set(item[1])
                elif action == "deleted": self.count_deleted.set(item[1])
                elif action == "progress": self.progress_var.set(item[1])
                elif action == "confirm_mirror": self.ask_mirror_confirmation(item[1], item[2])
                elif action == "done":
                    self.log(f"Succès : {item[1]}")
                    self.finish_operation()
                elif action == "error":
                    messagebox.showerror("Erreur", item[1])
                    self.finish_operation()
        except queue.Empty:
            pass
        self.root.after(100, self.process_ui_queue)

    def ask_mirror_confirmation(self, orphans, dst):
        msg = f"{len(orphans)} fichiers/dossiers orphelins détectés.\n\nVoulez-vous les supprimer de la destination ?"
        if messagebox.askyesno("Confirmation Miroir", msg):
            threading.Thread(target=self.delete_worker, args=(orphans, dst), daemon=True).start()
        else:
            self.log("Miroir annulé.")
            self.finish_operation()

    def finish_operation(self):
        self.btn_sync.config(state='normal', text="Lancer la synchronisation", bg="#2ecc71")
        self.is_running = False


if __name__ == "__main__":
    root = tk.Tk()
    app = SyncApp(root)
    root.mainloop()
