import tkinter as tk
from tkinter import ttk, filedialog, messagebox, simpledialog
import json
import os
import zipfile

class ZipperApp:
    def __init__(self, root):
        self.root = root
        self.root.title("Partenaire de Code - Project Zipper")
        self.root.geometry("900x550")

        # Structure : { "Nom": {"root_path": "...", "items": []} }
        self.projects = self.load_data()
        self.current_project = None

        self.setup_ui()

    def setup_ui(self):
        """Crée la structure de l'interface graphique"""
        # --- Panneau Gauche : Liste des Projets ---
        self.sidebar = tk.Frame(self.root, width=200, bg="#f0f0f0")
        self.sidebar.pack(side="left", fill="y")

        tk.Label(self.sidebar, text="Mes Projets", bg="#f0f0f0", font=("Arial", 12, "bold")).pack(pady=10)
        
        self.project_listbox = tk.Listbox(self.sidebar)
        self.project_listbox.pack(fill="both", expand=True, padx=5, pady=5)
        self.project_listbox.bind('<<ListboxSelect>>', self.on_project_select)

        btn_frame = tk.Frame(self.sidebar, bg="#f0f0f0")
        btn_frame.pack(fill="x", pady=5)
        tk.Button(btn_frame, text="+ Nouveau", command=self.add_project).pack(side="left", padx=5)
        tk.Button(btn_frame, text="- Suppr.", command=self.delete_project).pack(side="right", padx=5)

        # --- Panneau Droit : Contenu du Projet ---
        self.main_area = tk.Frame(self.root)
        self.main_area.pack(side="right", fill="both", expand=True, padx=10, pady=10)

        # Affichage du Nom du Projet
        self.lbl_project_name = tk.Label(self.main_area, text="Sélectionnez un projet", font=("Arial", 14, "bold"))
        self.lbl_project_name.pack(anchor="w")

        # --- AJOUT : Affichage du répertoire racine ---
        self.root_path_frame = tk.LabelFrame(self.main_area, text="Répertoire racine du projet", fg="blue", font=("Arial", 9, "bold"))
        self.root_path_frame.pack(fill="x", pady=5)
        
        self.lbl_root_path = tk.Label(self.root_path_frame, text="(Aucun dossier sélectionné)", font=("Arial", 9, "italic"), wraplength=600, justify="left")
        self.lbl_root_path.pack(anchor="w", padx=5, pady=5)

        self.file_listbox = tk.Listbox(self.main_area, selectmode="multiple")
        self.file_listbox.pack(fill="both", expand=True, pady=10)

        # Boutons d'action
        action_frame = tk.Frame(self.main_area)
        action_frame.pack(fill="x")
        
        tk.Button(action_frame, text="Ajouter Fichiers", command=self.add_files).pack(side="left", padx=5)
        tk.Button(action_frame, text="Ajouter Dossier", command=self.add_directory).pack(side="left", padx=5)
        tk.Button(action_frame, text="Retirer élément", command=self.remove_item).pack(side="left", padx=5)
        
        self.btn_zip = tk.Button(self.main_area, text="GÉNÉRER LE ZIP", bg="#4CAF50", fg="white", 
                                 font=("Arial", 10, "bold"), command=self.create_zip, state="disabled")
        self.btn_zip.pack(fill="x", pady=10)

        self.refresh_project_list()

    def load_data(self):
        if os.path.exists("projects.json"):
            try:
                with open("projects.json", "r", encoding="utf-8") as f:
                    data = json.load(f)
                    # Migration auto pour l'ancien format vers le nouveau (dict)
                    for k, v in data.items():
                        if isinstance(v, list):
                            data[k] = {"root_path": "Non défini", "items": v}
                    return data
            except: return {}
        return {}

    def save_data(self):
        with open("projects.json", "w", encoding="utf-8") as f:
            json.dump(self.projects, f, indent=4)

    def add_project(self):
        name = simpledialog.askstring("Nouveau Projet", "Nom du projet :")
        if not name: return
        
        if name in self.projects:
            messagebox.showwarning("Erreur", "Ce projet existe déjà.")
            return

        # On force la sélection du dossier racine à la création
        messagebox.showinfo("Racine du projet", f"Veuillez choisir le répertoire principal pour le projet '{name}'.\nL'explorateur s'ouvrira ici par défaut.")
        root_dir = filedialog.askdirectory(title="Choisir le dossier racine")
        
        if root_dir:
            self.projects[name] = {
                "root_path": root_dir,
                "items": []
            }
            self.save_data()
            self.refresh_project_list()
        else:
            messagebox.showwarning("Annulé", "Le projet n'a pas été créé car aucun dossier racine n'a été choisi.")

    def on_project_select(self, event):
        selection = self.project_listbox.curselection()
        if selection:
            self.current_project = self.project_listbox.get(selection[0])
            proj_data = self.projects[self.current_project]
            
            # Mise à jour des affichages
            self.lbl_project_name.config(text=f"Projet : {self.current_project}")
            self.lbl_root_path.config(text=proj_data['root_path'], fg="black")
            self.btn_zip.config(state="normal")
            self.refresh_file_list()

    def add_files(self):
        if not self.current_project: return
        root_dir = self.projects[self.current_project]["root_path"]
        
        paths = filedialog.askopenfilenames(initialdir=root_dir)
        if paths:
            self.projects[self.current_project]["items"].extend(paths)
            self.save_data()
            self.refresh_file_list()

    def add_directory(self):
        if not self.current_project: return
        root_dir = self.projects[self.current_project]["root_path"]
        
        path = filedialog.askdirectory(initialdir=root_dir)
        if path:
            self.projects[self.current_project]["items"].append(path)
            self.save_data()
            self.refresh_file_list()

    # ... (les autres méthodes restent identiques au script précédent)
    
    def delete_project(self):
        selection = self.project_listbox.curselection()
        if selection:
            name = self.project_listbox.get(selection[0])
            if messagebox.askyesno("Confirmation", f"Supprimer le projet '{name}' ?"):
                del self.projects[name]
                self.current_project = None
                self.lbl_root_path.config(text="(Aucun dossier sélectionné)")
                self.save_data()
                self.refresh_project_list()
                self.refresh_file_list()

    def refresh_project_list(self):
        self.project_listbox.delete(0, tk.END)
        for name in self.projects:
            self.project_listbox.insert(tk.END, name)

    def remove_item(self):
        selection = self.file_listbox.curselection()
        if selection and self.current_project:
            items = self.projects[self.current_project]["items"]
            for index in reversed(selection):
                items.pop(index)
            self.save_data()
            self.refresh_file_list()

    def refresh_file_list(self):
        self.file_listbox.delete(0, tk.END)
        if self.current_project:
            for path in self.projects[self.current_project]["items"]:
                self.file_listbox.insert(tk.END, path)

    def create_zip(self):
        if not self.current_project: return
        proj_data = self.projects[self.current_project]
        if not proj_data["items"]:
            messagebox.showwarning("Vide", "La liste est vide !")
            return
        save_path = filedialog.asksaveasfilename(defaultextension=".zip", initialdir=proj_data["root_path"], initialfile=f"{self.current_project}.zip")
        if not save_path: return
        try:
            with zipfile.ZipFile(save_path, 'w', zipfile.ZIP_DEFLATED) as zipf:
                for path in proj_data["items"]:
                    if os.path.exists(path):
                        if os.path.isfile(path):
                            zipf.write(path, os.path.basename(path))
                        elif os.path.isdir(path):
                            for root, dirs, files in os.walk(path):
                                for file in files:
                                    full_path = os.path.join(root, file)
                                    arcname = os.path.relpath(full_path, os.path.dirname(path))
                                    zipf.write(full_path, arcname)
            messagebox.showinfo("Succès", f"Archive créée :\n{save_path}")
        except Exception as e:
            messagebox.showerror("Erreur", f"Échec : {e}")

if __name__ == "__main__":
    root = tk.Tk()
    app = ZipperApp(root)
    root.mainloop()
