#!/usr/bin/env python3
import tkinter as tk
from tkinter import ttk, messagebox, simpledialog
import sounddevice as sd
import numpy as np
import json
import os
import traceback

class AudioPassthroughGUI:
    def __init__(self, root):
        self.root = root
        self.root.title("🎸 Pedalier Pro - Config & Presets Séparés")
        
        # --- Définition des fichiers ---
        base_path = os.path.dirname(__file__)
        self.config_file = os.path.join(base_path, 'pedalier_app_config.json')
        self.presets_file = os.path.join(base_path, 'pedalier_library.json')
        
        self.is_running = False
        self.stream = None
        self.blocksize = 128
        self.chorus_phase = 0
        
        # État interne par défaut
        self.effects = {
            "dist": {"active": False, "gain": 3.0, "mix": 0.5},
            "chorus": {"active": False, "speed": 1.5, "depth": 0.3},
            "reverb": {"active": False, "size": 0.5, "mix": 0.3},
            "delay": {"active": False, "time": 0.3, "feedback": 0.4}
        }
        
        self.presets = {}
        self.slider_widgets = {}
        self.input_device_name = None
        self.output_device_name = None

        try:
            self.setup_dark_theme()
            self.create_menu_bar()
            
            # --- Chargement séquentiel ---
            self.load_app_config()  # 1. Charger la fenêtre et l'audio
            self.load_presets()     # 2. Charger la bibliothèque de sons
            
            self.create_widgets()    
            self.load_devices()
            self.restore_device_selection()
        except Exception as e:
            self.log_error("Initialisation", e)
        
        self.root.protocol("WM_DELETE_WINDOW", self.on_closing)

    # --- Gestion de la Configuration de l'App (Fenêtre + Audio) ---
    def load_app_config(self):
        if os.path.exists(self.config_file):
            try:
                with open(self.config_file, 'r') as f:
                    data = json.load(f)
                    if 'geometry' in data: self.root.geometry(data['geometry'])
                    self.saved_in = data.get('input_device')
                    self.saved_out = data.get('output_device')
            except Exception as e: print(f"Erreur config app: {e}")

    def save_app_config(self):
        try:
            config = {
                'geometry': self.root.geometry(),
                'input_device': getattr(self, 'input_device_name', None),
                'output_device': getattr(self, 'output_device_name', None)
            }
            with open(self.config_file, 'w') as f:
                json.dump(config, f, indent=4)
        except Exception as e: print(f"Erreur sauvegarde app: {e}")

    # --- Gestion de la Bibliothèque de Presets (Sons) ---
    def load_presets(self):
        if os.path.exists(self.presets_file):
            try:
                with open(self.presets_file, 'r') as f:
                    data = json.load(f)
                    self.presets = data.get('library', {})
                    # On restaure aussi le dernier réglage utilisé pour les effets
                    last = data.get('last_used', {})
                    for k, v in last.items():
                        if k in self.effects: self.effects[k].update(v)
            except Exception as e: print(f"Erreur bibliothèque: {e}")

    def save_presets(self):
        try:
            # Nettoyage des widgets pour le JSON
            clean_last = {}
            for k, v in self.effects.items():
                clean_last[k] = {pk: pv for pk, pv in v.items() if pk != "button_widget"}

            data = {
                'library': self.presets,
                'last_used': clean_last
            }
            with open(self.presets_file, 'w') as f:
                json.dump(data, f, indent=4)
        except Exception as e: print(f"Erreur sauvegarde bibliothèque: {e}")

    # --- Interface et Logic ---
    def create_widgets(self):
        tk.Label(self.root, text="GUITAR RACK PRO", font=("Courier", 20, "bold"),
                 bg=self.bg_color, fg=self.accent_color).pack(pady=10)

        # Section Presets
        p_frame = tk.LabelFrame(self.root, text="📚 Bibliothèque de Sons", bg=self.bg_color, fg=self.accent_color)
        p_frame.pack(fill=tk.X, padx=15, pady=5)
        
        self.preset_combo = ttk.Combobox(p_frame, values=list(self.presets.keys()), state="readonly")
        self.preset_combo.pack(side=tk.LEFT, padx=10, pady=10, expand=True, fill=tk.X)
        self.preset_combo.bind("<<ComboboxSelected>>", self.on_preset_selected)
        
        tk.Button(p_frame, text="💾 Sauver", bg="#03DAC6", command=self.save_new_preset).pack(side=tk.LEFT, padx=5)
        tk.Button(p_frame, text="🗑️ Suppr.", bg="#CF6679", command=self.delete_preset).pack(side=tk.LEFT, padx=5)

        # Boutons de contrôle
        top_frame = tk.Frame(self.root, bg=self.bg_color)
        top_frame.pack(pady=10)
        self.start_btn = tk.Button(top_frame, text="▶ START", bg="#03DAC6", command=self.start_passthrough, width=12)
        self.start_btn.pack(side=tk.LEFT, padx=10)
        self.stop_btn = tk.Button(top_frame, text="⏹ STOP", bg="#CF6679", command=self.stop_passthrough, state=tk.DISABLED, width=12)
        self.stop_btn.pack(side=tk.LEFT, padx=10)

        # Bloc des effets
        eff_container = tk.Frame(self.root, bg=self.bg_color)
        eff_container.pack(fill=tk.BOTH, expand=True, padx=15)

        configs = [
            ("DISTORTION", "dist", [("Gain", "gain", 1, 20), ("Mix", "mix", 0, 1)]),
            ("CHORUS", "chorus", [("Vitesse", "speed", 0.1, 5.0), ("Profondeur", "depth", 0, 1)]),
            ("REVERB", "reverb", [("Taille", "size", 0.1, 0.9), ("Mix", "mix", 0, 1)]),
            ("DELAY", "delay", [("Temps", "time", 0.1, 1.0), ("Feedback", "feedback", 0, 0.9)])
        ]
        for title, key, params in configs:
            self.create_effect_block(eff_container, title, key, params)

    def create_effect_block(self, parent, title, key, params):
        frame = tk.LabelFrame(parent, text=title, bg=self.bg_color, fg=self.fg_color, font=("Arial", 10, "bold"))
        frame.pack(fill=tk.X, pady=5, padx=5)

        active = self.effects[key]["active"]
        btn = tk.Button(frame, text="ON" if active else "OFF", 
                        bg=self.accent_color if active else self.button_bg,
                        command=lambda k=key: self.toggle_effect(k), width=8)
        btn.pack(side=tk.LEFT, padx=10, pady=10)
        self.effects[key]["button_widget"] = btn

        s_frame = tk.Frame(frame, bg=self.bg_color)
        s_frame.pack(side=tk.LEFT, fill=tk.X, expand=True)

        for p_name, p_key, p_min, p_max in params:
            f = tk.Frame(s_frame, bg=self.bg_color)
            f.pack(fill=tk.X)
            tk.Label(f, text=p_name, bg=self.bg_color, fg=self.fg_color, width=8).pack(side=tk.LEFT)
            s = tk.Scale(f, from_=p_min, to=p_max, orient=tk.HORIZONTAL, resolution=0.01,
                         bg=self.bg_color, fg=self.accent_color, highlightthickness=0,
                         command=lambda v, k=key, pk=p_key: self.update_param(k, pk, v))
            s.set(self.effects[key][p_key])
            s.pack(side=tk.LEFT, fill=tk.X, expand=True)
            self.slider_widgets[f"{key}_{p_key}"] = s

    def on_preset_selected(self, event):
        name = self.preset_combo.get()
        if name in self.presets:
            settings = self.presets[name]
            for eff_key, params in settings.items():
                if eff_key in self.effects:
                    for p_key, p_val in params.items():
                        self.effects[eff_key][p_key] = p_val
                        if f"{eff_key}_{p_key}" in self.slider_widgets:
                            self.slider_widgets[f"{eff_key}_{p_key}"].set(p_val)
                    active = self.effects[eff_key]["active"]
                    self.effects[eff_key]["button_widget"].config(
                        text="ON" if active else "OFF",
                        bg=self.accent_color if active else self.button_bg,
                        fg="black" if active else "white"
                    )

    def save_new_preset(self):
        name = simpledialog.askstring("Nouveau Preset", "Nom du son :")
        if name:
            current = {}
            for k, v in self.effects.items():
                current[k] = {pk: pv for pk, pv in v.items() if pk != "button_widget"}
            self.presets[name] = current
            self.preset_combo['values'] = list(self.presets.keys())
            self.preset_combo.set(name)
            self.save_presets()

    def delete_preset(self):
        name = self.preset_combo.get()
        if name and name in self.presets:
            if messagebox.askyesno("Confirmation", f"Supprimer '{name}' ?"):
                del self.presets[name]
                self.preset_combo['values'] = list(self.presets.keys())
                self.preset_combo.set("")
                self.save_presets()

    # --- Standard Boilerplate (Audio/Theme/Error) ---
    def setup_dark_theme(self):
        self.bg_color, self.fg_color, self.accent_color, self.button_bg = "#121212", "#E0E0E0", "#BB86FC", "#1F1B24"
        self.root.configure(bg=self.bg_color)
        style = ttk.Style()
        style.theme_use('clam')
        style.configure("TCombobox", fieldbackground=self.button_bg, background=self.accent_color, foreground="white")

    def create_menu_bar(self):
        menubar = tk.Menu(self.root)
        self.root.config(menu=menubar)
        dev_menu = tk.Menu(menubar, tearoff=0)
        menubar.add_cascade(label="Périphériques", menu=dev_menu)
        self.input_menu, self.output_menu = tk.Menu(dev_menu, tearoff=0), tk.Menu(dev_menu, tearoff=0)
        dev_menu.add_cascade(label="Entrée", menu=self.input_menu)
        dev_menu.add_cascade(label="Sortie", menu=self.output_menu)

    def log_error(self, ctx, e):
        print(f"\n! ERROR: {ctx} !\n{traceback.format_exc()}")
        messagebox.showerror("Erreur", f"{ctx}: {e}")

    def toggle_effect(self, key):
        self.effects[key]["active"] = not self.effects[key]["active"]
        a = self.effects[key]["active"]
        self.effects[key]["button_widget"].config(text="ON" if a else "OFF", bg=self.accent_color if a else self.button_bg, fg="black" if a else "white")

    def update_param(self, ek, pk, v): self.effects[ek][pk] = float(v)

    def load_devices(self):
        try:
            devs = sd.query_devices()
            for i, d in enumerate(devs):
                if d['max_input_channels'] > 0: self.input_menu.add_command(label=d['name'], command=lambda idx=i, n=d['name']: self.select_in(idx, n))
                if d['max_output_channels'] > 0: self.output_menu.add_command(label=d['name'], command=lambda idx=i, n=d['name']: self.select_out(idx, n))
        except: pass

    def select_in(self, i, n): self.selected_input_device, self.input_device_name = i, n
    def select_out(self, i, n): self.selected_output_device, self.output_device_name = i, n

    def restore_device_selection(self):
        devs = sd.query_devices()
        for i, d in enumerate(devs):
            if hasattr(self, 'saved_in') and d['name'] == self.saved_in: self.select_in(i, d['name'])
            if hasattr(self, 'saved_out') and d['name'] == self.saved_out: self.select_out(i, d['name'])

    def start_passthrough(self):
        if self.selected_input_device is None: return
        try:
            fs = int(sd.query_devices(self.selected_input_device)['default_samplerate'])
            self.delay_buffer = np.zeros((fs * 2, 1))
            def cb(indata, outdata, frames, time, status):
                audio = indata.copy()
                if self.effects["dist"]["active"]: audio = np.tanh(audio * self.effects["dist"]["gain"]) * self.effects["dist"]["mix"]
                if self.effects["chorus"]["active"]:
                    self.chorus_phase += self.effects["chorus"]["speed"] / fs
                    offset = int(self.effects["chorus"]["depth"] * 100 * np.sin(2 * np.pi * self.chorus_phase))
                    audio = (audio + np.roll(audio, offset)) / 2
                if self.effects["delay"]["active"]:
                    audio += self.delay_buffer[:frames] * self.effects["delay"]["feedback"]
                    self.delay_buffer = np.roll(self.delay_buffer, -frames, axis=0)
                    self.delay_buffer[-frames:] = audio
                outdata[:] = np.clip(audio, -1, 1)
            self.stream = sd.Stream(device=(self.selected_input_device, self.selected_output_device), samplerate=fs, channels=1, callback=cb, blocksize=self.blocksize)
            self.stream.start()
            self.is_running = True
            self.start_btn.config(state=tk.DISABLED); self.stop_btn.config(state=tk.NORMAL)
        except Exception as e: self.log_error("Audio", e)

    def stop_passthrough(self):
        if self.stream: self.stream.stop(); self.stream.close()
        self.is_running = False
        self.start_btn.config(state=tk.NORMAL); self.stop_btn.config(state=tk.DISABLED)

    def on_closing(self):
        self.stop_passthrough()
        self.save_app_config()
        self.save_presets()
        self.root.destroy()

if __name__ == "__main__":
    root = tk.Tk()
    root.geometry("550x900")
    app = AudioPassthroughGUI(root)
    root.mainloop()
