import customtkinter as ctk
import ollama
import subprocess
import sys
import threading
import json
import os
import re
import time
from tkinter import messagebox, simpledialog, PanedWindow, ttk

# Imports pour la coloration syntaxique
from pygments import lexers
from pygments.lexers import PythonLexer

CONFIG_FILE = "config.json"

class LineNumbers(ctk.CTkCanvas):
    def __init__(self, master, text_widget, **kwargs):
        super().__init__(master, **kwargs)
        self.text_widget = text_widget
        self.configure(bg='#2b2b2b', highlightthickness=0, width=40)

    def redraw(self):
        self.delete("all")
        i = self.text_widget.index("@0,0")
        while True:
            dline = self.text_widget.dlineinfo(i)
            if dline is None: break
            y = dline[1]
            linenum = str(i).split(".")[0]
            self.create_text(35, y, anchor="ne", text=linenum, fill="#75715e", font=("Consolas", 12))
            i = self.text_widget.index("%s+1line" % i)

class AIApp(ctk.CTk):
    def __init__(self):
        super().__init__()
        self.title("PyGenius - IDE IA ajustable")
        
        self.base_projects_dir = os.path.join(os.path.dirname(__file__), "projets")
        if not os.path.exists(self.base_projects_dir):
            os.makedirs(self.base_projects_dir)
            
        self.current_project_path = ""
        self.open_files = {} 
        self.sessions = {}  # Mémorise les fichiers ouverts par projet
        self.chat_histories = {} # Mémorise les chats par projet
        self.thinking_active = False
        self.ensure_ollama_running()
        self.installed_models = self.get_ollama_models()

        # --- PALETTE DE COULEURS SYNTAXIQUE ---
        self.syntax_colors = {
            "Token.Keyword": "#f92672",
            "Token.Name.Function": "#a6e22e",
            "Token.Name.Class": "#a6e22e",
            "Token.String": "#e6db74",
            "Token.Comment": "#75715e",
            "Token.Literal.Number": "#ae81ff",
            "Token.Operator": "#f92672",
            "Token.Name.Builtin": "#66d9ef",
        }

        self.load_config()

        # Configuration de la grille
        self.grid_columnconfigure(0, weight=0)
        self.grid_columnconfigure(1, weight=1)
        self.grid_rowconfigure(0, weight=1)

        # --- SIDEBAR ---
        self.sidebar = ctk.CTkFrame(self, width=250)
        self.sidebar.grid(row=0, column=0, rowspan=2, sticky="nsew")
        self.sidebar.grid_propagate(False) # Garder une largeur fixe
        
        ctk.CTkLabel(self.sidebar, text="PYGENIUS", font=ctk.CTkFont(size=20, weight="bold")).pack(pady=10)
        
        # Sélection du modèle
        ctk.CTkLabel(self.sidebar, text="Modèle IA :", font=ctk.CTkFont(weight="bold")).pack(pady=(10, 0))
        self.model_menu = ctk.CTkOptionMenu(self.sidebar, values=self.installed_models)
        self.model_menu.pack(pady=5, padx=10)
        if hasattr(self, 'saved_model') and self.saved_model in self.installed_models:
            self.model_menu.set(self.saved_model)

        # Sélecteur de Projet
        ctk.CTkLabel(self.sidebar, text="Projet :").pack(pady=(15,0))
        self.proj_menu = ctk.CTkOptionMenu(self.sidebar, values=self.get_list_projects(), command=self.change_project)
        self.proj_menu.pack(pady=5, padx=10)
        ctk.CTkButton(self.sidebar, text="+ Nouveau Projet", command=self.create_new_project, fg_color="#34495e").pack(pady=2, padx=10)

        # Explorateur de fichiers (Treeview)
        ctk.CTkLabel(self.sidebar, text="EXPLORATEUR", font=ctk.CTkFont(size=11, weight="bold")).pack(pady=(10, 0))
        self.tree_frame = ctk.CTkFrame(self.sidebar)
        self.tree_frame.pack(pady=(5, 10), padx=10, fill="both", expand=True)
        
        # Style pour le Treeview
        self.setup_tree_style()
        
        self.tree = ttk.Treeview(self.tree_frame, show="tree", selectmode="browse")
        self.tree.pack(side="left", fill="both", expand=True)
        
        self.tree_scroll = ctk.CTkScrollbar(self.tree_frame, orientation="vertical", command=self.tree.yview)
        self.tree_scroll.pack(side="right", fill="y")
        self.tree.configure(yscrollcommand=self.tree_scroll.set)
        
        self.tree.bind("<Double-1>", lambda e: self.open_selected_file())
        
        # Boutons d'action
        ctk.CTkButton(self.sidebar, text="+ Nouveau Fichier", command=self.create_new_file, fg_color="#34495e").pack(pady=2, padx=10)
        ctk.CTkButton(self.sidebar, text="Sauvegarder Tout", fg_color="#27ae60", command=self.save_all_open_files).pack(pady=2, padx=10)

        # Sélection Script Principal
        ctk.CTkLabel(self.sidebar, text="Script Principal :", font=ctk.CTkFont(weight="bold")).pack(pady=(15, 0))
        self.script_menu = ctk.CTkOptionMenu(self.sidebar, values=["Aucun"])
        self.script_menu.pack(pady=5, padx=10)
        
        self.run_button = ctk.CTkButton(self.sidebar, text="🚀 LANCER", command=self.run_main_script, fg_color="#e67e22", hover_color="#d35400")
        self.run_button.pack(pady=10, padx=10)

        # Init display
        self.refresh_file_list_display()

        # Thème
        self.appearance_mode_optionemenu = ctk.CTkOptionMenu(self.sidebar, values=["Dark", "Light"], command=self.change_appearance_mode)
        self.appearance_mode_optionemenu.pack(side="bottom", padx=20, pady=20)
        self.appearance_mode_label = ctk.CTkLabel(self.sidebar, text="Thème :", anchor="w")
        self.appearance_mode_label.pack(side="bottom", padx=20, pady=0)

        # --- ZONE CENTRALE AVEC TERMINAL ---
        self.main_vertical_paned = PanedWindow(self, orient="vertical", bd=0, sashwidth=4, bg="#2b2b2b")
        self.main_vertical_paned.grid(row=0, column=1, sticky="nsew", padx=5, pady=5)

        self.top_paned_window = PanedWindow(self.main_vertical_paned, orient="horizontal", bd=0, sashwidth=4, bg="#2b2b2b")
        
        self.chat_frame = ctk.CTkFrame(self.top_paned_window)
        self.chat_display = ctk.CTkTextbox(self.chat_frame, state="disabled", border_width=2)
        self.chat_display.pack(fill="both", expand=True, padx=5, pady=5)
        
        self.editor_frame = ctk.CTkFrame(self.top_paned_window)
        self.editor_tabs = ctk.CTkTabview(self.editor_frame, command=self.save_all_open_files)
        self.editor_tabs.pack(fill="both", expand=True)

        self.top_paned_window.add(self.chat_frame, stretch="always")
        self.top_paned_window.add(self.editor_frame, stretch="always")
        
        # Terminal
        self.terminal_frame = ctk.CTkFrame(self.main_vertical_paned)
        ctk.CTkLabel(self.terminal_frame, text="TERMINAL SORTIE", font=ctk.CTkFont(size=10, weight="bold")).pack(anchor="w", padx=10)
        self.terminal_output = ctk.CTkTextbox(self.terminal_frame, height=150, font=("Consolas", 12), fg_color="#000000", text_color="#00ff00")
        self.terminal_output.pack(fill="both", expand=True, padx=5, pady=5)
        self.terminal_output.configure(state="disabled")

        # Entrée commande terminal
        self.terminal_input_frame = ctk.CTkFrame(self.terminal_frame)
        self.terminal_input_frame.pack(fill="x", side="bottom", padx=5, pady=(0, 5))
        
        ctk.CTkLabel(self.terminal_input_frame, text="> ", font=ctk.CTkFont(size=12, weight="bold")).pack(side="left", padx=(5, 0))
        self.terminal_entry = ctk.CTkEntry(self.terminal_input_frame, placeholder_text="Taper une commande (ex: pip install ..., dir, etc.)")
        self.terminal_entry.pack(fill="x", side="left", expand=True, padx=5)
        self.terminal_entry.bind("<Return>", lambda e: self.send_terminal_command())

        self.main_vertical_paned.add(self.top_paned_window, stretch="always")
        self.main_vertical_paned.add(self.terminal_frame, stretch="always")

        # --- ENTRÉE CHAT (MULTI-LIGNE) ---
        self.input_frame = ctk.CTkFrame(self, fg_color="transparent")
        self.input_frame.grid(row=1, column=1, padx=20, pady=(0, 10), sticky="ew")
        
        self.input_entry = ctk.CTkTextbox(self.input_frame, height=80, border_width=2)
        self.input_entry.pack(side="left", fill="x", expand=True, padx=(0, 10))
        
        # On peut attacher des placeholders via un tag ou simplement laisser vide
        # Pour faire "Entrée pour envoyer" :
        self.input_entry.bind("<Return>", self._handle_return)
        
        self.send_button = ctk.CTkButton(self.input_frame, text="ENVOYER", width=100, command=self.send_message)
        self.send_button.pack(side="right")

        self.protocol("WM_DELETE_WINDOW", self.on_closing)

    # --- LOGIQUE DE SESSION ---

    def save_current_session(self):
        """Enregistre quels fichiers sont ouverts pour le projet actuel"""
        if self.current_project_path:
            project_name = os.path.basename(self.current_project_path)
            self.sessions[project_name] = list(self.open_files.keys())

    def restore_project_session(self, project_name):
        """Réouvre les fichiers enregistrés pour ce projet"""
        if project_name in self.sessions:
            for filename in self.sessions[project_name]:
                self.open_file_in_tab(filename)

    # --- LOGIQUE IA ---

    def ensure_ollama_running(self):
        """Vérifie si Ollama est lancé, sinon tente de le démarrer."""
        try:
            ollama.list()
            print("[SYSTEM] Ollama est déjà en cours d'exécution.")
        except Exception:
            print("[SYSTEM] Ollama n'est pas détecté. Tentative de démarrage...")
            try:
                # Lance Ollama serve en arrière-plan sans bloquer
                if sys.platform == "win32":
                    subprocess.Popen(["ollama", "serve"], creationflags=subprocess.CREATE_NO_WINDOW)
                else:
                    subprocess.Popen(["ollama", "serve"], stdout=subprocess.DEVNULL, stderr=subprocess.DEVNULL)
                
                # Attendre quelques secondes que le service démarre
                retries = 5
                while retries > 0:
                    time.sleep(2)
                    try:
                        ollama.list()
                        print("[SYSTEM] Ollama a été démarré avec succès.")
                        return
                    except Exception:
                        retries -= 1
                        print(f"[SYSTEM] Attente du démarrage d'Ollama... ({retries} essais restants)")
                
                print("[SYSTEM] Échec du démarrage automatique d'Ollama.")
            except Exception as e:
                print(f"[SYSTEM] Erreur lors de la tentative de lancement d'Ollama : {e}")

    def get_ollama_models(self):
        try:
            models_info = ollama.list()
            # La structure retournée peut varier selon la version de la lib ollama-python
            # On tente de récupérer les noms de manière robuste
            if isinstance(models_info, dict) and 'models' in models_info:
                names = [m['name'] for m in models_info['models']]
            elif hasattr(models_info, 'models'): # Pour les versions plus récentes qui retournent un objet
                names = [m.model for m in models_info.models]
            else:
                names = []
            
            if not names:
                print("[SYSTEM] Aucun modèle trouvé dans Ollama.")
                return ["qwen2.5-coder:7b"]
            return names
        except Exception as e:
            print(f"[SYSTEM] Erreur lors de la récupération des modèles : {e}")
            return ["qwen2.5-coder:7b"]

    def get_project_context(self):
        """Récupère le contenu de tous les fichiers ouverts pour donner du contexte à l'IA."""
        context = ""
        if not self.open_files:
            return "Aucun fichier ouvert pour le moment."
        
        for filename, editor in self.open_files.items():
            content = editor.get("1.0", "end-1c")
            context += f"### FICHIER : {filename}\n```python\n{content}\n```\n\n"
        return context

    def start_thinking_animation(self):
        """Démarre l'animation d'attente IA."""
        self.thinking_active = True
        self.animate_thinking(0)

    def animate_thinking(self, count):
        """Réalise l'animation avec un petit spinner."""
        if not self.thinking_active:
            return
        
        frames = ["⠋", "⠙", "⠹", "⠸", "⠼", "⠴", "⠦", "⠧", "⠇", "⠏"]
        frame = frames[count % len(frames)]
        
        self.chat_display.configure(state="normal")
        # On cherche si notre balise existe déjà pour la remplacer
        ranges = self.chat_display._textbox.tag_ranges("thinking")
        if ranges:
            self.chat_display._textbox.delete(ranges[0], ranges[1])
        
        # On insère le frame avec le tag "thinking"
        # On s'assure d'insérer à la toute fin
        self.chat_display._textbox.insert("end", frame, "thinking")
        self.chat_display.configure(state="disabled")
        self.chat_display.see("end")
        
        self.after(100, lambda: self.animate_thinking(count + 1))

    def stop_thinking_animation(self):
        """Arrête l'animation et nettoie le tag."""
        self.thinking_active = False
        self.chat_display.configure(state="normal")
        ranges = self.chat_display._textbox.tag_ranges("thinking")
        if ranges:
            self.chat_display._textbox.delete(ranges[0], ranges[1])
        self.chat_display.configure(state="disabled")

    def format_duration(self, seconds):
        mins = int(seconds // 60)
        secs = int(seconds % 60)
        return f"{mins:02d}m {secs:02d}s"

    def run_ai(self, prompt):
        selected_model = self.model_menu.get()
        context = self.get_project_context()
        
        system_prompt = (
            "Tu es un assistant de programmation expert au sein de l'IDE PyGenius.\n"
            "Voici le contexte actuel du projet (fichiers ouverts) :\n\n"
            f"{context}\n\n"
            "IMPORTANT : Si tu proposes des modifications de code, utilise TOUJOURS le format suivant :\n"
            "### FICHIER : nom_du_fichier.py\n"
            "```python\n"
            "le code complet ici\n"
            "```\n\n"
            "L'IDE détectera ce format pour mettre à jour automatiquement les fichiers."
        )
        
        full_response = ""
        start_time = time.time()
        first_token_time = None
        
        try:
            messages = [
                {'role': 'system', 'content': system_prompt},
                {'role': 'user', 'content': prompt}
            ]
            
            stream = ollama.chat(model=selected_model, messages=messages, stream=True)
            for chunk in stream:
                if first_token_time is None:
                    first_token_time = time.time()
                    self.after(0, self.stop_thinking_animation)
                
                content = chunk['message']['content']
                full_response += content
                self.after(0, self.append_chat, content)
            
            end_time = time.time()
            
            # Calcul et affichage des durées
            if first_token_time:
                reflection_dur = first_token_time - start_time
                generation_dur = end_time - first_token_time
                total_dur = end_time - start_time
                
                stats = (f"\n\n[DURÉE] Réflexion: {self.format_duration(reflection_dur)} | "
                         f"Génération: {self.format_duration(generation_dur)} | "
                         f"Total: {self.format_duration(total_dur)}\n")
                self.after(0, self.append_chat, stats)

            # Après la fin du stream, on cherche des blocs à mettre à jour
            self.after(0, lambda: self.process_ai_response_for_updates(full_response))
            
        except Exception as e:
            self.after(0, self.stop_thinking_animation)
            self.after(0, lambda: messagebox.showerror("Erreur IA", str(e), parent=self))

    def process_ai_response_for_updates(self, text):
        """Parcourt la réponse de l'IA pour trouver des blocs '### FICHIER : ...' et met à jour l'éditeur."""
        pattern = r"### FICHIER\s*:\s*([\w\.-]+)\s*[\r\n]+```[\w]*[\r\n]+([\s\S]*?)```"
        matches = re.finditer(pattern, text)
        
        updated_any = False
        for match in matches:
            filename = match.group(1).strip()
            new_code = match.group(2)
            
            if filename in self.open_files:
                editor = self.open_files[filename]
                
                # --- SAUVEGARDE DE SÉCURITÉ (BACKUP) ---
                self.create_backup(filename)
                
                # Mise à jour de l'éditeur
                editor.delete("1.0", "end")
                editor.insert("1.0", new_code)
                self.apply_pygments(editor)
                self.save_all_open_files()
                self.append_chat(f"\n\n[SYSTEM]: Le fichier '{filename}' a été mis à jour automatiquement (Backup créé).\n")
                updated_any = True
            else:
                self.append_chat(f"\n\n[SYSTEM]: Impossible de mettre à jour '{filename}' (le fichier doit être ouvert dans un onglet).\n")
        
        if updated_any:
            self.refresh_file_list_display()

    def create_backup(self, filename):
        """Crée une copie de sauvegarde dans un dossier 'backups' au sein du projet."""
        if not self.current_project_path:
            return
            
        backup_dir = os.path.join(self.current_project_path, "backups")
        if not os.path.exists(backup_dir):
            os.makedirs(backup_dir)
            
        source_path = os.path.join(self.current_project_path, filename)
        if os.path.exists(source_path):
            timestamp = time.strftime("%Y%m%d_%H%M%S")
            backup_filename = f"{timestamp}_{filename}"
            backup_path = os.path.join(backup_dir, backup_filename)
            
            try:
                with open(source_path, "r", encoding="utf-8") as src, \
                     open(backup_path, "w", encoding="utf-8") as dst:
                    dst.write(src.read())
                # On ne pollue pas trop le terminal mais on peut mettre un petit log
            except Exception as e:
                print(f"Erreur backup : {e}")

    # --- COLORATION ---

    def setup_syntax_tags(self, editor):
        for token_name, color in self.syntax_colors.items():
            editor.tag_config(token_name, foreground=color)

    def apply_pygments(self, editor):
        code = editor.get("1.0", "end-1c")
        for tag in self.syntax_colors.keys():
            editor.tag_remove(tag, "1.0", "end")
        lexer = PythonLexer()
        editor.mark_set("range_start", "1.0")
        for token, value in lexer.get_tokens(code):
            token_type = str(token)
            end_index = editor.index(f"range_start + {len(value)} chars")
            if token_type in self.syntax_colors:
                editor.tag_add(token_type, "range_start", end_index)
            editor.mark_set("range_start", end_index)

    # --- GESTION DES FICHIERS ---

    def create_new_file(self):
        if not self.current_project_path:
            messagebox.showwarning("Attention", "Veuillez d'abord sélectionner un projet.", parent=self)
            return
        fn = simpledialog.askstring("Nouveau Fichier", "Nom du fichier (ex: script.py) :", parent=self)
        if fn:
            path = os.path.join(self.current_project_path, fn)
            try:
                with open(path, "w", encoding="utf-8") as f: f.write("")
                self.refresh_file_list_display()
                self.open_file_in_tab(fn)
            except Exception as e:
                messagebox.showerror("Erreur", f"Impossible de créer le fichier : {e}", parent=self)

    def close_tab(self, filename):
        """Ferme un onglet spécifique."""
        if not filename:
            return
            
        # Sauvegarde avant fermeture
        self.save_all_open_files()
        
        # Fermeture de l'onglet et suppression de l'éditeur du tracking
        if filename in list(self.editor_tabs._tab_dict.keys()):
            self.editor_tabs.delete(filename)
            
        if filename in self.open_files:
            del self.open_files[filename]
            
        # Mise à jour de la session
        self.save_current_session()

    def setup_tree_style(self):
        style = ttk.Style()
        theme = ctk.get_appearance_mode()
        
        # Couleurs adaptées
        if theme == "Dark":
            bg_color = "#2b2b2b"
            fg_color = "#ffffff"
            select_color = "#1f538d"
        else:
            bg_color = "#ffffff"
            fg_color = "#000000"
            select_color = "#9fb6cd"
            
        style.theme_use("clam") # "clam" est souvent plus flexible pour le style
        style.configure("Treeview", 
                        background=bg_color, 
                        foreground=fg_color, 
                        fieldbackground=bg_color,
                        rowheight=25,
                        borderwidth=0,
                        font=("Consolas", 10))
        style.map("Treeview", background=[('selected', select_color)])
        
        # Supprime les bordures autour du treeview
        style.configure("Treeview", indent=20)
        style.layout("Treeview", [('Treeview.treearea', {'sticky': 'nswe'})])

    def open_file_in_tab(self, filename, relative_path=None):
        if relative_path:
            file_path = os.path.join(self.current_project_path, relative_path)
            tab_name = relative_path
        else:
            file_path = os.path.join(self.current_project_path, filename)
            tab_name = filename
            
        if not os.path.exists(file_path) or tab_name in self.open_files: 
            if tab_name in self.open_files:
                self.editor_tabs.set(tab_name)
            return
        
        self.editor_tabs.add(tab_name)
        tab = self.editor_tabs.tab(tab_name)
        
        # Frame pour aligner numéros et texte
        editor_frame = ctk.CTkFrame(tab, fg_color="#2b2b2b")
        editor_frame.pack(fill="both", expand=True)
        
        editor = ctk.CTkTextbox(editor_frame, font=("Consolas", 14), undo=True, wrap="none", border_width=0)
        line_nums = LineNumbers(editor_frame, editor._textbox) # Accès au widget interne tkinter
        line_nums.pack(side="left", fill="y")
        editor.pack(side="right", fill="both", expand=True)

        # Petit bouton de fermeture en haut à droite (Option 1)
        # Créé après l'éditeur pour être au-dessus
        close_btn = ctk.CTkButton(editor_frame, text="✕", width=25, height=25, 
                                  fg_color="#c0392b", hover_color="#e74c3c",
                                  command=lambda f=tab_name: self.close_tab(f))
        close_btn.place(relx=1.0, rely=0.0, anchor="ne", x=-10, y=10)
        close_btn.lift() # Force le passage au premier plan
        
        self.setup_syntax_tags(editor)
        try:
            with open(file_path, "r", encoding="utf-8") as f:
                editor.insert("1.0", f.read())
        except Exception as e:
            editor.insert("1.0", f"Erreur de lecture : {e}")
            
        self.apply_pygments(editor)
        
        # Liaison pour les numéros de ligne
        def on_scroll(*args):
            line_nums.redraw()
            editor._textbox.yview(*args)
            
        # Hook pour redessiner au scroll et modif
        editor._textbox.configure(yscrollcommand=lambda *args: line_nums.redraw())
        editor.bind("<KeyRelease>", lambda e: [self.apply_pygments(editor), line_nums.redraw()])
        
        self.open_files[tab_name] = editor
        self.editor_tabs.set(tab_name)
        # Premier dessin
        self.after(100, line_nums.redraw)

    # --- UI & CONFIG ---

    def change_appearance_mode(self, new_mode):
        ctk.set_appearance_mode(new_mode)
        bg = "#2b2b2b" if new_mode == "Dark" else "#dbdbdb"
        self.main_vertical_paned.configure(bg=bg)
        self.top_paned_window.configure(bg=bg)
        self.setup_tree_style() # Mettre à jour le style du Treeview

    def change_project(self, name):
        # Sauvegarde la session du projet actuel avant de changer
        if self.current_project_path:
            self.save_current_session()
            self.save_all_open_files()
            # On ne vide pas chat_histories ici, c'est fait au chargement global ou via config
            for filename in list(self.open_files.keys()):
                self.editor_tabs.delete(filename)
            self.open_files.clear()
        
        self.current_project_path = os.path.join(self.base_projects_dir, name)
        self.setup_tree_style() # Refresh style for theme
        self.refresh_file_list_display()
        
        # Reset affichage chat et chargement historique
        self.chat_display.configure(state="normal")
        self.chat_display.delete("1.0", "end")
        if name in self.chat_histories:
            self.chat_display.insert("1.0", self.chat_histories[name])
        self.chat_display.configure(state="disabled")
        self.chat_display.see("end")

        # S'assurer que l'environnement virtuel existe
        if name != "Aucun":
            self.ensure_venv(self.current_project_path)

        # Restaure les onglets du nouveau projet
        self.restore_project_session(name)

    def save_all_open_files(self, event=None):
        for filename, editor in self.open_files.items():
            content = editor.get("1.0", "end-1c")
            file_path = os.path.join(self.current_project_path, filename)
            try:
                with open(file_path, "w", encoding="utf-8") as f: f.write(content)
            except Exception as e:
                print(f"Erreur sauvegarde {filename}: {e}")

    def open_selected_file(self):
        selected_item = self.tree.selection()
        if not selected_item:
            return
            
        item_id = selected_item[0]
        full_rel_path = self.get_tree_full_path(item_id)
        
        # On n'ouvre que si c'est un fichier
        abs_path = os.path.join(self.current_project_path, full_rel_path)
        if os.path.isfile(abs_path):
            self.open_file_in_tab(None, relative_path=full_rel_path)

    def get_tree_full_path(self, item_id):
        """Reconstruit le chemin relatif à partir de l'arborescence du Treeview."""
        path_parts = []
        curr = item_id
        while curr:
            name = self.tree.item(curr, "text").replace("📁 ", "").replace("📄 ", "")
            if name:
                path_parts.insert(0, name)
            curr = self.tree.parent(curr)
        return os.path.join(*path_parts) if path_parts else ""

    def load_config(self):
        if os.path.exists(CONFIG_FILE):
            try:
                with open(CONFIG_FILE, "r") as f:
                    config = json.load(f)
                    self.geometry(config.get("geometry", "1200x800"))
                    ctk.set_appearance_mode(config.get("theme", "Dark"))
                    self.saved_model = config.get("model", "qwen2.5-coder:7b")
                    self.sessions = config.get("sessions", {}) # Charge les sessions
                    self.chat_histories = config.get("chat_histories", {}) # Charge les historiques
            except: 
                self.sessions = {}
                self.chat_histories = {}
        else:
            ctk.set_appearance_mode("Dark")

    def on_closing(self):
        self.save_current_session() # Sauvegarde avant de quitter
        self.save_all_open_files()
        
        # Sauvegarde historique chat actuel avant de fermer
        if self.current_project_path:
            project_name = os.path.basename(self.current_project_path)
            self.chat_histories[project_name] = self.chat_display.get("1.0", "end-1c")

        config = {
            "geometry": self.geometry(),
            "theme": ctk.get_appearance_mode(),
            "model": self.model_menu.get(),
            "sessions": self.sessions,
            "chat_histories": self.chat_histories
        }
        with open(CONFIG_FILE, "w") as f: json.dump(config, f, indent=4)
        self.destroy()

    def get_list_projects(self):
        if not os.path.exists(self.base_projects_dir): return ["Aucun"]
        projs = [d for d in os.listdir(self.base_projects_dir) if os.path.isdir(os.path.join(self.base_projects_dir, d))]
        return projs if projs else ["Aucun"]

    def create_new_project(self):
        """Demande un nom et crée un nouveau dossier de projet."""
        name = simpledialog.askstring("Nouveau Projet", "Nom du dossier projet :", parent=self)
        if name:
            # Nettoyage minimal du nom
            name = name.strip().replace(" ", "_")
            path = os.path.join(self.base_projects_dir, name)
            if not os.path.exists(path):
                try:
                    os.makedirs(path)
                    # Mise à jour du menu
                    self.proj_menu.configure(values=self.get_list_projects())
                    self.proj_menu.set(name)
                    self.change_project(name)
                    self.append_chat(f"\n[SYSTEM]: Nouveau projet '{name}' créé.\n")
                except Exception as e:
                    messagebox.showerror("Erreur", f"Erreur lors de la création : {e}", parent=self)
            else:
                messagebox.showwarning("Attention", "Ce dossier existe déjà.", parent=self)

    def refresh_file_list_display(self):
        # Nettoyage
        for item in self.tree.get_children():
            self.tree.delete(item)
            
        python_files = []
        if os.path.exists(self.current_project_path):
            self.populate_tree("", self.current_project_path)
            
            # Rechercher récursivement les scripts Python pour le menu de lancement
            for root, dirs, files in os.walk(self.current_project_path):
                # Ignorer les dossiers cachés et venv
                dirs[:] = [d for d in dirs if not d.startswith('.') and d != ".venv"]
                for f in files:
                    if f.endswith(".py"):
                        # Stocker le chemin relatif pour le lancement
                        rel = os.path.relpath(os.path.join(root, f), self.current_project_path)
                        python_files.append(rel)

        # Mise à jour du menu des scripts
        if not python_files:
            self.script_menu.configure(values=["Aucun"])
            self.script_menu.set("Aucun")
        else:
            self.script_menu.configure(values=python_files)
            if self.script_menu.get() not in python_files:
                self.script_menu.set(python_files[0])

    def populate_tree(self, parent_node, path):
        """Remplit le Treeview récursivement."""
        try:
            items = os.listdir(path)
            # Trier : dossiers d'abord, puis fichiers
            items.sort(key=lambda x: (not os.path.isdir(os.path.join(path, x)), x.lower()))
            
            for item in items:
                if item.startswith('.') or item == "__pycache__" or item == ".venv":
                    continue
                    
                full_path = os.path.join(path, item)
                is_dir = os.path.isdir(full_path)
                
                icon = "📁 " if is_dir else "📄 "
                node = self.tree.insert(parent_node, "end", text=f"{icon}{item}", open=False)
                
                if is_dir:
                    self.populate_tree(node, full_path)
        except Exception as e:
            print(f"Erreur populate_tree : {e}")

    def _handle_return(self, event):
        """Gère l'appui sur Entrée (Envoi) vs Shift+Entrée (Nouvelle ligne)"""
        if not event.state & 0x0001: # Si Shift n'est PAS pressé (0x0001 est le mask pour Shift)
            self.send_message()
            return "break" # Empêche l'insertion du saut de ligne
        return None

    def send_message(self, event=None):
        prompt = self.input_entry.get("1.0", "end-1c").strip()
        if prompt:
            self.append_chat(f"\nVOUS: {prompt}\n\nIA: ")
            self.input_entry.delete("1.0", "end")
            self.start_thinking_animation()
            threading.Thread(target=self.run_ai, args=(prompt,), daemon=True).start()

    def append_chat(self, text):
        self.chat_display.configure(state="normal")
        self.chat_display.insert("end", text)
        self.chat_display.configure(state="disabled")
        self.chat_display.see("end")
        
        # Mémorisation en temps réel de l'historique
        if self.current_project_path:
            project_name = os.path.basename(self.current_project_path)
            self.chat_histories[project_name] = self.chat_display.get("1.0", "end-1c")

    def ensure_venv(self, project_path):
        """Vérifie si un environnement virtuel existe, sinon le crée."""
        venv_path = os.path.join(project_path, ".venv")
        if not os.path.exists(venv_path):
            self.append_chat(f"\n[SYSTEM]: Environnement Python manquant dans {os.path.basename(project_path)}. Création en cours...\n")
            threading.Thread(target=self._create_venv, args=(project_path, venv_path), daemon=True).start()
        else:
            self.append_chat(f"\n[SYSTEM]: Environnement Python détecté et prêt.\n")

    def _create_venv(self, project_path, venv_path):
        """Effectue la création du venv en arrière-plan."""
        try:
            # Création du venv
            subprocess.run([sys.executable, "-m", "venv", ".venv"], check=True, cwd=project_path)
            self.after(0, lambda: self.append_chat(f"\n[SYSTEM]: Environnement créé avec succès dans {os.path.basename(project_path)}.\n"))
        except Exception as e:
            self.after(0, lambda: messagebox.showerror("Erreur Venv", f"Échec de création du venv : {e}", parent=self))

    def append_terminal(self, text):
        """Ajoute du texte au terminal de manière sécurisée."""
        self.terminal_output.configure(state="normal")
        self.terminal_output.insert("end", text)
        self.terminal_output.configure(state="disabled")
        self.terminal_output.see("end")

    def run_main_script(self):
        """Lance l'exécution du script sélectionné dans le venv."""
        self.save_all_open_files()
        script_name = self.script_menu.get()
        if script_name == "Aucun" or not self.current_project_path:
            messagebox.showwarning("Attention", "Veuillez sélectionner un script valide.", parent=self)
            return

        # Chemin vers python dans le venv
        if sys.platform == "win32":
            python_exe = os.path.join(self.current_project_path, ".venv", "Scripts", "python.exe")
        else:
            python_exe = os.path.join(self.current_project_path, ".venv", "bin", "python")

        if not os.path.exists(python_exe):
            messagebox.showerror("Erreur", "Environnement virtuel introuvable. Veuillez patienter ou recréer le projet.", parent=self)
            return

        self.append_terminal(f"\n--- LANCEMENT : {script_name} ---\n")
        
        cmd = [python_exe, "-u", script_name]
        threading.Thread(target=self._execute_command, args=(cmd,), daemon=True).start()

    def send_terminal_command(self):
        """Récupère et exécute la commande tapée dans le terminal."""
        command = self.terminal_entry.get().strip()
        if not command:
            return
        
        self.terminal_entry.delete(0, 'end')
        
        if not self.current_project_path:
            messagebox.showwarning("Attention", "Veuillez d'abord sélectionner un projet.", parent=self)
            return

        self.append_terminal(f"\n> {command}\n")
        
        # Prépare la commande (gestion venv)
        cmd_parts = command.split()
        
        # Si on utilise pip ou python, on force le venv si disponible
        if cmd_parts[0] in ["pip", "python"]:
            if sys.platform == "win32":
                exe_name = "pip.exe" if cmd_parts[0] == "pip" else "python.exe"
                venv_exe = os.path.join(self.current_project_path, ".venv", "Scripts", exe_name)
            else:
                exe_name = "pip" if cmd_parts[0] == "pip" else "python"
                venv_exe = os.path.join(self.current_project_path, ".venv", "bin", exe_name)
            
            if os.path.exists(venv_exe):
                cmd_parts[0] = venv_exe
        
        threading.Thread(target=self._execute_command, args=(cmd_parts,), daemon=True).start()

    def _execute_command(self, cmd):
        """Exécute la commande (liste de chaînes) et capture la sortie."""
        try:
            # Sur Windows, shell=True peut être nécessaire pour certaines commandes système comme 'dir'
            use_shell = False
            if sys.platform == "win32" and isinstance(cmd, list) and cmd[0] in ["dir", "echo", "cls", "del", "copy"]:
                use_shell = True
                cmd = " ".join(cmd) # subprocess.Popen veut une string avec shell=True sur Windows
            
            process = subprocess.Popen(
                cmd,
                stdout=subprocess.PIPE,
                stderr=subprocess.STDOUT,
                text=True,
                cwd=self.current_project_path,
                bufsize=1,
                shell=use_shell
            )
            
            for line in process.stdout:
                self.after(0, self.append_terminal, line)
            
            process.wait()
            self.after(0, self.append_terminal, f"\n--- FIN : Code sortie {process.returncode} ---\n")
        except Exception as e:
            self.after(0, self.append_terminal, f"\n[ERREUR EXÉCUTION]: {e}\n")

if __name__ == "__main__":
    app = AIApp()
    app.mainloop()
