import os
import re
import html
import urllib.request
import json
from urllib.parse import quote

# --- Configuration ---
SITE_ROOT = "site_genere"
INDEX_FILENAME = "index.html"
ARTICLES_DIR_NAME = "articles"
ASSETS_DIR_NAME = "assets"

# Bibliothèque locale pour la recherche
FUSE_JS_URL = "https://cdn.jsdelivr.net/npm/fuse.js@7.0.0/dist/fuse.basic.min.js"

# Template avec le look "GitHub Dark" harmonisé
INDEX_TEMPLATE = """<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title>Index des articles</title>
    <style>
        :root {{
            --bg-color: #0d1117;
            --card-bg: #161b22;
            --text-color: #c9d1d9;
            --accent-color: #58a6ff;
            --border-color: #30363d;
        }}
        body {{ 
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Helvetica, Arial, sans-serif; 
            line-height: 1.6; 
            max-width: 1000px; 
            margin: auto; 
            padding: 40px 20px; 
            background-color: var(--bg-color); 
            color: var(--text-color); 
        }}
        h1 {{ 
            color: white; 
            border-bottom: 1px solid var(--border-color); 
            padding-bottom: 15px;
            font-size: 2em;
        }}
        .search-container {{ margin: 30px 0; display: flex; gap: 10px; }}
        #search-input {{ 
            flex: 1; padding: 12px 15px; font-size: 16px; 
            background: var(--card-bg); border: 1px solid var(--border-color);
            border-radius: 6px; color: white; outline: none;
        }}
        #search-input:focus {{ border-color: var(--accent-color); box-shadow: 0 0 0 3px rgba(88,166,255,0.3); }}
        
        .sort-buttons {{ display: flex; gap: 8px; }}
        .sort-btn {{ 
            padding: 10px 16px; background: var(--card-bg); 
            border: 1px solid var(--border-color); color: var(--text-color);
            border-radius: 6px; cursor: pointer; font-size: 14px;
        }}
        .sort-btn.active {{ background: var(--accent-color); color: white; border-color: var(--accent-color); }}

        #article-list {{ list-style: none; padding: 0; display: grid; grid-template-columns: 1fr; gap: 12px; }}
        .article-card {{ 
            background: var(--card-bg); padding: 20px; border-radius: 6px; 
            border: 1px solid var(--border-color); transition: 0.2s;
        }}
        .article-card:hover {{ border-color: var(--accent-color); transform: translateY(-2px); }}
        .article-card a {{ 
            text-decoration: none; color: var(--accent-color); 
            font-weight: 600; font-size: 1.2em; display: block; margin-bottom: 5px;
        }}
        .article-date {{ color: #8b949e; font-size: 0.85em; }}
    </style>
</head>
<body>
    <h1>📚 Archive des Articles</h1>
    <div class="search-container">
        <input type="text" id="search-input" placeholder="Rechercher un article...">
        <div class="sort-buttons">
            <button class="sort-btn active" id="sort-date">Plus récents</button>
            <button class="sort-btn" id="sort-name">A-Z</button>
        </div>
    </div>
    <ul id="article-list"></ul>

    <script src="assets/fuse.basic.min.js"></script>
    <script>
        const articles = {articles_data};
        const searchInput = document.getElementById('search-input');
        const articleList = document.getElementById('article-list');
        const sortNameBtn = document.getElementById('sort-name');
        const sortDateBtn = document.getElementById('sort-date');
        
        let currentSort = 'date';
        const fuse = new Fuse(articles, {{ keys: ['title'], threshold: 0.3 }});
        
        function renderArticles(list) {{
            articleList.innerHTML = '';
            list.forEach(art => {{
                const li = document.createElement('li');
                li.className = 'article-card';
                li.innerHTML = `
                    <a href="${{art.url}}">${{art.title}}</a>
                    <div class="article-date">📅 ${{art.date || 'Date inconnue'}}</div>
                `;
                articleList.appendChild(li);
            }});
        }}

        function update() {{
            const query = searchInput.value.trim();
            let results = query ? fuse.search(query).map(r => r.item) : [...articles];
            if (currentSort === 'name') {{
                results.sort((a, b) => a.title.localeCompare(b.title));
            }} else {{
                results.sort((a, b) => (b.date || '').localeCompare(a.date || ''));
            }}
            renderArticles(results);
        }}
        
        searchInput.addEventListener('input', update);
        sortNameBtn.addEventListener('click', () => {{ currentSort = 'name'; setActive(sortNameBtn); update(); }});
        sortDateBtn.addEventListener('click', () => {{ currentSort = 'date'; setActive(sortDateBtn); update(); }});
        function setActive(btn) {{
            document.querySelectorAll('.sort-btn').forEach(b => b.classList.remove('active'));
            btn.classList.add('active');
        }}
        update();
    </script>
</body>
</html>
"""

# --- Parsing ---

def get_page_title(file_path):
    try:
        with open(file_path, 'r', encoding='utf-8') as f:
            content = f.read()
            match = re.search(r'<title>(.*?)</title>', content, re.IGNORECASE)
            return html.unescape(match.group(1).strip()) if match else os.path.basename(file_path)
    except: return os.path.basename(file_path)

def get_page_date(file_path):
    """
    Extrait la date en fonction du format généré dans wp_xml_to_html.py :
    <p class="date">Publié le YYYY-MM-DD</p>
    """
    try:
        with open(file_path, 'r', encoding='utf-8') as f:
            content = f.read()
            # Cherche spécifiquement la structure générée par l'autre script
            match = re.search(r'class="date">Publié le ([\d-]+)', content)
            if match:
                return match.group(1).strip()
    except: pass
    return None

# --- Utilitaires de navigation ---

def add_back_link(file_path):
    try:
        with open(file_path, 'r', encoding='utf-8') as f:
            content = f.read()
        if 'class="back-to-index"' in content: return
        # Lien stylisé pour le mode sombre
        link_html = '\n<div style="margin-bottom:20px;"><a href="../index.html" class="back-to-index" style="color:#58a6ff; text-decoration:none;">← Retour à l\'index</a></div>\n'
        new_content = content.replace('<body>', '<body>' + link_html, 1)
        with open(file_path, 'w', encoding='utf-8') as f:
            f.write(new_content)
    except: pass

def add_nav_bar(file_path, prev_path, next_path):
    try:
        with open(file_path, 'r', encoding='utf-8') as f:
            content = f.read()
        if 'class="nav-bar"' in content: return
        nav = ['<div class="nav-bar" style="margin-top:40px; border-top:1px solid #30363d; padding-top:20px; display:flex; justify-content:space-between;">']
        if prev_path:
            nav.append(f'<a href="{quote(os.path.basename(prev_path))}" style="color:#8b949e; text-decoration:none;">« Précédent</a>')
        else: nav.append('<span></span>')
        if next_path:
            nav.append(f'<a href="{quote(os.path.basename(next_path))}" style="color:#8b949e; text-decoration:none;">Suivant »</a>')
        nav.append('</div>')
        new_content = content.replace('</article>', '\n'.join(nav) + '\n</article>', 1)
        with open(file_path, 'w', encoding='utf-8') as f:
            f.write(new_content)
    except: pass

def download_asset(url, target_path):
    if os.path.exists(target_path): return
    try:
        print(f"Téléchargement de l'asset local : {os.path.basename(target_path)}...")
        req = urllib.request.Request(url, headers={"User-Agent": "Mozilla/5.0"})
        with urllib.request.urlopen(req, timeout=10) as response:
            with open(target_path, "wb") as f: f.write(response.read())
    except Exception as e: print(f"Erreur asset: {e}")

# --- Orchestration ---

def generate_index():
    articles_dir = os.path.join(SITE_ROOT, ARTICLES_DIR_NAME)
    assets_dir = os.path.join(SITE_ROOT, ASSETS_DIR_NAME)
    os.makedirs(assets_dir, exist_ok=True)

    download_asset(FUSE_JS_URL, os.path.join(assets_dir, "fuse.basic.min.js"))

    html_files = []
    for root, _, files in os.walk(articles_dir):
        for file in files:
            if file.lower().endswith('.html'):
                html_files.append(os.path.join(root, file))
    html_files.sort()
    
    articles_data = []
    for f_path in html_files:
        rel_url = os.path.relpath(f_path, SITE_ROOT).replace(os.path.sep, '/')
        articles_data.append({
            'url': quote(rel_url),
            'title': get_page_title(f_path),
            'date': get_page_date(f_path)
        })
        add_back_link(f_path)

    for i, f_path in enumerate(html_files):
        prev = html_files[i-1] if i > 0 else None
        next_f = html_files[i+1] if i < len(html_files) - 1 else None
        add_nav_bar(f_path, prev, next_f)

    with open(os.path.join(SITE_ROOT, INDEX_FILENAME), 'w', encoding='utf-8') as f:
        f.write(INDEX_TEMPLATE.format(articles_data=json.dumps(articles_data, ensure_ascii=False)))
    
    print(f"✅ Terminé : {len(articles_data)} articles indexés avec succès.")

if __name__ == "__main__":
    generate_index()
