import threading
import random
import copy

class SequencerModel:
    def __init__(self, num_tracks=6, num_steps=16):
        self.num_tracks = num_tracks
        self.num_steps = num_steps
        self.grid = [[False for _ in range(num_steps)] for _ in range(num_tracks)]
        self.bpm = 120
        self.current_kit = "808_Kit"
        self.track_names = []
        self.grid_lock = threading.Lock()
        self.is_loading = False # Flag de chargement
        self.auto_trim = False # Nouvelle option
        self.master_volume = 1.0 # Volume général (0.0 à 1.0)
        self.track_volumes = [1.0] * num_tracks # Volumes par piste
        self.mute_states = [False] * num_tracks # État Mute
        self.solo_states = [False] * num_tracks # État Solo
        self.time_signature = "4/4" # Signature par défaut
        
        # New Features V4
        self.swing = 0.0 # 0.0 to 0.5
        self.playlist = [] # List of grid snapshots (deep copies)
        self.song_mode = False
        self.song_index = 0
        self.grid_has_changed = False # Flag for UI update
        self.grid_has_changed = False # Flag for UI update
        self.current_style_name = "Aucun"
        
        # V7: Pan & Pitch
        self.track_pan = [0.0] * num_tracks # -1.0 (Left) to 1.0 (Right)
        self.track_pitch = [1.0] * num_tracks # 1.0 = Normal speed
        
        # V9: Smart Accompaniment
        # V9: Smart Accompaniment
        self.chords = [None] * num_steps 
        self.fill_chords = True # Repeat last chord if empty (Activates Generator)
        self.bass_style = "Arpeggio" # Default Style
        # Per-step pitch override (0.0 = Use Track Pitch, > 0.0 = Absolute Pitch Factor)
        self.grid_pitch = [[0.0 for _ in range(num_steps)] for _ in range(num_tracks)]

    def set_tracks(self, names):
        with self.grid_lock:
            self.track_names = names
            self.num_tracks = len(names)
            # Redimensionner la grille et les volumes si nécessaire
            new_grid = [[False for _ in range(self.num_steps)] for _ in range(self.num_tracks)]
            new_volumes = [1.0] * self.num_tracks
            new_mutes = [False] * self.num_tracks
            new_solos = [False] * self.num_tracks
            new_pans = [0.0] * self.num_tracks
            new_pitches = [1.0] * self.num_tracks
            new_grid_pitch = [[0.0 for _ in range(self.num_steps)] for _ in range(self.num_tracks)]
            for r in range(min(len(self.grid), self.num_tracks)):
                for c in range(self.num_steps):
                    new_grid[r][c] = self.grid[r][c]
                if r < len(self.track_volumes):
                    new_volumes[r] = self.track_volumes[r]
                    if r < len(self.mute_states):
                        new_mutes[r] = self.mute_states[r]
                    if r < len(self.solo_states):
                        new_solos[r] = self.solo_states[r]
                    if r < len(self.track_pan):
                        new_pans[r] = self.track_pan[r]
                    if r < len(self.track_pitch):
                        new_pitches[r] = self.track_pitch[r]
                    if r < len(self.grid_pitch):
                         for c in range(self.num_steps):
                             new_grid_pitch[r][c] = self.grid_pitch[r][c]
                             
            self.grid = new_grid
            self.track_volumes = new_volumes
            self.mute_states = new_mutes
            self.solo_states = new_solos
            self.track_pan = new_pans
            self.track_pitch = new_pitches
            self.grid_pitch = new_grid_pitch
            
    def delete_track_at(self, index):
        with self.grid_lock:
            if 0 <= index < self.num_tracks:
                del self.grid[index]
                del self.track_volumes[index]
                del self.mute_states[index]
                del self.solo_states[index]
                del self.track_pan[index]
                del self.track_pitch[index]
                # track_names sera mis à jour par set_tracks lors du reload du kit
                # mais le modèle interne doit rester cohérent en attendant
                self.num_tracks -= 1
                
    def insert_track_at(self, index):
        """Insère une piste vide à l'index donné (ou à la fin si index == -1)"""
        with self.grid_lock:
            new_row = [False] * self.num_steps
            if index == -1 or index >= self.num_tracks:
                self.grid.append(new_row)
                self.track_volumes.append(1.0)
                self.mute_states.append(False)
                self.solo_states.append(False)
                self.track_pan.append(0.0)
                self.track_pitch.append(1.0)
            else:
                self.grid.insert(index, new_row)
                self.track_volumes.insert(index, 1.0)
                self.mute_states.insert(index, False)
                self.solo_states.insert(index, False)
                self.track_pan.insert(index, 0.0)
                self.track_pitch.insert(index, 1.0)
            
            self.num_tracks += 1
            self.mute_states = new_mutes
            self.solo_states = new_solos
            self.track_pan = new_pans
            self.track_pitch = new_pitches

    def change_num_steps(self, new_steps):
        """Redimensionne le nombre de pas de la grille."""
        with self.grid_lock:
            new_grid = [[False for _ in range(new_steps)] for _ in range(self.num_tracks)]
            for r in range(self.num_tracks):
                for c in range(min(self.num_steps, new_steps)):
                    new_grid[r][c] = self.grid[r][c]
            self.grid = new_grid
            # Resize Chords
            if new_steps > self.num_steps:
                self.chords.extend([None] * (new_steps - self.num_steps))
            else:
                self.chords = self.chords[:new_steps]

            # Resize Grid Pitch
            for r in range(self.num_tracks):
                current_row_p = self.grid_pitch[r]
                if new_steps > self.num_steps:
                    current_row_p.extend([0.0] * (new_steps - self.num_steps))
                else:
                    self.grid_pitch[r] = current_row_p[:new_steps]
                    
            self.num_steps = new_steps

    def toggle_step(self, track_idx, step_idx, velocity=1.0):
        # Support Grid Lock if it exists (it was in __init__ in previous steps, assuming it's there)
        # Using getattr to be safe if lock not initialized in this specific file version for some reason
        lock = getattr(self, 'grid_lock', None)
        if lock:
            lock.acquire()
            
        try:
            if 0 <= track_idx < self.num_tracks and 0 <= step_idx < self.num_steps:
                current = self.grid[track_idx][step_idx]
                
                # Cast legacy
                if current is True: current = 1.0
                if current is False: current = 0.0
                
                if current == 0.0:
                    self.grid[track_idx][step_idx] = velocity
                elif abs(current - velocity) < 0.01:
                    self.grid[track_idx][step_idx] = 0.0
                else:
                    self.grid[track_idx][step_idx] = velocity

                self.grid_has_changed = True
        finally:
            if lock:
                lock.release()

    def get_effective_steps(self):
        """Retourne le nombre de pas réellement utilisés (jusqu'à la dernière note)."""
        if not self.auto_trim:
            return self.num_steps
            
        max_step = 0
        for r in range(self.num_tracks):
            for c in range(self.num_steps - 1, -1, -1):
                if self.grid[r][c]:
                    if c > max_step:
                        max_step = c
                    break
        return max_step + 1

    def clear_grid(self):
        self.grid = [[False for _ in range(self.num_steps)] for _ in range(self.num_tracks)]

    def randomize_grid(self):
        """Génère un rythme semi-aléatoire cohérent."""
        with self.grid_lock:
            # On efface d'abord
            self.clear_grid()
            
            # Pour chaque piste, on applique une logique différente
            # On suppose un ordre standard 808 : Snare, CHH, OHH, Clap, Cymbal, Kick
            # Ou on essaie de deviner par le nom de la piste si possible, sinon random générique
            
            for r in range(self.num_tracks):
                name = self.track_names[r].upper() if r < len(self.track_names) else ""
                
                # Probabilités par pas (0-15)
                # Temps forts: 0, 4, 8, 12
                
                if "KICK" in name or "GROSSE" in name:
                    # Kick: Fort sur 0 et 8, aléatoire ailleurs mais pas trop chargé
                    self.grid[r][0] = True
                    if random.random() > 0.3: self.grid[r][8] = True
                    if random.random() > 0.6: self.grid[r][4] = True # Kick sur snare parfois ?
                    # Syncopes
                    if random.random() > 0.7: self.grid[r][10] = True
                    if random.random() > 0.7: self.grid[r][14] = True
                    if random.random() > 0.8: self.grid[r][2] = True
                    
                elif "SNARE" in name or "CAISSE" in name or "CLAP" in name:
                    # Snare standard sur 4 et 12
                    if random.random() > 0.1: self.grid[r][4] = True
                    if random.random() > 0.1: self.grid[r][12] = True
                    # Ghost notes
                    if random.random() > 0.8: self.grid[r][15] = True
                    if random.random() > 0.8: self.grid[r][6] = True
                    
                elif "HAT" in name or "CHARLEST" in name:
                    if "OPEN" in name or "OUVERT" in name:
                        # Open Hat sur le contre-temps (2, 6, 10, 14)
                        for i in range(2, 16, 4):
                            if random.random() > 0.3: self.grid[r][i] = True
                    else:
                        # Closed Hat : Souvent tous les 2 pas ou tous les pas
                        pattern = random.choice([1, 2, 4])
                        for i in range(0, 16, pattern):
                            if random.random() > 0.1: self.grid[r][i] = True
                            
                else:
                    # Percus / Autres : Aléatoire clairsemé
                    for i in range(16):
                        if random.random() > 0.85: # 15% de chance
                            self.grid[r][i] = True
