import math
import random

class SmartComposer:
    # Mapping Note Name -> Semitone Offset (Relative to C)
    NOTES_MAP = {
        "C": 0, "C#": 1, "Db": 1,
        "D": 2, "D#": 3, "Eb": 3,
        "E": 4,
        "F": 5, "F#": 6, "Gb": 6,
        "G": 7, "G#": 8, "Ab": 8,
        "A": 9, "A#": 10, "Bb": 10,
        "B": 11
    }
    
    # Intervals relative to root
    CHORD_INTERVALS = {
        "Maj": [0, 4, 7],         # Fundamental, Major Third, Perfect Fifth
        "Min": [0, 3, 7],         # Fundamental, Minor Third, Perfect Fifth
        "7": [0, 4, 7, 10],       # Dom 7
        "Maj7": [0, 4, 7, 11],
        "Min7": [0, 3, 7, 10]
    }
    
    @staticmethod
    def get_pitch_factor(root_name, octave_offset=-1, interval=0):
        """
        Calcule le facteur de pitch.
        interval: décalage en demi-tons par rapport à la fondamentale (ex: 7 pour quinte).
        """
        semitone = SmartComposer.NOTES_MAP.get(root_name, 0)
        total_semitones = semitone + (octave_offset * 12) + interval
        
        # P = 2^(n/12)
        return 2.0 ** (total_semitones / 12.0)

    @staticmethod
    def get_style_interval(style_name, step, current_quality="Maj"):
        """Retourne l'intervalle (demi-tons) pour le style et le pas donnés."""
        s = style_name.lower()
        
        # 1. ARPEGGIO (Random)
        if "arpeggio" in s:
            intervals = SmartComposer.CHORD_INTERVALS.get(current_quality, [0, 7])
            return random.choice(intervals)
            
        # 2. DISCO / FUNK (Octaves)
        if "disco" in s or "funk" in s or "dance" in s or "house" in s:
            # Offbeat Octave: 0, 12, 0, 12
            if step % 2 == 0: return 0
            else: return 12
            
        # 3. ROCK / METAL (Driving Root)
        if "rock" in s or "metal" in s or "pop" in s:
            # 8th notes driving: Root.
            # Maybe accented 0, 0, 0, 0.
            # Occasionally 5th?
            if step % 8 == 7: return 7 # Turnaround
            return 0
            
        # 4. BLUES / JAZZ (Walking - Simplified)
        if "blues" in s or "jazz" in s or "swing" in s:
            # Root, 3rd, 5th, 6th
            seq = [0, 3 if "Min" in current_quality else 4, 7, 9]
            return seq[step % 4]
            
        # 5. WALTZ (Oom-Pah-Pah) - 3/4 usually, but mapped to 4/4 grid
        if "waltz" in s or "valse" in s:
            # 0, 7, 7, 0...
            pat = [0, 7, 7, 0]
            return pat[step % 4]
            
        # Default: Root
        return 0

    @staticmethod
    def update_bass_track(model):
        """
        Met à jour la piste Bass.
        """
        kick_idx = -1
        bass_idx = -1
        
        # 1. Identifier les pistes
        for i, name in enumerate(model.track_names):
            lower = name.lower()
            if "kick" in lower or "grosse" in lower or "bd" in lower:
                kick_idx = i
            if "bass" in lower or "basse" in lower or "808" in lower:
                bass_idx = i
        
        if kick_idx == -1 or bass_idx == -1:
            return False
            
        current_chord_name = "C"
        current_quality = "Maj"
        
        is_fill_mode = getattr(model, 'fill_chords', False)
        style_name = getattr(model, 'bass_style', "Arpeggio")
        
        with model.grid_lock:
            for step in range(model.num_steps):
                # Update Chord State
                if step < len(model.chords) and model.chords[step] is not None:
                    val = model.chords[step]
                    if isinstance(val, tuple):
                        current_chord_name = val[0]
                        current_quality = val[1]
                    else:
                        current_chord_name = str(val)
                        current_quality = "Maj"
                
                # Determine Activation
                should_play = False
                velocity = 1.0
                
                if is_fill_mode:
                    # Fill Mode: Always play (unless Silence style?)
                    should_play = True
                    velocity = 0.9
                else:
                    # Kick Lock Mode
                    kick_val = False
                    if kick_idx != -1 and kick_idx < len(model.grid) and step < len(model.grid[kick_idx]):
                         kick_val = model.grid[kick_idx][step]
                         
                    if kick_val:
                        should_play = True
                        velocity = kick_val if isinstance(kick_val, float) else 1.0

                if bass_idx < len(model.grid) and step < len(model.grid[bass_idx]):
                    if should_play:
                        model.grid[bass_idx][step] = velocity
                        
                        # Determine Pitch via Style
                        interval = SmartComposer.get_style_interval(style_name, step, current_quality)
                        pitch = SmartComposer.get_pitch_factor(current_chord_name, octave_offset=0, interval=interval)
                        
                        if bass_idx < len(model.grid_pitch) and step < len(model.grid_pitch[bass_idx]):
                            model.grid_pitch[bass_idx][step] = pitch
                    else:
                        model.grid[bass_idx][step] = False
                        if bass_idx < len(model.grid_pitch) and step < len(model.grid_pitch[bass_idx]):
                            model.grid_pitch[bass_idx][step] = 0.0
        
        model.grid_has_changed = True
        return True
        
        model.grid_has_changed = True
        return True
