import tkinter as tk
import copy
import customtkinter as ctk
import threading
import os
import sys
import json
import copy
from tkinter import filedialog, messagebox, simpledialog
from .models import SequencerModel
from .audio import AudioController
from .smart_logic import SmartComposer
import webbrowser

# Configuration ctk
ctk.set_appearance_mode("Dark")
ctk.set_default_color_theme("blue")

class PyBeatApp(ctk.CTk):
    BUTTON_COLORS = ["#FF5733", "#33FF57", "#3357FF", "#F3FF33", "#FF33F3", "#33FFF3", 
                     "#FF8C00", "#9932CC", "#8B4513", "#2F4F4F"]
    CONFIG_FILE = "config.json"

    def __init__(self):
        super().__init__()

        self.title("PyBeat - Séquenceur Rythmique")
        
        # Charger la géométrie mémorisée ou défaut
        self._load_window_config()

        self.model = SequencerModel()
        self.audio = AudioController(self.model)

        self._setup_ui()
        
        # Suivi du curseur de lecture pour l'affichage
        self.last_playing_step = -1

        # Intercepter la fermeture pour sauvegarder la config
        self.protocol("WM_DELETE_WINDOW", self._on_closing)
        
        # Init Background Update Loops
        self.after(100, self._check_grid_update)
        
        # Init Background Update Loops
        self.after(100, self._check_grid_update)
        
        # Raccourcis Clavier (Global)
        self.bind_all("<space>", lambda e: self._toggle_play())
        self.bind_all("<Delete>", lambda e: self._clear())
        self.bind_all("<Escape>", lambda e: self._stop())
        self.bind_all("<BackSpace>", lambda e: self._clear()) # Alternative clear

    def _finalize_kit_change(self, names, was_playing, new_grid=None):
        try:
            self.model.set_tracks(names)
            if new_grid:
                # Adapter la grille chargée au nombre de pistes réel du kit
                # (On évite les IndexErrors si le kit a moins de pistes que le style)
                adapted_grid = [[False for _ in range(self.model.num_steps)] for _ in range(self.model.num_tracks)]
                for r in range(min(len(new_grid), self.model.num_tracks)):
                    for c in range(min(len(new_grid[r]), self.model.num_steps)):
                        adapted_grid[r][c] = new_grid[r][c]
                self.model.grid = adapted_grid

            # Afficher un message de chargement
            for widget in self.grid_frame.winfo_children():
                widget.destroy()
            self.loading_label = ctk.CTkLabel(self.grid_frame, text="Chargement de l'interface...", font=("Arial", 20))
            self.loading_label.pack(expand=True, pady=100) # Centrer verticalement à peu près
            self.update_idletasks()
            
            # Démarrer la création incrémentale
            self.buttons = []
            self.chord_buttons = []
            self._incremental_create_tracks(0, was_playing)
        except Exception as e:
            print(f"Erreur durant la finalisation du kit : {e}")
            self.model.is_loading = False

    def _incremental_create_tracks(self, track_idx, was_playing):
        try:
            if track_idx == 0:
                # Nettoyage final avant de commencer la grille
                if hasattr(self, 'loading_label'):
                    self.loading_label.destroy()
                for widget in self.grid_frame.winfo_children():
                    widget.destroy()
                # On s'assure que le frame est bien affiché
                self.grid_frame.pack(pady=10, padx=20, fill="both", expand=True)

                # --- ROW 0: CHORD LANE ---
                chord_label_frame = ctk.CTkFrame(self.grid_frame, fg_color="transparent")
                chord_label_frame.grid(row=0, column=0, padx=5, pady=5, sticky="w")
                
                ctk.CTkLabel(chord_label_frame, text="CHORDS", font=("Arial", 12, "bold"), text_color="#FFD700").pack(side="top", anchor="w")
                
                # Options Button "O"
                opt_btn = ctk.CTkButton(chord_label_frame, text="O", width=30, height=20, font=("Arial", 10, "bold"),
                                        fg_color="#444", hover_color="#666",
                                        command=self._open_bass_options_window)
                opt_btn.pack(side="bottom", anchor="w", pady=2)

                # Colonnes 1,2,3,4 vides pour l'alignement
                
                for i in range(self.model.num_steps):
                    # Bouton Accord
                    txt = self.model.chords[i] if (i < len(self.model.chords) and self.model.chords[i]) else ""
                    if isinstance(txt, tuple): txt = txt[0] + txt[1] # "C" + "Maj"
                    if not txt: txt = "-"
                    
                    btn = ctk.CTkButton(self.grid_frame, text=txt, width=40, height=25, 
                                        fg_color="#333", text_color="#FFD700", font=("Arial", 9),
                                        command=lambda s=i: self._on_chord_click(s))
                    btn.grid(row=0, column=i+5, padx=2, pady=2)
                    self.chord_buttons.append(btn)

                # --- ROW 1: HEADER (Numéros) ---
                ctk.CTkLabel(self.grid_frame, text="", width=120).grid(row=1, column=0)
                ctk.CTkLabel(self.grid_frame, text="VOL", width=80, font=("Arial", 10, "bold"), text_color="gray60").grid(row=1, column=1, padx=5)
                ctk.CTkLabel(self.grid_frame, text="M", width=30, font=("Arial", 10, "bold"), text_color="gray60").grid(row=1, column=2, padx=2)
                ctk.CTkLabel(self.grid_frame, text="S", width=30, font=("Arial", 10, "bold"), text_color="gray60").grid(row=1, column=3, padx=2)
                
                for i in range(self.model.num_steps):
                    color = "cyan" if (i % 4 == 0) else "gray60"
                    ctk.CTkLabel(self.grid_frame, text=str(i+1), width=40, font=("Arial", 10, "bold"), text_color=color).grid(row=1, column=i+5, padx=2)
            
            if track_idx < self.model.num_tracks:
                row_buttons = []
                grid_row = track_idx + 2 # Offset: Row 0=Chords, Row 1=Header
                color_idx = track_idx % len(self.BUTTON_COLORS)
                
                # Track Label
                lbl_name = ctk.CTkLabel(self.grid_frame, text=self.model.track_names[track_idx], width=120, anchor="w")
                lbl_name.grid(row=grid_row, column=0, padx=5, pady=5)
                lbl_name.bind("<Button-3>", lambda e, r=track_idx: self._show_track_menu(e, r))
                
                # Slider Volume
                vol_slider = ctk.CTkSlider(self.grid_frame, from_=0.0, to=1.0, width=80, 
                                           command=lambda v, r=track_idx: self._on_track_vol_change(r, v))
                vol_slider.set(self.model.track_volumes[track_idx])
                vol_slider.grid(row=grid_row, column=1, padx=5, pady=5)

                # Bouton Mute
                mute_color = "red" if self.model.mute_states[track_idx] else "gray40"
                mute_btn = ctk.CTkButton(self.grid_frame, text="M", width=30, height=30, fg_color=mute_color,
                                         command=lambda r=track_idx: self._toggle_mute(r))
                mute_btn.grid(row=grid_row, column=2, padx=2)

                # Bouton Solo
                solo_color = "orange" if self.model.solo_states[track_idx] else "gray40"
                solo_btn = ctk.CTkButton(self.grid_frame, text="S", width=25, height=25, fg_color=solo_color,
                                         command=lambda r=track_idx: self._toggle_solo(r))
                solo_btn.grid(row=grid_row, column=3, padx=1)
                
                # Bouton Mix
                mix_btn = ctk.CTkButton(self.grid_frame, text="MIX", width=35, height=25, fg_color="#555",
                                        font=("Arial", 10),
                                        command=lambda r=track_idx: self._open_track_settings(r))
                mix_btn.grid(row=grid_row, column=4, padx=1)

                for c in range(self.model.num_steps):
                    btn = ctk.CTkButton(self.grid_frame, text="", width=40, height=40, 
                                        fg_color="gray25", hover_color=self.BUTTON_COLORS[color_idx])
                    
                    btn.bind("<Button-1>", lambda e, r=track_idx, c=c: self._on_step_click(e, r, c, velocity=1.0))
                    btn.bind("<Button-3>", lambda e, r=track_idx, c=c: self._on_step_click(e, r, c, velocity=0.6))
                    
                    btn.grid(row=grid_row, column=c+5, padx=2, pady=2)
                    row_buttons.append(btn)
                self.buttons.append(row_buttons)
                
                # Passer à la piste suivante au prochain cycle
                self.after(5, lambda: self._incremental_create_tracks(track_idx + 1, was_playing))
            else:
                # Terminé
                #print("Grille reconstruite. Mise à jour finale des styles...")
                self._update_all_buttons()
                self.model.is_loading = False
                if was_playing:
                    self.audio.start_playback()
                #print("Grille reconstruite avec succès.")
        except Exception as e:
            print(f"Erreur dans la création incrémentale : {e}")
            self.model.is_loading = False

    def _setup_ui(self):
        # Zone Haute (BPM, Kit)
        self.top_frame = ctk.CTkFrame(self)
        self.top_frame.pack(pady=20, padx=20, fill="x")

        # BPM et Kit à GAUCHE
        ctk.CTkLabel(self.top_frame, text="BPM:").pack(side="left", padx=(10, 2))
        self.bpm_slider = ctk.CTkSlider(self.top_frame, from_=60, to=200, width=150, number_of_steps=140, command=self._on_bpm_change)
        self.bpm_slider.set(self.model.bpm)
        self.bpm_slider.pack(side="left", padx=5)
        self.bpm_label = ctk.CTkLabel(self.top_frame, text=str(self.model.bpm), width=30)
        self.bpm_label.pack(side="left", padx=(2, 10))

        ctk.CTkLabel(self.top_frame, text="Kit:").pack(side="left", padx=(10, 2))
        kits = self._discover_kits()
        self.kit_menu = ctk.CTkOptionMenu(self.top_frame, values=kits, width=120, command=self._on_kit_change)
        self.kit_menu.set(self.model.current_kit)
        self.kit_menu.pack(side="left", padx=5)

        ctk.CTkLabel(self.top_frame, text="Measure:").pack(side="left", padx=(15, 2))
        signatures = ["4/4", "3/4", "2/4", "6/8"]
        self.sig_menu = ctk.CTkOptionMenu(self.top_frame, values=signatures, width=80, command=self._on_signature_change)
        self.sig_menu.set(self.model.time_signature)
        self.sig_menu.pack(side="left", padx=5)

        ctk.CTkLabel(self.top_frame, text="Steps:").pack(side="left", padx=(15, 2))
        self.steps_slider = ctk.CTkSlider(self.top_frame, from_=4, to=32, width=100, number_of_steps=28, command=self._on_steps_change)
        self.steps_slider.set(self.model.num_steps)
        self.steps_slider.pack(side="left", padx=5)
        self.steps_label = ctk.CTkLabel(self.top_frame, text=str(self.model.num_steps), width=30)
        self.steps_label.pack(side="left", padx=(2, 10))
        
        # Swing
        ctk.CTkLabel(self.top_frame, text="Swing:").pack(side="left", padx=(10, 2))
        self.swing_slider = ctk.CTkSlider(self.top_frame, from_=0.0, to=0.5, width=100, number_of_steps=50, command=self._on_swing_change)
        self.swing_slider.set(self.model.swing)
        self.swing_slider.pack(side="left", padx=5)
        self.swing_label = ctk.CTkLabel(self.top_frame, text=f"{self.model.swing:.2f}", width=35)
        self.swing_label.pack(side="left", padx=(2, 10))

        # Master Vol et Auto-Trim à DROITE
        ctk.CTkLabel(self.top_frame, text="MASTER VOL:").pack(side="right", padx=(2, 10))
        self.master_vol_slider = ctk.CTkSlider(self.top_frame, from_=0.0, to=1.0, width=100, command=self._on_master_vol_change)
        self.master_vol_slider.set(self.model.master_volume)
        self.master_vol_slider.pack(side="right", padx=5)

        self.auto_trim_var = tk.BooleanVar(value=False)
        self.auto_trim_cb = ctk.CTkCheckBox(self.top_frame, text="Auto-Trim", 
                                            variable=self.auto_trim_var, command=self._on_auto_trim_toggle)
        self.auto_trim_cb.pack(side="right", padx=15)

        # STYLE au MILIEU (ou ce qui reste)
        ctk.CTkLabel(self.top_frame, text="STYLE:").pack(side="left", padx=(30, 2))
        self.style_label = ctk.CTkLabel(self.top_frame, text="Aucun", text_color="white", font=("Arial", 14, "bold"))
        self.style_label.pack(side="left", padx=(2, 10))

        # Zone Centrale (Grille Défilante)
        self.grid_frame = ctk.CTkScrollableFrame(self, orientation="horizontal")
        self.grid_frame.pack(pady=10, padx=20, fill="both", expand=True)
        # Initialisation vide
        self.buttons = []
        self._on_kit_change(self.model.current_kit) # Charger le kit initial proprement

        # Zone Basse (Transport, Fichiers)
        self.bottom_frame = ctk.CTkFrame(self)
        self.bottom_frame.pack(pady=20, padx=20, fill="x")

        # 1. Transport Frame (Gauche)
        transport_frame = ctk.CTkFrame(self.bottom_frame, fg_color="transparent")
        transport_frame.pack(side="left", padx=0)

        self.play_btn = ctk.CTkButton(transport_frame, text="PLAY", command=self._play, fg_color="green", width=60)
        self.play_btn.pack(side="left", padx=5)
        self.stop_btn = ctk.CTkButton(transport_frame, text="STOP", command=self._stop, fg_color="red", width=60)
        self.stop_btn.pack(side="left", padx=5)
        self.clear_btn = ctk.CTkButton(transport_frame, text="CLEAR", command=self._clear, width=60)
        self.clear_btn.pack(side="left", padx=5)
        self.magic_btn = ctk.CTkButton(transport_frame, text="MAGIC", command=self._magic, fg_color="#9932CC", width=60)
        self.magic_btn.pack(side="left", padx=5)

        # 2. Song Mode Frame (Milieu)
        song_frame = ctk.CTkFrame(self.bottom_frame, fg_color="transparent")
        song_frame.pack(side="left", padx=30)
        
        ctk.CTkButton(song_frame, text="+ ADD", command=self._add_to_playlist, width=50, fg_color="#2E8B57").pack(side="left", padx=2)
        ctk.CTkButton(song_frame, text="VIEW", command=self._show_playlist, width=50, fg_color="#4682B4").pack(side="left", padx=2)
        self.playlist_label = ctk.CTkLabel(song_frame, text="Playlist: 0")
        self.playlist_label.pack(side="left", padx=5)
        self.song_mode_switch = ctk.CTkSwitch(song_frame, text="SONG MODE", command=self._toggle_song_mode, width=100)
        self.song_mode_switch.pack(side="left", padx=5)

        # 3. File Frame (Droite)
        file_frame = ctk.CTkFrame(self.bottom_frame, fg_color="transparent")
        file_frame.pack(side="right", padx=0)
        
        # On utilise Grid ou Pack pour file_frame ? Pack side="right" inverse l'ordre visuel.
        # On veut: Save, Load, Preset, Export (de gauche à droite)
        # Donc on packe right: Export, Preset, Load, Save
        
        ctk.CTkButton(file_frame, text="?", command=self._open_help, width=30, fg_color="gray").pack(side="right", padx=5)
        ctk.CTkButton(file_frame, text="EXPORT WAV", command=self._export_wav, fg_color="#E0A020", text_color="black", width=90).pack(side="right", padx=5)
        ctk.CTkButton(file_frame, text="LOAD PRESET", command=self._load_preset, fg_color="blue", width=90).pack(side="right", padx=5)
        ctk.CTkButton(file_frame, text="LOAD", command=self._load, width=60).pack(side="right", padx=5)
        ctk.CTkButton(file_frame, text="SAVE", command=self._save, width=60).pack(side="right", padx=5)

        # Indicateur de pas (visuel)
        self.after(50, self._update_visual_indicator)

    def _discover_kits(self):
        # Adjusted path to look in ../buttons/ (actually ../assets/kits) relative to src/ui.py
        base_path = os.path.join(os.path.dirname(os.path.dirname(os.path.abspath(__file__))), "assets", "kits")
        if not os.path.exists(base_path):
            return ["808_Kit"]
        return sorted([d for d in os.listdir(base_path) if os.path.isdir(os.path.join(base_path, d))])

    def _toggle_mute(self, r):
        self.model.mute_states[r] = not self.model.mute_states[r]
        self._update_track_controls(r)

    def _toggle_solo(self, r):
        self.model.solo_states[r] = not self.model.solo_states[r]
        # Si on active un solo, on doit peut-être mettre à jour l'affichage de TR de tous les s (ou audio gère le silence)
        self._update_track_controls(r)

    def _update_track_controls(self, r):
        # On doit retrouver les widgets M et S. 
        # Structure de grid_frame: 
        # Row 0: Headers
        # Row r+1: [Label Name, Slider Vol, Button M, Button S, Pad1, Pad2...]
        # C'est compliqué de retrouver les widgets par grid_slaves.
        # On va iterer sur les enfants de grid_frame et chercher ceux qui sont à la bonne ligne/colonne.
        
        # Optimisation: Garder une référence aux boutons M/S ?
        # Pour l'instant on fait bourrin:
        for widget in self.grid_frame.winfo_children():
            info = widget.grid_info()
            if int(info['row']) == r + 2: # Offset: Row 0=Chords, Row 1=Headers
                col = int(info['column'])
                if col == 2: # Mute
                    color = "red" if self.model.mute_states[r] else "gray40"
                    widget.configure(fg_color=color)
                elif col == 3: # Solo
                    color = "orange" if self.model.solo_states[r] else "gray40"
                    widget.configure(fg_color=color)

    def _magic(self):
        self.model.randomize_grid()
        self._update_all_buttons()

    def _add_to_playlist(self):
        # Ajouter une COPIE de la grille actuelle + LE NOM
        current_name = self.style_label.cget("text")
        if current_name == "Aucun" or not current_name:
            current_name = f"Pattern {len(self.model.playlist) + 1}"
            
        entry = {
            "name": current_name,
            "grid": copy.deepcopy(self.model.grid)
        }
        self.model.playlist.append(entry)
        count = len(self.model.playlist)
        self.playlist_label.configure(text=f"Playlist: {count}")
        print(f"Pattern ajouté ({current_name}). Total: {count}")

    def _toggle_song_mode(self):
        self.model.song_mode = bool(self.song_mode_switch.get())
        if self.model.song_mode:
            print("Song Mode: ON")
            self.model.song_index = 0 # Reset au début ? Ou on laisse courir ?
        else:
            print("Song Mode: OFF")

    def _show_playlist(self):
        top = ctk.CTkToplevel(self)
        top.title("Playlist")
        top.geometry("300x400")
        
        scroll = ctk.CTkScrollableFrame(top)
        scroll.pack(fill="both", expand=True, padx=10, pady=10)
        
        if not self.model.playlist:
            ctk.CTkLabel(scroll, text="Playlist vide").pack()
        else:
            for i, item in enumerate(self.model.playlist):
                name = item.get("name", f"Pattern {i+1}")
                # Indicateur si joué
                prefix = ">> " if (self.model.song_mode and i == self.model.song_index) else f"{i+1}. "
                lbl = ctk.CTkLabel(scroll, text=f"{prefix}{name}", anchor="w")
                lbl.pack(fill="x", padx=5, pady=2)

    def _check_grid_update(self):
        if self.model.grid_has_changed:
            with self.model.grid_lock:
                self._update_all_buttons()
                self.style_label.configure(text=self.model.current_style_name)
                self.model.grid_has_changed = False
        # Re-schedule
        self.after(100, self._check_grid_update)

    def _show_track_menu(self, event, track_idx):
        # Créer un menu Tkinter standard (ctk n'a pas de context menu natif simple)
        menu = tk.Menu(self, tearoff=0)
        menu.add_command(label="Changer le son...", command=lambda: self._import_sample_for_track(track_idx))
        menu.add_command(label="Renommer...", command=lambda: self._rename_track_action(track_idx))
        menu.add_separator()
        menu.add_command(label="Supprimer la piste", command=lambda: self._remove_track_action(track_idx))
        menu.add_separator()
        menu.add_command(label="Ajouter une piste...", command=self._add_track_action)
        
        menu.tk_popup(event.x_root, event.y_root)

    def _rename_track_action(self, track_idx):
        old_name = self.model.track_names[track_idx]
        new_name = simpledialog.askstring("Renommer", f"Nouveau nom pour '{old_name}':", initialvalue=old_name)
        if new_name and new_name != old_name:
            names = self.audio.rename_track_in_kit(self.model.current_kit, track_idx, new_name)
            self.after(0, lambda: self._finalize_kit_change(names, self.audio.is_playing))

    def _remove_track_action(self, track_idx):
        track_name = self.model.track_names[track_idx]
        if messagebox.askyesno("Supprimer", f"Voulez-vous vraiment supprimer la piste '{track_name}' ?"):
            # 1. Update Model Grid FIRST to avoid shifting issues
            self.model.delete_track_at(track_idx)
            # 2. Update Audio/JSON
            names = self.audio.remove_track_from_kit(self.model.current_kit, track_idx)
            # 3. Reload UI
            self.after(0, lambda: self._finalize_kit_change(names, self.audio.is_playing))

    def _add_track_action(self):
        file_path = filedialog.askopenfilename(filetypes=[("Audio Files", "*.wav")])
        if file_path:
            # Demander le nom
            default_name = os.path.splitext(os.path.basename(file_path))[0].replace("_", " ").title()
            track_name = simpledialog.askstring("Ajouter Piste", "Nom de la piste :", initialvalue=default_name)
            if track_name:
                names = self.audio.add_track_to_kit(self.model.current_kit, file_path, track_name)
                self.after(0, lambda: self._finalize_kit_change(names, self.audio.is_playing))

    def _import_sample_for_track(self, track_idx):
        filename = filedialog.askopenfilename(filetypes=[("Audio Files", "*.wav")])
        if filename:
            success, new_name = self.audio.replace_sample(track_idx, filename)
            if success:
                print(f"Sample remplacé pour piste {track_idx}: {new_name}")
                # Mettre à jour le nom dans le modèle et l'UI
                self.model.track_names[track_idx] = new_name
                
                # Mise à jour UI du label (un peu tricky sans référence directe)
                # On itère pour trouver le label à (row=track_idx+1, col=0)
                found = False
                for widget in self.grid_frame.winfo_children():
                    info = widget.grid_info()
                    if int(info['row']) == track_idx + 1 and int(info['column']) == 0:
                        widget.configure(text=new_name)
                        found = True
                        break
                if not found:
                    print("Warning: Label UI non trouvé pour mise à jour.")
            else:
                print(f"Erreur import: {new_name}")

    def _export_wav(self):
        # Créer le dossier d'export s'il n'existe pas
        export_dir = os.path.join(os.getcwd(), "PyBeat", "wav_exports")
        if not os.path.exists(export_dir):
            os.makedirs(export_dir)
            
        filename = filedialog.asksaveasfilename(defaultextension=".wav", 
                                                initialdir=export_dir,
                                                filetypes=[("WAV files", "*.wav")])
        if filename:
            # Demander le nombre de boucles
            dialog = ctk.CTkInputDialog(text="Nombre de répétitions (boucles) :", title="Export Options")
            loops_str = dialog.get_input()
            
            try:
                loops = int(loops_str) if loops_str else 1
                if loops < 1: loops = 1
            except ValueError:
                loops = 1
                
            self.model.is_loading = True
            # Faire l'export dans un thread pour ne pas geler l'UI si c'est long
            threading.Thread(target=self._async_export, args=(filename, loops), daemon=True).start()

    def _async_export(self, filename, loops):
        try:
            success = self.audio.export_wav(filename, loops=loops)
            msg = "Export réussi !" if success else "Échec de l'export."
            print(msg)
        except Exception as e:
            print(f"Erreur export: {e}")
        finally:
            self.model.is_loading = False

    def _open_track_settings(self, track_idx):
        top = ctk.CTkToplevel(self)
        top.title(f"Mix: {self.model.track_names[track_idx]}")
        top.geometry("250x200")
        
        # Pan
        ctk.CTkLabel(top, text="Panoramique (L - R)").pack(pady=(10, 0))
        pan_slider = ctk.CTkSlider(top, from_=-1.0, to=1.0, number_of_steps=20)
        pan_slider.set(self.model.track_pan[track_idx])
        pan_slider.pack(pady=5)
        # Callback update
        pan_slider.configure(command=lambda val, r=track_idx: self._update_pan(r, val))
        
        # Pitch (Simulé / Placeholder pour l'instant si pas de lib)
        ctk.CTkLabel(top, text="Pitch (Speed)").pack(pady=(10, 0))
        pitch_slider = ctk.CTkSlider(top, from_=0.5, to=2.0, number_of_steps=20)
        pitch_slider.set(self.model.track_pitch[track_idx])
        pitch_slider.pack(pady=5)
        pitch_slider.configure(command=lambda val, r=track_idx: self._update_pitch(r, val))
        
        # Bouton Reset
        def reset_defaults():
            pan_slider.set(0.0)
            pitch_slider.set(1.0)
            self._update_pan(track_idx, 0.0)
            self._update_pitch(track_idx, 1.0)
            
        reset_btn = ctk.CTkButton(top, text="Reset Default", fg_color="gray", command=reset_defaults)
        reset_btn.pack(pady=10)
        
    def _update_pan(self, track_idx, val):
        if track_idx < len(self.model.track_pan):
            self.model.track_pan[track_idx] = float(val)
            
    def _update_pitch(self, track_idx, val):
        if track_idx < len(self.model.track_pitch):
            self.model.track_pitch[track_idx] = float(val)
            # Todo: Apply pitch Logic if possible (requires reload or special playback)

    def _on_step_click(self, event, r, c, velocity=1.0):
        # Check SHIFT modifier for Button-1
        if event.num == 1 and (event.state & 0x0001): # Shift mask is often bit 0 or similar. Safe way: custom arg.
             pass

        # Si on a cliqué Shift (state 1 sur Linux/Mac souvent, 4 sur Windows? C'est complexe portablement)
        # On va assumer : Button-3 (Droit) est explicitement Ghost (0.6)
        if event.state & 1: # Shift pressed
            velocity = 0.6
            
        self.model.toggle_step(r, c, velocity)
        
        # V9: Smart Bass Trigger
        # Si on touche au Kick ou à autre chose, on tente de regenerer la basse
        # Optimisation: vérifier si c'est la piste Kick ? 
        # Pour l'instant on appelle tout le temps, c'est léger.
        if SmartComposer.update_bass_track(self.model):
            # Si la basse a changé, il faut update l'UI de la basse (et potentiellement tout le monde)
            pass
            
        self._update_button_style(r, c)
        
        # Si Bass a changé, on doit update la ligne de basse
        # On peut être bourrin et tout update
        self._update_all_buttons()


    def _toggle_step(self, r, c):
        # Legacy / Direct call
        self.model.toggle_step(r, c, 1.0)
        self._update_button_style(r, c)

    def _update_button_style(self, r, c):
        # Sécurité : vérifier que les boutons existent et que les index sont valides
        if not hasattr(self, 'buttons') or r >= len(self.buttons):
            return
        if c >= len(self.buttons[r]):
            return
            
        color_idx = r % len(self.BUTTON_COLORS)
        val = self.model.grid[r][c]
        
        # Gestion des types (bool old vs float new)
        # Si c'est True (bool), on considère 1.0
        if val is True: val = 1.0
        if val is False: val = 0.0
        
        if val > 0.0:
            base_color = self.BUTTON_COLORS[color_idx]
            # Si Ghost Note (0.6), on assombrit ou change la couleur
            if val < 0.9:
                 # Hack couleur : On peut pas facilement assombrir un string name.
                 # On va utiliser une couleur fixe pour ghost ou une logique.
                 # Simple: Ghost = "gray60" ou Cyan/Olive selon la track ?
                 # Mieux : Transparence alpha pas supporté.
                 # On met une couleur distincte pour Ghost : "OliveDrab" ou similaire
                 self.buttons[r][c].configure(fg_color="SlateBlue" if color_idx % 2 == 0 else "DarkGoldenrod")
            else:
                 self.buttons[r][c].configure(fg_color=base_color)
        else:
            self.buttons[r][c].configure(fg_color="gray25")

    def _update_all_buttons(self):
        if not hasattr(self, 'buttons') or not self.buttons:
            return
            
        for r in range(min(len(self.buttons), self.model.num_tracks)):
            for c in range(min(len(self.buttons[r]), self.model.num_steps)):
                self._update_button_style(r, c)

    def _on_bpm_change(self, val):
        self.model.bpm = int(float(val))
        self.bpm_label.configure(text=str(self.model.bpm))

    def _on_signature_change(self, val):
        self.model.time_signature = val
        # Suggérer un nombre de pas cohérent
        if val == "3/4":
            self._on_steps_change(12)
        elif val == "6/8":
            self._on_steps_change(12)
        elif val == "2/4":
            self._on_steps_change(8)
        else: # 4/4
            self._on_steps_change(16)
        
        # Mettre à jour le slider UI si on l'a changé par code
        self.steps_slider.set(self.model.num_steps)
        self.steps_label.configure(text=str(self.model.num_steps))

    def _on_steps_change(self, val):
        new_steps = int(float(val))
        if new_steps != self.model.num_steps:
            self.model.is_loading = True
            # Arrêter la lecture
            was_playing = self.audio.is_playing
            if was_playing:
                self.audio.stop_playback()
            
            # Changer la taille dans le modèle
            self.model.change_num_steps(new_steps)
            self.steps_label.configure(text=str(new_steps))
            
            # Reconstruire la grille proprement
            self.after(0, lambda: self._finalize_kit_change(self.model.track_names, was_playing))
            
    def _on_swing_change(self, val):
        self.model.swing = float(val)
        self.swing_label.configure(text=f"{self.model.swing:.2f}")

    def _on_master_vol_change(self, val):
        self.model.master_volume = float(val)
        self.audio.update_volumes()

    def _on_track_vol_change(self, track_idx, val):
        if track_idx < len(self.model.track_volumes):
            self.model.track_volumes[track_idx] = float(val)
            self.audio.update_volumes()

    def _on_auto_trim_toggle(self):
        self.model.auto_trim = self.auto_trim_var.get()
        print(f"Auto-Trim Loop: {self.model.auto_trim}")

    def _on_kit_change(self, val):
        self.model.is_loading = True
        self.style_label.configure(text="Aucun") # Reset style name when changing kit manually
        # Arrêter le son pendant le changement pour libérer les ressources
        was_playing = self.audio.is_playing
        if was_playing:
            self.audio.stop_playback()
            
        print(f"Chargement du kit : {val}...")
        threading.Thread(target=self._async_load_kit, args=(val, was_playing), daemon=True).start()

    def _open_bass_options_window(self):
        """Ouvre la fenêtre d'options de la basse."""
        top = ctk.CTkToplevel(self)
        top.title("Options Basse")
        top.geometry("300x400")
        top.attributes("-topmost", True)
        
        # 1. Fill / Active
        fill_var = ctk.BooleanVar(value=self.model.fill_chords)
        
        def on_param_change(*args):
            self.model.fill_chords = fill_var.get()
            self.model.bass_style = style_var.get()
            SmartComposer.update_bass_track(self.model)
            self._update_all_buttons() # Update grid if bass changed
            
        chk = ctk.CTkCheckBox(top, text="Activer Remplissage (Fill)", variable=fill_var, command=on_param_change)
        chk.pack(pady=10, padx=20, anchor="w")
        
        # 2. Styles
        ctk.CTkLabel(top, text="Style de Basse :").pack(pady=(10, 5))
        
        # Load Styles
        styles = ["Arpeggio", "Disco", "Rock", "Funk", "Waltz"] # Defaults
        try:
            # rythmbox/styles.txt
            root_dir = os.path.dirname(os.path.dirname(os.path.dirname(os.path.abspath(__file__))))
            style_path = os.path.join(root_dir, "styles.txt")
            if os.path.exists(style_path):
                with open(style_path, 'r', encoding='utf-8') as f:
                    lines = [l.strip() for l in f.readlines()]
                    # Filter clean styles (starting with - or plain text, ignore headers usually numbered)
                    # User file format: "1. POP..." or "- Pop"
                    # We want only the names.
                    loaded_styles = []
                    for line in lines:
                        if line.startswith("-"):
                             loaded_styles.append(line.replace("-", "").strip())
                    if loaded_styles:
                        styles = ["Arpeggio"] + loaded_styles
        except Exception as e:
            print(f"Erreur chargement styles: {e}")

        current_style = getattr(self.model, 'bass_style', "Arpeggio")
        if current_style not in styles:
            styles.insert(0, current_style)
            
        style_var = ctk.StringVar(value=current_style)
        combo = ctk.CTkComboBox(top, values=styles, variable=style_var, command=on_param_change, width=200)
        combo.pack(pady=10)
        
        ctk.CTkLabel(top, text="Infos :\n'Fill' active le générateur.\nSinon la basse suit uniquement le Kick.", 
                     font=("Arial", 10), text_color="gray").pack(pady=20)
                     
    def _async_load_kit(self, kit_name, was_playing):
        try:
            self.model.current_kit = kit_name
            names = self.audio.update_kit(kit_name)
            #print(f"Kit chargé ({len(names)} pistes). Mise à jour UI...")
            # Revenir sur le thread UI
            self.after(0, lambda: self._finalize_kit_change(names, was_playing))
        except Exception as e:
            print(f"Erreur fatale dans le thread de chargement : {e}")
            self.after(0, lambda: setattr(self.model, 'is_loading', False))


    def _toggle_play(self):
        if self.audio.is_playing:
            self._stop()
        else:
            self._play()

    def _on_chord_click(self, step_idx):
        # Popup pour choisir l'accord (Note + Qualité)
        top = ctk.CTkToplevel(self)
        top.title(f"Accord - Step {step_idx+1}")
        top.geometry("300x400")
        top.attributes("-topmost", True)
        
        notes = ["C", "C#", "D", "D#", "E", "F", "F#", "G", "G#", "A", "A#", "B"]
        
        # Variable pour la qualité (Maj par défaut)
        # On essaie de récupérer la qualité actuelle si possible
        current_val = self.model.chords[step_idx]
        start_quality = "Maj"
        if isinstance(current_val, tuple) and len(current_val) > 1:
            start_quality = current_val[1]
            
        quality_var = ctk.StringVar(value=start_quality)

        def set_chord(note):
            qual = quality_var.get()
            self.model.chords[step_idx] = (note, qual)
            
            # Update Button Text Immediately (Helper)
            if step_idx < len(self.chord_buttons):
                txt = note + ("m" if qual == "Min" else "")
                self.chord_buttons[step_idx].configure(text=txt)
            
            # Trigger Smart Bass
            SmartComposer.update_bass_track(self.model)
            self._update_all_buttons()
            
            top.destroy()
            
        def clear_chord():
            self.model.chords[step_idx] = None
            if step_idx < len(self.chord_buttons):
                self.chord_buttons[step_idx].configure(text="-")
            
            SmartComposer.update_bass_track(self.model)
            self._update_all_buttons()
            top.destroy()
            
        # Selecteur Qualité
        ctk.CTkLabel(top, text="Qualité :").pack(pady=(10, 0))
        q_frame = ctk.CTkFrame(top, fg_color="transparent")
        q_frame.pack(pady=5)
        ctk.CTkRadioButton(q_frame, text="Majeur", variable=quality_var, value="Maj").pack(side="left", padx=10)
        ctk.CTkRadioButton(q_frame, text="Mineur", variable=quality_var, value="Min").pack(side="left", padx=10)

        ctk.CTkLabel(top, text="Fondamentale :").pack(pady=10)
        
        grid = ctk.CTkFrame(top, fg_color="transparent")
        grid.pack(pady=5)
        
        for i, n in enumerate(notes):
            r = i // 4
            c = i % 4
            # On passe la note à la lambda
            btn = ctk.CTkButton(grid, text=n, width=40, height=30, 
                                command=lambda x=n: set_chord(x))
            btn.grid(row=r, column=c, padx=3, pady=3)
            
        ctk.CTkButton(top, text="Effacer (Clear)", fg_color="red", command=clear_chord).pack(pady=20)

    def _update_chord_ui(self):
        """Met à jour l'affichage des boutons d'accords."""
        for i, btn in enumerate(self.chord_buttons):
            if i < len(self.model.chords):
                val = self.model.chords[i]
                txt = "-"
                if val:
                    if isinstance(val, tuple): 
                        # Format "Cm" for Minor, "C" for Major
                        note = val[0]
                        qual = val[1] if len(val) > 1 else "Maj"
                        txt = note + ("m" if qual == "Min" else "")
                    else: 
                        txt = str(val)
                btn.configure(text=txt)

    def _play(self):
        self.audio.start_playback()

    def _stop(self):
        self.audio.stop_playback()

    def _clear(self):
        self.model.clear_grid()
        self._update_all_buttons()

    def _save(self):
        # Utiliser les dossiers relatifs au CWD (rythmbox/)
        initial_dir = os.path.join(os.getcwd(), "PyBeat", "saves")
        if not os.path.exists(initial_dir):
            os.makedirs(initial_dir)
            
        file_path = filedialog.asksaveasfilename(defaultextension=".json", initialdir=initial_dir)
        if file_path:
            data = {
                "bpm": self.model.bpm,
                "kit": self.model.current_kit,
                "auto_trim": self.model.auto_trim,
                "master_volume": self.model.master_volume,
                "track_volumes": self.model.track_volumes,
                "time_signature": self.model.time_signature,
                "num_steps": self.model.num_steps,
                "grid": self.model.grid,
                "playlist": self.model.playlist,
                "track_pan": self.model.track_pan,
                "track_pitch": self.model.track_pitch,
                "chords": self.model.chords,
                "grid_pitch": self.model.grid_pitch,
                "bass_style": getattr(self.model, 'bass_style', "Arpeggio"),
                "fill_chords": getattr(self.model, 'fill_chords', True)
            }
            with open(file_path, 'w') as f:
                json.dump(data, f)

    def _load(self):
        initial_dir = os.path.join(os.getcwd(), "PyBeat", "saves")
        if not os.path.exists(initial_dir):
            os.makedirs(initial_dir)

        file_path = filedialog.askopenfilename(defaultextension=".json", initialdir=initial_dir)
        if file_path:
            self._process_load(file_path)

    def _load_preset(self):
        initial_dir = os.path.join(os.getcwd(), "PyBeat", "presets")
        if not os.path.exists(initial_dir):
            os.makedirs(initial_dir)
            
        file_path = filedialog.askopenfilename(defaultextension=".json", initialdir=initial_dir)
        if file_path:
            # Pour les presets, on conserve la playlist actuelle pour permettre le chaînage !
            self._process_load(file_path, keep_playlist=True)

    def _open_help(self):
        help_path = os.path.join(os.getcwd(), "PyBeat", "aide.html")
        if os.path.exists(help_path):
            webbrowser.open("file://" + help_path)
        else:
            print("Fichier d'aide introuvable :", help_path)

    def _process_load(self, file_path, keep_playlist=False):
        style_name = os.path.splitext(os.path.basename(file_path))[0].title()
        with open(file_path, 'r') as f:
            data = json.load(f)
            
            # Mise à jour du BPM
            self.model.bpm = data["bpm"]
            self.bpm_slider.set(self.model.bpm)
            self.bpm_label.configure(text=str(self.model.bpm))
            
            # Mise à jour de l'Auto-Trim
            self.model.auto_trim = data.get("auto_trim", False)
            self.auto_trim_var.set(self.model.auto_trim)

            # Mise à jour de la signature et du nombre de pas
            self.model.time_signature = data.get("time_signature", "4/4")
            self.sig_menu.set(self.model.time_signature)
            
            # Application du nombre de pas chargé
            new_steps = data.get("num_steps", 16)
            self.model.change_num_steps(new_steps)
            self.steps_slider.set(new_steps)
            self.steps_label.configure(text=str(new_steps))

            # Re-scanner le kit actuel pour voir s'il y a de nouvelles pistes
            print(f"Re-scan du kit '{self.model.current_kit}' pour nouvelles pistes...")
            names = self.audio.update_kit(self.model.current_kit)
            
            # Charger la grille en l'adaptant au kit
            self._finalize_kit_change(names, False, new_grid=data["grid"])

            # Mise à jour des volumes (Après adaptation du kit pour éviter l'écrasement)
            # Mise à jour des volumes
            self.model.master_volume = data.get("master_volume", 1.0)
            self.master_vol_slider.set(self.model.master_volume)
            
            loaded_track_vols = data.get("track_volumes", [])
            loaded_track_pan = data.get("track_pan", [])
            loaded_track_pitch = data.get("track_pitch", [])
            
            if loaded_track_vols:
                # Appliquer les volumes chargés aux pistes disponibles (en respectant les limites)
                for i in range(min(len(loaded_track_vols), self.model.num_tracks)):
                    self.model.track_volumes[i] = loaded_track_vols[i]
                    
            # Restaurer Pan et Pitch si disponibles
            if loaded_track_pan:
                for i in range(min(len(loaded_track_pan), self.model.num_tracks)):
                    self.model.track_pan[i] = loaded_track_pan[i]
            else:
                self.model.track_pan = [0.0] * self.model.num_tracks
                
            if loaded_track_pitch:
                for i in range(min(len(loaded_track_pitch), self.model.num_tracks)):
                    self.model.track_pitch[i] = loaded_track_pitch[i]
            else:
                self.model.track_pitch = [1.0] * self.model.num_tracks

            # V9: Chords & Grid Pitch
            raw_chords = data.get("chords", [None] * self.model.num_steps)
            # Sanitize: JSON converts tuples to lists. We need tuples (or strings/None).
            self.model.chords = [tuple(c) if isinstance(c, list) else c for c in raw_chords]
            self.model.grid_pitch = data.get("grid_pitch", [[0.0 for _ in range(self.model.num_steps)] for _ in range(self.model.num_tracks)])
            
            self.model.bass_style = data.get("bass_style", "Arpeggio")
            self.model.fill_chords = data.get("fill_chords", True)
            
            self.audio.update_volumes()
            
            # Force UI Refresh (Async to wait for _incremental_create)
            self.after(200, self._update_chord_ui)
            
            print(f"Loaded successfully. {len(self.model.chords)} steps.")
            
            # Mise à jour Playlist (si on ne demande pas de la conserver pour chaînage)
            if not keep_playlist:
                # Retro-compatibilité logic
                raw_playlist = data.get("playlist", [])
                self.model.playlist = []
                for i, item in enumerate(raw_playlist):
                    if isinstance(item, dict):
                        self.model.playlist.append(item)
                    else:
                        self.model.playlist.append({
                            "name": f"Pattern {i+1}",
                            "grid": item
                        })
                
                # Si une playlist LOADÉE existe, on auto-active le song mode
                if self.model.playlist:
                    self.model.song_mode = True
                    self.model.song_index = 0
                    self.song_mode_switch.select()
                    print("Playlist chargée : Mode Song activé automatiquement.")
                else:
                    self.model.song_mode = False
                    self.song_mode_switch.deselect()
            
            self.playlist_label.configure(text=f"Playlist: {len(self.model.playlist)}")
            
            # Mise à jour du nom du style à l'écran
            self.style_label.configure(text=style_name)
            
            print(f"Fichier '{style_name}' chargé avec succès.")

    def _scroll_to_step(self, step_idx):
        """Fait défiler la grille pour que le pas actuel soit visible."""
        if not hasattr(self, 'grid_frame') or not isinstance(self.grid_frame, ctk.CTkScrollableFrame):
            return
            
        try:
            # On récupère le canvas interne de CustomTkinter
            canvas = self.grid_frame._parent_canvas
            
            # Nombre total de pas
            total_steps = self.model.num_steps
            if total_steps <= 1:
                return

            # Calcul approximatif de la position
            # On veut que le pas soit visible. Si on est au début, on reste à 0.
            # Si on dépasse la moitié de l'écran (environ 12-14 pas visibles), on scrolle.
            
            # Fraction de défilement (0.0 à 1.0)
            # On décale un peu pour ne pas coller au bord
            fraction = step_idx / total_steps
            
            # On ne scrolle que si nécessaire (ex: plus de 16 pas)
            if total_steps > 16:
                # Ajustement empirique pour centrer à peu près
                target = max(0, fraction - 0.2)
                canvas.xview_moveto(target)
        except Exception as e:
            # On échoue silencieusement si CustomTkinter change sa structure interne
            pass

    def _load_window_config(self):
        """Charge la position et la taille de la fenêtre depuis le fichier config."""
        try:
            if os.path.exists(self.CONFIG_FILE):
                with open(self.CONFIG_FILE, 'r') as f:
                    config = json.load(f)
                    geometry = config.get("geometry")
                    if geometry:
                        self.geometry(geometry)
                    else:
                        self.geometry("1000x600")
            else:
                self.geometry("1000x600")
        except Exception as e:
            print(f"Erreur lors du chargement de la config : {e}")
            self.geometry("1000x600")

    def _save_window_config(self):
        """Sauvegarde la position et la taille actuelle de la fenêtre."""
        try:
            config = {"geometry": self.geometry()}
            with open(self.CONFIG_FILE, 'w') as f:
                json.dump(config, f)
        except Exception as e:
            print(f"Erreur lors de la sauvegarde de la config : {e}")

    def _on_closing(self):
        """Gère la fermeture propre de l'application."""
        self._save_window_config()
        self.destroy()
        sys.exit()

    def _update_visual_indicator(self):
        # Ne rien faire si on est en cours de chargement
        if self.model.is_loading:
            self.after(100, self._update_visual_indicator)
            return
            
        # Mettre en évidence la colonne actuelle
        try:
            current = self.audio.current_step
            
            # Auto-scroll si on joue
            if self.audio.is_playing:
                self._scroll_to_step(current)

            # Mise à jour optimisée : seulement le pas actuel et le précédent
            if hasattr(self, 'buttons') and len(self.buttons) == self.model.num_tracks:
                # 1. Restaurer le pas précédent
                if self.last_playing_step != -1 and self.last_playing_step != current:
                    for r in range(self.model.num_tracks):
                        if r < len(self.buttons) and self.last_playing_step < len(self.buttons[r]):
                            color_idx = r % len(self.BUTTON_COLORS)
                            # Logic identique à _update_button_style
                            val = self.model.grid[r][self.last_playing_step]
                            if val is True: val = 1.0
                            if val is False: val = 0.0
                            
                            orig_color = "gray25"
                            if val > 0.0:
                                if val < 0.9: # Ghost
                                    orig_color = "SlateBlue" if color_idx % 2 == 0 else "DarkGoldenrod"
                                else:
                                    orig_color = self.BUTTON_COLORS[color_idx]
                                    
                            self.buttons[r][self.last_playing_step].configure(border_width=0, fg_color=orig_color)
                
                # 2. Mettre en évidence le pas actuel
                for r in range(self.model.num_tracks):
                    if r < len(self.buttons) and current < len(self.buttons[r]):
                        # Bordure blanche pour tout le monde au pas actuel
                        self.buttons[r][current].configure(border_width=2, border_color="white")
                        
                        # Changement de couleur de fond (Highlight)
                        # Changement de couleur de fond (Highlight)
                        if self.audio.is_playing:
                            val = self.model.grid[r][current]
                            # Cast bool -> float legacy
                            if val is True: val = 1.0
                            if val is False: val = 0.0
                            
                            if val > 0.0:
                                # Note active : Blanc ou couleur très vive
                                self.buttons[r][current].configure(fg_color="white")
                            else:
                                # Vide actif : Gris un peu plus clair
                                self.buttons[r][current].configure(fg_color="gray40")
                        else:
                            # Si on ne joue pas, on remet les couleurs normales
                            color_idx = r % len(self.BUTTON_COLORS)
                            orig_color = self.BUTTON_COLORS[color_idx] if self.model.grid[r][current] else "gray25"
                            self.buttons[r][current].configure(fg_color=orig_color, border_width=0)

                self.last_playing_step = current
        except Exception:
            pass
            
        self.after(50, self._update_visual_indicator)
