#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import tkinter as tk
from tkinter import ttk, messagebox, filedialog, simpledialog
import subprocess
import os
import sys
import json

CONFIG_FILE = os.path.expanduser("~/.samba_gui_config.json")

COLORS = {
    'bg': '#121212',         # Fond plus sombre (Material Dark)
    'fg': '#e0e0e0',         # Texte clair
    'accent': '#1f1f1f',     # En-têtes et zones secondaires
    'btn': '#333333',        # Boutons standard
    'btn_hover': '#444444',  # Survol boutons
    'add_btn': '#2e7d32',    # Bouton ajout
    'add_hover': '#388e3c',
    'del_btn': '#c62828',    # Bouton suppression
    'del_hover': '#d32f2f',
    'restore_btn': '#455a64', # Bouton restauration
    'restore_hover': '#546e7a',
    'entry_bg': '#1e1e1e',
    'border': '#333333'
}

class PasswordDialog(simpledialog.Dialog):
    def body(self, master):
        self.configure(bg=COLORS['bg'])
        tk.Label(master, text="Mot de passe Sudo requis :", bg=COLORS['bg'], fg=COLORS['fg'], font=("Inter", 10)).pack(pady=10)
        self.password_entry = tk.Entry(master, show="*", bg=COLORS['entry_bg'], fg=COLORS['fg'], 
                                       insertbackground='white', relief='flat', highlightthickness=1, 
                                       highlightbackground=COLORS['border'], highlightcolor="#4a90e2")
        self.password_entry.pack(pady=5, padx=30, ipady=3)
        return self.password_entry

    def apply(self):
        self.result = self.password_entry.get()

class SambaGUI:
    def __init__(self, root):
        self.root = root
        self.root.title("Samba Manager")
        self.root.configure(bg=COLORS['bg'])
        
        # Charger la configuration (géométrie)
        self.load_config()

        self.style = ttk.Style()
        self.style.theme_use('clam')
        
        # Personnalisation des polices
        self.font_main = ("Inter", 10)
        self.font_bold = ("Inter", 11, "bold")
        self.font_title = ("Inter", 16, "bold")

        self.setup_styles()
        self.setup_ui()
        self.refresh_list()
        
        # Sauvegarder la config à la fermeture
        self.root.protocol("WM_DELETE_WINDOW", self.on_close)

    def load_config(self):
        default_geom = "700x650"
        if os.path.exists(CONFIG_FILE):
            try:
                with open(CONFIG_FILE, 'r') as f:
                    config = json.load(f)
                    self.root.geometry(config.get("geometry", default_geom))
            except:
                self.root.geometry(default_geom)
        else:
            self.root.geometry(default_geom)

    def save_config(self):
        config = {"geometry": self.root.winfo_geometry()}
        try:
            with open(CONFIG_FILE, 'w') as f:
                json.dump(config, f)
        except:
            pass

    def setup_styles(self):
        self.style.configure("Treeview", 
                             background=COLORS['entry_bg'], 
                             foreground=COLORS['fg'], 
                             fieldbackground=COLORS['entry_bg'], 
                             borderwidth=0, 
                             font=self.font_main,
                             rowheight=30)
        self.style.map("Treeview", background=[('selected', '#2c3e50')])
        self.style.configure("Treeview.Heading", 
                             background=COLORS['accent'], 
                             foreground=COLORS['fg'], 
                             borderwidth=0, 
                             font=self.font_bold,
                             padding=5)
        self.style.configure("TScrollbar", gripcount=0, background=COLORS['btn'], darkcolor=COLORS['bg'], lightcolor=COLORS['bg'], bordercolor=COLORS['bg'], troughcolor=COLORS['bg'])

    def setup_ui(self):
        # Header avec icône
        header_frame = tk.Frame(self.root, bg=COLORS['bg'], pady=20)
        header_frame.pack(fill=tk.X)
        tk.Label(header_frame, text="📁 Gestionnaire de Partages Samba", font=self.font_title, bg=COLORS['bg'], fg=COLORS['fg']).pack()

        # Liste des partages
        list_frame = tk.Frame(self.root, bg=COLORS['bg'])
        list_frame.pack(fill=tk.BOTH, expand=True, padx=25)

        self.tree = ttk.Treeview(list_frame, columns=("Name", "Path", "Access"), show='headings', height=10)
        self.tree.heading("Name", text="Nom")
        self.tree.heading("Path", text="Chemin Local")
        self.tree.heading("Access", text="Accès")
        self.tree.column("Name", width=120, anchor='w')
        self.tree.column("Path", width=350, anchor='w')
        self.tree.column("Access", width=100, anchor='center')
        self.tree.pack(side=tk.LEFT, fill=tk.BOTH, expand=True)

        scrollbar = ttk.Scrollbar(list_frame, orient=tk.VERTICAL, command=self.tree.yview)
        self.tree.configure(yscroll=scrollbar.set)
        scrollbar.pack(side=tk.RIGHT, fill=tk.Y)

        # Formulaire plus aéré
        form_wrapper = tk.Frame(self.root, bg=COLORS['bg'], pady=20)
        form_wrapper.pack(fill=tk.X, padx=25)
        
        form_frame = tk.Frame(form_wrapper, bg=COLORS['accent'], padx=15, pady=15, highlightthickness=1, highlightbackground=COLORS['border'])
        form_frame.pack(fill=tk.X)

        tk.Label(form_frame, text="Nouveau Partage", font=self.font_bold, bg=COLORS['accent'], fg=COLORS['fg']).grid(row=0, column=0, columnspan=3, sticky='w', pady=(0, 10))

        tk.Label(form_frame, text="Nom du partage", bg=COLORS['accent'], fg=COLORS['fg'], font=self.font_main).grid(row=1, column=0, sticky='w')
        self.name_entry = tk.Entry(form_frame, bg=COLORS['entry_bg'], fg=COLORS['fg'], insertbackground='white', borderwidth=0, font=self.font_main, highlightthickness=1, highlightbackground=COLORS['border'])
        self.name_entry.grid(row=2, column=0, columnspan=3, sticky='ew', pady=(0, 10), ipady=4)

        tk.Label(form_frame, text="Chemin du dossier", bg=COLORS['accent'], fg=COLORS['fg'], font=self.font_main).grid(row=3, column=0, sticky='w')
        self.path_entry = tk.Entry(form_frame, bg=COLORS['entry_bg'], fg=COLORS['fg'], insertbackground='white', borderwidth=0, font=self.font_main, highlightthickness=1, highlightbackground=COLORS['border'])
        self.path_entry.grid(row=4, column=0, columnspan=2, sticky='ew', ipady=4)
        
        self.btn_browse = tk.Button(form_frame, text="📁 Parcourir", command=self.browse_path, bg=COLORS['btn'], fg=COLORS['fg'], relief='flat', padx=10, cursor="hand2")
        self.btn_browse.grid(row=4, column=2, padx=(10, 0))
        self.add_hover_effect(self.btn_browse, COLORS['btn'], COLORS['btn_hover'])

        form_frame.columnconfigure(1, weight=1)

        # Boutons d'action
        btn_container = tk.Frame(self.root, bg=COLORS['bg'], pady=10)
        btn_container.pack(fill=tk.X, padx=25)

        self.btn_add = tk.Button(btn_container, text="➕ Ajouter le partage", command=self.add_share, bg=COLORS['add_btn'], fg="white", relief='flat', padx=15, pady=8, font=self.font_bold, cursor="hand2")
        self.btn_add.pack(side=tk.LEFT)
        self.add_hover_effect(self.btn_add, COLORS['add_btn'], COLORS['add_hover'])

        self.btn_restore = tk.Button(btn_container, text="🔄 Restaurer Backup", command=self.restore_share, bg=COLORS['restore_btn'], fg="white", relief='flat', padx=15, pady=8, font=self.font_bold, cursor="hand2")
        self.btn_restore.pack(side=tk.LEFT, padx=10)
        self.add_hover_effect(self.btn_restore, COLORS['restore_btn'], COLORS['restore_hover'])

        self.btn_del = tk.Button(btn_container, text="🗑️ Supprimer la sélection", command=self.del_share, bg=COLORS['del_btn'], fg="white", relief='flat', padx=15, pady=8, font=self.font_bold, cursor="hand2")
        self.btn_del.pack(side=tk.RIGHT)
        self.add_hover_effect(self.btn_del, COLORS['del_btn'], COLORS['del_hover'])

        # Barre de statut
        self.status_var = tk.StringVar(value="Prêt")
        self.status_bar = tk.Label(self.root, textvariable=self.status_var, bd=1, relief=tk.SUNKEN, anchor=tk.W, bg=COLORS['accent'], fg="#888888", font=("Inter", 9))
        self.status_bar.pack(side=tk.BOTTOM, fill=tk.X)

    def add_hover_effect(self, widget, normal_bg, hover_bg):
        widget.bind("<Enter>", lambda e: widget.config(bg=hover_bg))
        widget.bind("<Leave>", lambda e: widget.config(bg=normal_bg))

    def set_status(self, text, color="#888888"):
        self.status_var.set(text)
        self.status_bar.config(fg=color)

    def refresh_list(self):
        for item in self.tree.get_children():
            self.tree.delete(item)
        try:
            result = subprocess.run(['python3', '/home/pi/env_gravity/samba/samba_tool.py', '-l'], capture_output=True, text=True)
            content = result.stdout
            current_share, path, access = None, "", ""
            for line in content.splitlines():
                if "Partage : [" in line:
                    current_share = line.split("[")[1].split("]")[0]
                elif "Chemin" in line:
                    path = line.split(":")[1].strip()
                elif "Accès" in line:
                    access = line.split(":")[1].strip()
                    if current_share:
                        self.tree.insert("", tk.END, iid=current_share, values=(current_share, path, access))
                        current_share = None
            self.set_status(f"Liste mise à jour : {len(self.tree.get_children())} partage(s) trouvé(s)")
        except Exception as e:
            self.set_status(f"Erreur de lecture : {e}", COLORS['del_btn'])

    def browse_path(self):
        directory = filedialog.askdirectory()
        if directory:
            self.path_entry.delete(0, tk.END)
            self.path_entry.insert(0, directory)

    def ask_sudo_password(self):
        d = PasswordDialog(self.root, "Configuration Samba")
        return d.result

    def run_with_sudo(self, cmd_args, password):
        full_cmd = ['sudo', '-S'] + cmd_args
        process = subprocess.Popen(full_cmd, stdin=subprocess.PIPE, stdout=subprocess.PIPE, stderr=subprocess.PIPE, text=True)
        stdout, stderr = process.communicate(input=password + '\n')
        return process.returncode, stdout, stderr

    def add_share(self):
        name = self.name_entry.get().strip()
        path = self.path_entry.get().strip()
        if not name or not path:
            messagebox.showwarning("Incomplet", "Nom et chemin requis.")
            return
        password = self.ask_sudo_password()
        if not password: return
        self.set_status("Ajout en cours...")
        code, out, err = self.run_with_sudo(['python3', '/home/pi/env_gravity/samba/samba_tool.py', '-a', name, '--path', path], password)
        if code == 0:
            self.set_status(f"Succès : Partage [{name}] ajouté", "#4caf50")
            self.name_entry.delete(0, tk.END)
            self.path_entry.delete(0, tk.END)
            self.refresh_list()
        else:
            self.set_status("Erreur lors de l'ajout", COLORS['del_btn'])
            messagebox.showerror("Erreur", f"Échec : {err or 'Vérifiez votre mot de passe.'}")

    def del_share(self):
        selected = self.tree.selection()
        if not selected:
            messagebox.showwarning("Sélection requise", "Choisissez un partage dans la liste.")
            return
        name = selected[0]
        if not messagebox.askyesno("Confirmation", f"Supprimer définitivement [{name}] ?"): return
        password = self.ask_sudo_password()
        if not password: return
        self.set_status(f"Suppression de [{name}]...")
        code, out, err = self.run_with_sudo(['python3', '/home/pi/env_gravity/samba/samba_tool.py', '-d', name], password)
        if code == 0:
            self.set_status(f"Succès : Partage [{name}] supprimé", "#4caf50")
            self.refresh_list()
        else:
            self.set_status("Erreur lors de la suppression", COLORS['del_btn'])
            messagebox.showerror("Erreur", f"Échec : {err or 'Vérifiez votre mot de passe.'}")

    def restore_share(self):
        if not messagebox.askyesno("Restauration", "Voulez-vous restaurer la configuration depuis la dernière sauvegarde automatique ?"):
            return
        password = self.ask_sudo_password()
        if not password: return
        self.set_status("Restauration en cours...")
        code, out, err = self.run_with_sudo(['python3', '/home/pi/env_gravity/samba/samba_tool.py', '--restore'], password)
        if code == 0:
            self.set_status("Succès : Configuration restaurée", "#4caf50")
            self.refresh_list()
        else:
            self.set_status("Erreur lors de la restauration", COLORS['del_btn'])
            messagebox.showerror("Erreur", f"Échec : {err or 'Aucune sauvegarde trouvée.'}")

    def on_close(self):
        self.save_config()
        self.root.destroy()

if __name__ == "__main__":
    root = tk.Tk()
    app = SambaGUI(root)
    root.mainloop()
