#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import argparse
import subprocess
import sys
import re
import os

SMB_CONF = "/etc/samba/smb.conf"
SMB_BAK = "/etc/samba/smb.conf.bak"

def get_current_user():
    """Détecte l'utilisateur qui a lancé la commande (même via sudo)."""
    user = os.environ.get('SUDO_USER') or os.environ.get('USER') or os.getlogin()
    return user

def backup_config():
    """Crée une sauvegarde de la configuration actuelle."""
    if os.path.exists(SMB_CONF):
        try:
            subprocess.run(['sudo', 'cp', SMB_CONF, SMB_BAK], check=True)
            return True
        except:
            return False
    return False

def restore_config():
    """Restaure la configuration depuis la dernière sauvegarde."""
    if os.path.exists(SMB_BAK):
        try:
            subprocess.run(['sudo', 'cp', SMB_BAK, SMB_CONF], check=True)
            reload_samba()
            print("Configuration restaurée avec succès depuis la sauvegarde.")
            return True
        except subprocess.CalledProcessError as e:
            print(f"Erreur lors de la restauration : {e}")
    else:
        print("Erreur : Aucune sauvegarde trouvée.")
    return False

def reload_samba():
    """Recharge la configuration de Samba."""
    try:
        subprocess.run(['sudo', 'systemctl', 'reload', 'smbd'], check=True)
        print("Samba a été rechargé avec succès.")
    except subprocess.CalledProcessError as e:
        print(f"Erreur lors du rechargement de Samba : {e}", file=sys.stderr)

def parse_testparm_output():
    """Exécute testparm et parse la sortie pour extraire les détails des partages."""
    try:
        result = subprocess.run(['testparm', '-s'], capture_output=True, text=True, check=True)
        content = result.stdout if "[" in result.stdout else result.stderr
        
        sections = {}
        current_section = None
        
        for line in content.splitlines():
            line = line.strip()
            if not line or line.startswith('#'):
                continue
            
            match_section = re.match(r'^\[(.+)\]$', line)
            if match_section:
                current_section = match_section.group(1)
                sections[current_section] = {}
                continue
            
            if current_section and '=' in line:
                key, value = line.split('=', 1)
                sections[current_section][key.strip()] = value.strip()
                
        return sections
            
    except subprocess.CalledProcessError:
        return None
    except FileNotFoundError:
        return None

def lister_partages_details():
    """Affiche la liste des partages avec leurs détails."""
    data = parse_testparm_output()
    if not data:
        print("Erreur : Impossible de lire la configuration Samba.")
        return

    system_sections = ['global', 'homes', 'printers', 'print$']
    shares = {k: v for k, v in data.items() if k not in system_sections}

    if not shares:
        print("Aucun partage spécifique trouvé.")
        return

    print("Détails des partages Samba détectés :")
    print("=" * 40)
    for name, params in shares.items():
        print(f"Partage : [{name}]")
        print(f"  - Chemin         : {params.get('path', 'Non défini')}")
        print(f"  - Accès          : {'Lecture seule' if params.get('read only', 'Yes').lower() == 'yes' else 'Lecture/Écriture'}")
        print(f"  - Authentification: {'Invités autorisés' if params.get('guest ok', 'No').lower() == 'yes' else 'Authentification requise'}")
        print("-" * 20)

def supprimer_partage(nom):
    """Supprime un partage du fichier smb.conf."""
    if not os.path.exists(SMB_CONF):
        print(f"Erreur : {SMB_CONF} introuvable.")
        return

    # Créer un backup avant modification
    backup_config()

    try:
        with open(SMB_CONF, 'r') as f:
            lines = f.readlines()

        new_lines = []
        skip = False
        found = False
        
        for line in lines:
            if line.strip() == f"[{nom}]":
                skip = True
                found = True
                continue
            if skip and line.strip().startswith("["):
                skip = False
            
            if not skip:
                new_lines.append(line)

        if not found:
            print(f"Le partage [{nom}] n'a pas été trouvé.")
            return

        content = "".join(new_lines)
        process = subprocess.Popen(['sudo', 'tee', SMB_CONF], stdin=subprocess.PIPE, stdout=subprocess.DEVNULL)
        process.communicate(input=content.encode())
        
        print(f"Partage [{nom}] supprimé avec succès.")
        reload_samba()
        
    except Exception as e:
        print(f"Erreur lors de la suppression : {e}")

def ajouter_partage(nom, path):
    """Ajoute un nouveau partage au fichier smb.conf."""
    if not os.path.exists(path):
        print(f"Avertissement : Le chemin {path} n'existe pas localement.")

    data = parse_testparm_output()
    if data and nom in data:
        print(f"Erreur : Le partage [{nom}] existe déjà.")
        return

    # Créer un backup avant modification
    backup_config()

    user = get_current_user()
    config_block = f"\n[{nom}]\n   path = {path}\n   read only = no\n   guest ok = yes\n   force user = {user}\n   create mask = 0777\n   directory mask = 0777\n"
    
    try:
        process = subprocess.Popen(['sudo', 'tee', '-a', SMB_CONF], stdin=subprocess.PIPE, stdout=subprocess.DEVNULL)
        process.communicate(input=config_block.encode())
        print(f"Partage [{nom}] ajouté avec succès pour l'utilisateur {user}.")
        reload_samba()
    except Exception as e:
        print(f"Erreur lors de l'ajout : {e}")

def main():
    parser = argparse.ArgumentParser(description="Outil de gestion Samba.")
    parser.add_argument('-l', action='store_true', help="Liste les partages détaillés")
    parser.add_argument('-a', '--add', metavar='NOM', help="Ajoute un partage (nécessite --path)")
    parser.add_argument('-d', '--delete', metavar='NOM', help="Supprime un partage")
    parser.add_argument('--path', metavar='CHEMIN', help="Chemin pour le nouveau partage")
    parser.add_argument('--restore', action='store_true', help="Restaure la configuration depuis la sauvegarde")

    if len(sys.argv) == 1:
        parser.print_help()
        sys.exit(0)

    args = parser.parse_args()

    if args.restore:
        restore_config()
    elif args.l:
        lister_partages_details()
    elif args.add:
        if not args.path:
            print("Erreur : L'option --path est requise pour ajouter un partage.")
        else:
            ajouter_partage(args.add, args.path)
    elif args.delete:
        supprimer_partage(args.delete)

if __name__ == "__main__":
    main()
