from direct.showbase.ShowBase import ShowBase
from panda3d.core import WindowProperties, AmbientLight, DirectionalLight, Vec4
import sys
import os
import faulthandler
faulthandler.enable()

# Ajout du chemin src pour les imports
base_dir = os.path.dirname(__file__)
sys.path.append(os.path.join(base_dir, "src"))

from engine.camera import FlightCamera
from engine.scene import SceneManager
from engine.buildings import BuildingManager
from ui.overlay import InfoOverlay
from utils.config_manager import load_config, save_config
from ui.launcher import run_launcher

class AeroSim(ShowBase):
    def __init__(self, config):
        ShowBase.__init__(self)
        self.app_config = config
        
        # Configuration de la fenêtre
        properties = WindowProperties()
        properties.setTitle("Simulateur de Vue Aérienne 3D")
        properties.setSize(config["win_width"], config["win_height"])
        properties.setOrigin(config["win_x"], config["win_y"])
        self.win.requestProperties(properties)
        
        # Désactiver le contrôle de caméra par défaut
        self.disableMouse()
        
        # Configuration de l'éclairage
        self.setup_lighting()
        
        # Configuration de la caméra
        self.camLens.setFar(100000.0) # 100 km
        self.setBackgroundColor(0.5, 0.7, 1.0) # Bleu ciel
        
        # CORRECTION MIROIR : S'assurer que les axes ne sont pas inversés
        # Si le texte est à l'envers, on peut tenter de flipper aspect2d
        # self.aspect2d.setScale(1, 1, 1) # Valeur par défaut
        # Dans certains cas de driver, Panda3D peut inverser l'axe X.
        # On va ajouter un raccourci clavier 'f' pour "Flip" si besoin.
        
        # Initialisation de la caméra de vol
        self.flight_cam = FlightCamera(self, start_lat=config["lat"], start_lon=config["lon"])
        self.flight_cam.alt = config["alt"]
        self.camera.setPos(0, 0, self.flight_cam.alt) 
        
        # Initialisation du gestionnaire de scène
        self.scene_mgr = SceneManager(self)
        
        # Initialisation du gestionnaire de bâtiments
        self.building_mgr = BuildingManager(self)
        
        # États globaux
        self.active_requests = 0
        
        # Initialisation de l'interface utilisateur
        self.overlay = InfoOverlay(self)
        
        # AJUSTEMENT VIEWPORT : On laisse 20% en bas pour le panneau
        # dr est la DisplayRegion de la caméra par défaut (0.0 à 1.0)
        # On définit Left=0, Right=1, Bottom=0.2, Top=1
        dr = self.camNode.getDisplayRegion(0)
        dr.setDimensions(0, 1, 0.2, 1)
        
        # Gestion des événements globaux
        self.accept("escape", self.quit_app)
        self.accept("v", self.toggle_building_edges)
        self.accept("k", self.toggle_map_type)
        
        # Demander à être prévenu quand la fenêtre se ferme via la croix
        self.win.setCloseRequestEvent('window-close')
        self.accept('window-close', self.quit_app)
        
        # Localisation du chargement initial
        self.is_started = False
        self.taskMgr.add(self.check_loading_task, "check_loading_task")

    def toggle_map_type(self):
        new_type = self.scene_mgr.toggle_map_type()
        self.app_config["map_type"] = new_type
        print(f"Switched to {new_type}")

    def check_loading_task(self, task):
        if not self.is_started:
            if len(self.scene_mgr.loaded_tiles) >= 5: # Attendre au moins 5 tuiles
                self.overlay.hide_loading()
                self.is_started = True
                return task.done
        return task.cont
    def toggle_building_edges(self):
        self.building_mgr.toggle_edges()
        self.app_config["show_edges"] = self.building_mgr.show_edges
        print(f"Edges: {self.building_mgr.show_edges}")

    def quit_app(self):
        print("Fermeture de l'application...")
        # Arrêt forcé des pools de threads pour quitter instantanément
        try: self.scene_mgr.executor.shutdown(wait=False)
        except: pass
        try: self.building_mgr.executor.shutdown(wait=False)
        except: pass
        
        # Sauvegarde de la configuration avant de quitter (FENÊTRE UNIQUEMENT)
        props = self.win.getProperties()
        self.app_config["win_width"] = props.getXSize()
        self.app_config["win_height"] = props.getYSize()
        self.app_config["win_x"] = props.getXOrigin()
        self.app_config["win_y"] = props.getYOrigin()
        save_config(self.app_config)
        
        # On quitte radicalement pour éviter les hangs des threads I/O (requêtes bloquantes)
        import os
        os._exit(0)

    def setup_lighting(self):
        # Lumière ambiante
        alight = AmbientLight('alight')
        alight.setColor(Vec4(0.4, 0.4, 0.4, 1))
        alnp = self.render.attachNewNode(alight)
        self.render.setLight(alnp)
        
        # Lumière directionnelle (simulant le soleil)
        dlight = DirectionalLight('dlight')
        dlight.setColor(Vec4(0.8, 0.8, 0.8, 1))
        dlnp = self.render.attachNewNode(dlight)
        dlnp.setHpr(45, -45, 0)
        self.render.setLight(dlnp)

if __name__ == "__main__":
    config = load_config()
    final_config = run_launcher(config)
    
    if final_config:
        # On sauvegarde la position Lat/Lon/Alt choisie au lancement
        save_config(final_config)
        app = AeroSim(final_config)
        app.run()
    else:
        print("Lancement annulé.")
