import requests
import math
from panda3d.core import NodePath, CardMaker, Vec3, GeomNode, GeomVertexFormat, GeomVertexData, GeomVertexWriter, GeomTriangles, Geom
import threading
from concurrent.futures import ThreadPoolExecutor
from queue import Queue
import time
import random

OVERPASS_ENDPOINTS = [
    "https://overpass-api.de/api/interpreter",
    "https://lz4.overpass-api.de/api/interpreter",
    "https://overpass.kumi.systems/api/interpreter",
]

class BuildingManager:
    def __init__(self, base):
        self.base = base
        self.geo_utils = base.scene_mgr.geo_utils
        self.root = base.render.attachNewNode("buildings_root")
        self.root.setBin("opaque", 20)
        self.root.setZ(0.1)
        self.root.setDepthOffset(50) # Marge de sécurité augmentée
        self.root.setTwoSided(True) # Pour éviter le backface culling abusif
        self.root.setLightOff() # Désactiver l'éclairage pour éviter les murs noirs (normales inversées)
        
        self.show_edges = False # Pas d'arêtes par défaut pour le look "brique" solide
        self.apply_render_mode()
        
        self.loaded_areas = set() 
        self.failed_tiles = {} 
        self.executor = ThreadPoolExecutor(max_workers=4)
        self.endpoint_idx = 0 # Pour la rotation
        
        self.ready_buildings = Queue()
        self.waiting_for_terrain = [] 
        self.animating_buildings = [] 
        
        # Seuil de mouvement pour éviter de scanner Overpass trop souvent
        self.last_update_pos = (0, 0)
        self.move_threshold = 50.0 # mètres
        
        self.base.taskMgr.add(self.update_task, "update_buildings_task")

    def toggle_edges(self):
        self.show_edges = not self.show_edges
        self.apply_render_mode()

    def apply_render_mode(self):
        if self.show_edges:
            self.root.setRenderModeFilledWireframe(1)
            self.root.setRenderModeThickness(1)
        else:
            self.root.setRenderModeFilled()

    def update_task(self, task):
        # 1. Plus besoin de traiter les données brutes ici, c'est fait en asynchrone

        # 2. Traiter les nouveaux bâtiments prêts (ceux dont le mesh est fini)
        while not self.ready_buildings.empty():
            self.waiting_for_terrain.append(self.ready_buildings.get())
            
        # 3. Intégrer les bâtiments sur le terrain (Limite par frame pour fluidité)
        still_waiting = []
        count = 0
        for item in self.waiting_for_terrain:
            node, pos, color = item
            if node is None: continue
            
            # ÉCHANTILLONNAGE MULTI-POINT ("Ray-Tracing" de hauteur)
            # On vérifie le centre et 4 points autour (pour couvrir l'emprise moyenne)
            offsets = [(0,0), (5,5), (-5,5), (5,-5), (-5,-5)]
            heights = []
            for dx, dy in offsets:
                hz = self.base.scene_mgr.get_elevation_at(pos[0] + dx, pos[1] + dy)
                if hz is not None: heights.append(hz)
            
            if heights and count < 100:
                # On prend le maximum pour éviter que le bâtiment ne soit enterré
                # L'ancrage de -2m fera le reste pour les parties plus basses.
                ground_z = max(heights)
                count += 1
                try:
                    b_np = self.root.attachNewNode(node)
                    b_np.setPos(pos[0], pos[1], ground_z)
                    b_np.setSz(0.01)
                    
                    # Garantir le rendu correct (double face, pas d'ombres, relief au-dessus du sol)
                    b_np.setTwoSided(True)
                    b_np.setLightOff()
                    b_np.setDepthOffset(50)
                    
                    self.animating_buildings.append([b_np, 1.0, 0.01])
                except: pass
            else:
                still_waiting.append(item)
        self.waiting_for_terrain = still_waiting
            
        # 4. Animer la croissance
        dt = globalClock.getDt()
        speed = 1.0 # Un peu plus rapide
        still_animating = []
        for item in self.animating_buildings:
            b_np, target, current = item
            if b_np.isEmpty(): continue
            new_scale = min(target, current + dt * speed)
            b_np.setSz(new_scale)
            if new_scale < target:
                item[2] = new_scale
                still_animating.append(item)
        self.animating_buildings = still_animating

        # 5. Vérifier les tuiles (Si mouvement significatif)
        lat, lon = self.base.flight_cam.lat, self.base.flight_cam.lon
        curr_x, curr_y = self.geo_utils.latlon_to_meters(lat, lon)
        
        dist_sq = (curr_x - self.last_update_pos[0])**2 + (curr_y - self.last_update_pos[1])**2
        if dist_sq > self.move_threshold**2:
            xt, yt = self.geo_utils.get_tile_coords(lat, lon, 15)
            now = time.time()
            for dx in range(-2, 3):
                for dy in range(-2, 3):
                    tile_key = (xt + dx, yt + dy)
                    if tile_key in self.loaded_areas: continue
                    
                    if tile_key in self.failed_tiles:
                        last_fail, attempts = self.failed_tiles[tile_key]
                        if attempts > 5 or (now - last_fail) < 20:
                            continue
                    
                    self.loaded_areas.add(tile_key)
                    self.executor.submit(self.fetch_buildings, xt + dx, yt + dy, 15)
            self.last_update_pos = (curr_x, curr_y)
            
        self.root.setPos(self.base.scene_mgr.terrain_root.getPos())
        return task.cont

    def fetch_buildings(self, x, y, z):
        self.base.active_requests += 1
        try:
            s, w, n, e = self.geo_utils.get_tile_bbox(x, y, z)
            tile_key = (x, y)
            
            query = f"""
            [out:json][timeout:25];
            (
              nwr["building"]({s},{w},{n},{e});
              nwr["building:part"]({s},{w},{n},{e});
            );
            out body;
            >;
            out skel qt;
            """
            
            # Rotation de l'endpoint pour répartir la charge
            endpoint = OVERPASS_ENDPOINTS[self.endpoint_idx % len(OVERPASS_ENDPOINTS)]
            self.endpoint_idx += 1
            
            try:
                response = requests.get(endpoint, params={'data': query}, timeout=15)
                if response.status_code == 200:
                    data = response.json()
                    # Exécuté dans le thread de l'executor
                    self.process_overpass_data(data)
                    if tile_key in self.failed_tiles:
                        del self.failed_tiles[tile_key]
                else:
                    self.handle_fetch_error(tile_key, response.status_code)
            except Exception:
                self.handle_fetch_error(tile_key, "Exception")
        finally:
            self.base.active_requests -= 1

    def handle_fetch_error(self, tile_key, code):
        if tile_key in self.loaded_areas:
            self.loaded_areas.remove(tile_key)
        
        last_fail, attempts = self.failed_tiles.get(tile_key, (0, 0))
        self.failed_tiles[tile_key] = (time.time(), attempts + 1)
        
        # On ne logue l'erreur que si elle persiste pour ne pas polluer
        if attempts >= 2 and code != 429:
            print(f"Overpass Error {code} sur {tile_key} après {attempts+1} essais")

    def process_overpass_data(self, data):
        elements = data.get('elements', [])
        nodes = {node['id']: (node['lat'], node['lon']) for node in elements if node['type'] == 'node'}
        ways = {way['id']: way['nodes'] for way in elements if way['type'] == 'way'}
        
        for element in elements:
            height = 10.0
            if 'tags' in element:
                h_tag = element['tags'].get('height', '10')
                try: height = float(h_tag)
                except: height = 10.0
            
            all_coords_sets = []
            
            if element['type'] == 'way' and 'nodes' in element:
                coords = [nodes[nid] for nid in element['nodes'] if nid in nodes]
                if len(coords) > 3:
                    all_coords_sets.append(coords)
            
            elif element['type'] == 'relation' and 'members' in element:
                for member in element['members']:
                    # On traite tous les 'outer' pour couvrir les complexes
                    if member['type'] == 'way' and member['role'] == 'outer':
                        way_id = member['ref']
                        if way_id in ways:
                            coords = [nodes[nid] for nid in ways[way_id] if nid in nodes]
                            if len(coords) > 3:
                                all_coords_sets.append(coords)

            for coords in all_coords_sets:
                # Couleur des murs : Gris foncé bleuté pour un look urbain propre
                color = (0.35, 0.35, 0.4, 1)
                
                # APPEL DIRECT (Nous sommes déjà dans un worker thread)
                self.create_building_mesh(coords, height, color)

    def create_building_mesh(self, coords, height, color):
        try:
            # Cette fonction est maintenant exécutée dans un worker thread
            points_m = []
            for lat, lon in coords:
                mx, my = self.geo_utils.latlon_to_meters(lat, lon)
                points_m.append((mx, my))
                
            # FERMETURE DE LA BOUCLE : Si le dernier point n'est pas le premier, on le rajoute
            if len(points_m) > 0 and points_m[0] != points_m[-1]:
                points_m.append(points_m[0])
                
            format = GeomVertexFormat.getV3c4() # v3(position), c4(color)
            vdata = GeomVertexData('building', format, Geom.UHStatic)
            vertex = GeomVertexWriter(vdata, 'vertex')
            color_writer = GeomVertexWriter(vdata, 'color')
            
            cx = sum(p[0] for p in points_m) / len(points_m)
            cy = sum(p[1] for p in points_m) / len(points_m)
            
            roof_color = (0.8, 0.2, 0.2, 1)
            
            num_wall_verts = 0
            for i in range(len(points_m) - 1):
                p1, p2 = points_m[i], points_m[i+1]
                x1, y1 = p1[0] - cx, p1[1] - cy
                x2, y2 = p2[0] - cx, p2[1] - cy
                
                # Murs étendus vers le bas (-2m) pour l'ancrage robuste
                # (évite les trous sur pentes fortes en se calant sur le point haut)
                vertex.addData3(x1, y1, -2.0)
                vertex.addData3(x2, y2, -2.0)
                vertex.addData3(x2, y2, height)
                vertex.addData3(x1, y1, height)
                
                for _ in range(4): color_writer.addData4(color)
                num_wall_verts += 4

            vertex.addData3(0, 0, height)
            color_writer.addData4(roof_color)
            center_idx = num_wall_verts
            
            roof_start_idx = num_wall_verts + 1
            for px, py in [(p[0]-cx, p[1]-cy) for p in points_m]:
                vertex.addData3(px, py, height)
                color_writer.addData4(roof_color)
                
            tris = GeomTriangles(Geom.UHStatic)
            for i in range(0, num_wall_verts, 4):
                tris.addVertices(i, i+1, i+2)
                tris.addVertices(i, i+2, i+3)
                
            for i in range(len(points_m) - 1):
                tris.addVertices(center_idx, roof_start_idx + i, roof_start_idx + i + 1)
                
            geom = Geom(vdata)
            geom.addPrimitive(tris)
            node = GeomNode('building_node')
            node.addGeom(geom)
            
            # On s'assure que le nœud lui-même est configuré pour le double-face
            # au cas où l'héritage du root ne suffirait pas
            np = NodePath(node)
            np.setTwoSided(True)
            np.setLightOff()
            
            # Stocker le résultat pour le thread principal
            self.ready_buildings.put((node, (cx, cy, 0), color))
        except Exception as e:
            # print(f"Erreur création mesh: {e}")
            pass
