class FlightCamera:
    def __init__(self, base, start_lat=45.83, start_lon=6.86):
        self.base = base
        self.camera = base.camera
        
        # Position géographique réelle
        self.lat = start_lat
        self.lon = start_lon
        self.alt = 500.0 # Altitude au-dessus du sol
        
        # Orientation
        self.heading = 0.0 # Yaw
        self.pitch = -30.0 # Pitch initial
        self.speed = 0.0 # En m/s
        self.is_dynamic = False
        
        # Utilitaires
        from gis.geo_utils import GeoUtils
        self.geo_utils = GeoUtils()
        
        # Sensibilité
        self.move_speed = 50.0 # m/s en manuel
        self.turn_speed = 60.0 # deg/s
        self.pitch_speed = 40.0 # deg/s
        
        # État des touches
        self.key_map = {
            "forward": False, "backward": False,
            "left": False, "right": False,
            "up": False, "down": False,
            "turn_left": False, "turn_right": False,
            "pitch_up": False, "pitch_down": False,
            "speed_up": False, "speed_down": False
        }
        
        self.setup_controls()
        self.base.taskMgr.add(self.update_task, "update_camera_task")

    def setup_controls(self):
        # Mouvements
        self.accept_control("z", "forward")
        self.accept_control("s", "backward")
        self.accept_control("q", "turn_left")
        self.accept_control("d", "turn_right")
        self.accept_control("a", "pitch_up")
        self.accept_control("e", "pitch_down")
        
        self.base.accept("space", self.set_key, ["up", True])
        self.base.accept("space-up", self.set_key, ["up", False])
        self.base.accept("shift", self.set_key, ["down", True])
        self.base.accept("shift-up", self.set_key, ["down", False])
        
        # Vitesse
        for k in ["+", "add", "equal"]:
            self.base.accept(k, self.set_key, ["speed_up", True])
            self.base.accept(f"{k}-up", self.set_key, ["speed_up", False])
        for k in ["-", "subtract"]:
            self.base.accept(k, self.set_key, ["speed_down", True])
            self.base.accept(f"{k}-up", self.set_key, ["speed_down", False])
            
        # Contrôles de Zoom
        self.base.accept("wheel_up", self.change_zoom, [1])
        self.base.accept("wheel_down", self.change_zoom, [-1])
        self.accept_control("i", "zoom_in", callback=lambda: self.change_zoom(1))
        self.accept_control("o", "zoom_out", callback=lambda: self.change_zoom(-1))
        
        self.base.accept("m", self.toggle_mode)

    def accept_control(self, key, action, callback=None):
        """Helper pour accepter minuscule et majuscule pour une touche."""
        def press():
            self.set_key(action, True)
            if callback: callback()
        def release():
            self.set_key(action, False)
            
        self.base.accept(key, press)
        self.base.accept(f"{key}-up", release)
        self.base.accept(key.upper(), press)
        self.base.accept(f"{key.upper()}-up", release)

    def change_zoom(self, delta):
        self.base.scene_mgr.set_zoom(delta)

    def set_key(self, key, value):
        self.key_map[key] = value

    def toggle_mode(self):
        self.is_dynamic = not self.is_dynamic

    def update_task(self, task):
        dt = globalClock.getDt()
        
        # Rotation
        if self.key_map["turn_left"]: self.heading += self.turn_speed * dt
        if self.key_map["turn_right"]: self.heading -= self.turn_speed * dt
        if self.key_map["pitch_up"]: self.pitch = min(80, self.pitch + self.pitch_speed * dt)
        if self.key_map["pitch_down"]: self.pitch = max(-80, self.pitch - self.pitch_speed * dt)
            
        self.camera.setH(self.heading)
        self.camera.setP(self.pitch)
        
        # Calcul de la vitesse
        if self.key_map["speed_up"]: self.speed += 50.0 * dt
        if self.key_map["speed_down"]: self.speed = max(0, self.speed - 50.0 * dt)

        # Calcul du déplacement horizontal total
        dist = 0.0
        if self.is_dynamic:
            dist = self.speed * dt
        elif self.key_map["forward"]:
            dist = self.move_speed * dt
        elif self.key_map["backward"]:
            dist = -self.move_speed * dt
            
        # Mise à jour Lat/Lon via Haversine
        if dist != 0:
            # Conversion Panda3D Heading -> Boussole (0=N, 90=E)
            # Panda3D H: + = Gauche (CCW). Boussole: + = Droite (CW).
            bearing = -self.heading
            self.lat, self.lon = self.geo_utils.destination_point(self.lat, self.lon, bearing, dist)
            
        # Altitude
        if self.key_map["up"]: self.alt += self.move_speed * dt
        if self.key_map["down"]: self.alt = max(0, self.alt - self.move_speed * dt)
        
        # On ne change pas la position Camera Panda3D ici directement en X,Y car on va utiliser SceneManager
        # pour gerer les tuiles centrées sur Lat,Lon.
        # Pour le moment, on simule en gardant la caméra au centre (0,0) et en bougeant le monde dans SceneManager.
        self.camera.setZ(self.alt)
        
        return task.cont
