from concurrent.futures import ThreadPoolExecutor
from queue import Queue
from gis.geo_utils import GeoUtils
from gis.tiling import TileManager
from engine.terrain import TerrainGenerator
from panda3d.core import Texture
import time

class SceneManager:
    def __init__(self, base):
        self.base = base
        self.geo_utils = GeoUtils()
        self.tile_manager = TileManager()
        self.terrain_gen = TerrainGenerator(scale=1.0)
        
        self.terrain_root = self.base.render.attachNewNode("terrain_root")
        self.terrain_root.setBin("background", 10)
        self.loaded_tiles = {} # {(z, x, y): (terrain_np, elevation)}
        self.zoom = 15
        
        # File d'attente pour les tuiles prêtes à être ajoutées à la scène
        self.ready_tiles = Queue()
        self.loading_set = set() # Pour éviter les doublons de chargement en cours
        
        # Pool de threads pour le chargement asynchrone
        # On ne peut pas facilement forcer daemon=True sur ThreadPoolExecutor sans subclasser
        # mais os._exit(0) court-circuite tout.
        self.executor = ThreadPoolExecutor(max_workers=4)
        
        # Paramètres de chargement
        self.load_radius = 3
        self.base_zoom = 10
        
        # Seuil de mouvement pour éviter de scanner les tuiles à chaque frame
        self.last_update_pos = (0, 0)
        self.move_threshold = 50.0 # mètres
        
        self.map_type = self.base.app_config.get("map_type", "osm")
        
        self.base.taskMgr.add(self.update_task, "update_scene_task")

    def toggle_map_type(self):
        self.map_type = "sat" if self.map_type == "osm" else "osm"
        print(f"Map Type: {self.map_type}")
        
        # Supprimer visuellement toutes les tuiles actuelles
        for key in self.loaded_tiles:
            self.loaded_tiles[key][0].removeNode()
        self.loaded_tiles.clear()
        
        # Vider la file d'attente des tuiles prêtes (qui ont l'ancien type)
        while not self.ready_tiles.empty():
            try: self.ready_tiles.get_nowait()
            except: break
            
        # Vider les tuiles en cours de chargement
        self.loading_set.clear()
        self.last_update_pos = (0, 0)
        return self.map_type

    def get_elevation_at(self, x, y):
        """Retourne l'élévation interpolée à partir des tuiles chargées."""
        world_size = 40075016.68
        # On cherche dans les tuiles du zoom principal (15)
        n = 2**self.zoom
        tile_size_m = world_size / n
        
        # Calculer l'index de tuile pour (x, y)
        xtile = int((x + world_size / 2) / tile_size_m)
        ytile = int(((world_size / 2) - y) / tile_size_m)
        key = (self.zoom, xtile, ytile)
        
        if key in self.loaded_tiles:
            _, elevation = self.loaded_tiles[key]
            # Calculer la position relative dans la tuile
            rel_x = (x + world_size / 2) % tile_size_m
            rel_y = (world_size / 2 - y) % tile_size_m
            
            h, w = elevation.shape
            # Map rel_x, rel_y to grid index
            idx_x = (rel_x / tile_size_m) * (w - 1)
            idx_y = (rel_y / tile_size_m) * (h - 1)
            
            # Simple bilinear interpolation or nearest neighbor for performance
            ix, iy = int(idx_x), int(idx_y)
            # Clip for safety
            ix = max(0, min(w - 2, ix))
            iy = max(0, min(h - 2, iy))
            
            # Pour l'instant on prend le plus proche ou un facteur simple
            # L'élévation est multipliée par 0.1 dans terrain.py mais aussi par 10.0 dans l'échelle
            # En réalité, terrain.py fait z = elevation * 0.1 et scale=(..., ..., 10.0)
            # Donc l'élévation finale z_world = elevation
            return elevation[iy, ix]
        return None

    def set_zoom(self, delta):
        old_zoom = self.zoom
        self.zoom = max(5, min(19, self.zoom + delta))
        if self.zoom != old_zoom:
            print(f"Zoom: {self.zoom}")
            # On ne vide plus loaded_tiles ici !
            # Au lieu de ça, on force une mise à jour immédiate
            self.last_update_pos = (0, 0) 

    def update_task(self, task):
        # 1. Traiter les tuiles prêtes (max 4 par frame pour fluidité/vitesse)
        processed = 0
        while not self.ready_tiles.empty() and processed < 4:
            key, node, elevation, tex, pos, scale = self.ready_tiles.get()
            processed += 1
            z, x, y = key
            
            if key not in self.loaded_tiles:
                tile_np = self.terrain_root.attachNewNode(node)
                tile_np.setTwoSided(True)
                if tex:
                    tile_np.setTexture(tex)
                tile_np.setPos(pos)
                tile_np.setScale(scale)
                # Correction scintillement : Plus le zoom est élevé (détails), 
                # plus on pousse la tuile "devant" dans le buffer de profondeur.
                tile_np.setDepthOffset(z)
                self.loaded_tiles[key] = (tile_np, elevation)
                
                # NETTOYAGE INTELLIGENT : On ne supprime que si on a bcp de tuiles
                # ou si elles sont vraiment hors radar.
                if len(self.loaded_tiles) > 100:
                    bg_zoom = max(5, self.zoom - 2)
                    stale_keys = [k for k in self.loaded_tiles 
                                 if k[0] != self.zoom and k[0] != bg_zoom and k[0] != self.base_zoom]
                    
                    # On supprime les plus vieilles (ordre dictionary) ou les plus loin
                    # Pour faire simple : 5 max par frame
                    for i in range(min(len(stale_keys), 5)):
                        k = stale_keys[i]
                        self.loaded_tiles[k][0].removeNode()
                        del self.loaded_tiles[k]
            
            if key in self.loading_set:
                self.loading_set.remove(key)

        # 2. Vérifier si on doit charger de nouvelles tuiles (seulement si mouvement significatif)
        lat = self.base.flight_cam.lat
        lon = self.base.flight_cam.lon
        curr_x, curr_y = self.geo_utils.latlon_to_meters(lat, lon)
        
        dist_sq = (curr_x - self.last_update_pos[0])**2 + (curr_y - self.last_update_pos[1])**2
        if dist_sq > self.move_threshold**2:
            self.update_tiles(lat, lon)
            self.last_update_pos = (curr_x, curr_y)
        
        # 3. Recentrer le monde
        self.terrain_root.setPos(-curr_x, -curr_y, 0)
        
        return task.cont

    def update_tiles(self, lat, lon):
        # On collecte toutes les tuiles nécessaires
        tasks = []
        
        # 1. Zoom principal (Rayon 3)
        xtile, ytile = self.geo_utils.get_tile_coords(lat, lon, self.zoom)
        for dx in range(-self.load_radius, self.load_radius + 1):
            for dy in range(-self.load_radius, self.load_radius + 1):
                tasks.append((xtile + dx, ytile + dy, self.zoom))
                
        # 2. Zoom arrière-plan (Rayon 2)
        bg_zoom = max(5, self.zoom - 2)
        bg_x, bg_y = self.geo_utils.get_tile_coords(lat, lon, bg_zoom)
        for dx in range(-2, 3):
            for dy in range(-2, 3):
                tasks.append((bg_x + dx, bg_y + dy, bg_zoom))
                
        # 3. Fond permanent (Zoom 10 - Rayon 1)
        base_x, base_y = self.geo_utils.get_tile_coords(lat, lon, self.base_zoom)
        for dx in range(-1, 2):
            for dy in range(-1, 2):
                tasks.append((base_x + dx, base_y + dy, self.base_zoom))

        # TRI PAR PRIORITÉ :
        # - Priorité 1 : Zoom actuel
        # - Priorité 2 : Proximité du centre
        def priority_sort(t):
            z, tx, ty = t[2], t[0], t[1]
            # On calcule la distance approximative au centre par niveau de zoom
            cx, cy = self.geo_utils.get_tile_coords(lat, lon, z)
            dist = (tx - cx)**2 + (ty - cy)**2
            # On veut z le plus haut possible d'abord, puis dist la plus faible
            return (z != self.zoom, dist)

        tasks.sort(key=priority_sort)

        for tx, ty, tz in tasks:
            self.ensure_tile(tx, ty, tz)

    def ensure_tile(self, x, y, z):
        key = (z, x, y)
        if key not in self.loaded_tiles and key not in self.loading_set:
            self.loading_set.add(key)
            # Lancer le chargement dans le pool
            self.executor.submit(self.async_load_tile, x, y, z)

    def async_load_tile(self, x, y, z):
        try:
            elevation = self.tile_manager.get_elevation_data(x, y, z, base=self.base)
            texture_path = self.tile_manager.get_texture_tile(x, y, z, base=self.base, map_type=self.map_type)
            
            if elevation is not None:
                # ... (mesh gen)
                # ... (mesh gen)
                node = self.terrain_gen.generate_terrain_mesh(elevation)
                # ... (texture load)
                tex = None
                if texture_path:
                    try:
                        tex = self.base.loader.loadTexture(texture_path)
                        if tex:
                            tex.setMinfilter(Texture.FTLinearMipmapLinear)
                            tex.setMagfilter(Texture.FTLinear)
                            tex.setAnisotropicDegree(16)
                    except: pass

                num_intervals = elevation.shape[1] - 1
                scale_xy = (40075016.68 / (2**z)) / num_intervals
                
                pos_x = x * (40075016.68 / (2**z)) - (40075016.68 / 2)
                pos_y = (40075016.68 / 2) - (y + 1) * (40075016.68 / (2**z))
                
                # Plus d'offset Z physique, on utilise setDepthOffset dans update_task
                self.ready_tiles.put(((z, x, y), node, elevation, tex, (pos_x, pos_y, 0), (scale_xy, scale_xy, 10.0)))
            else:
                self.handle_load_error(x, y, z)
        except Exception as e:
            self.handle_load_error(x, y, z)
        finally:
            self.base.active_requests -= 1

    def handle_load_error(self, x, y, z):
        key = (z, x, y)
        if key in self.loading_set:
            self.loading_set.remove(key)
