import numpy as np
from panda3d.core import GeomVertexFormat, GeomVertexData, GeomVertexWriter
from panda3d.core import Geom, GeomTriangles, GeomNode, Texture, TextureStage

class TerrainGenerator:
    def __init__(self, size=256, scale=1.0):
        self.size = size
        self.scale = scale

    def generate_terrain_mesh(self, elevation_data, texture_path=None):
        """Génère un mesh à partir d'une matrice d'élévation numpy."""
        format = GeomVertexFormat.getV3n3t2()
        vdata = GeomVertexData('terrain', format, Geom.UHStatic)
        
        vertex = GeomVertexWriter(vdata, 'vertex')
        normal = GeomVertexWriter(vdata, 'normal')
        texcoord = GeomVertexWriter(vdata, 'texcoord')
        
        h, w = elevation_data.shape
        # On redimensionne les données pour qu'elles correspondent à self.size
        # Pour simplifier ici, on suppose que elevation_data est déjà à la bonne taille
        
        # Création des sommets
        for y in range(h):
            for x in range(w):
                z = elevation_data[y, x] * 0.1 # Facteur d'échelle pour l'altitude
                # Correction Y: le haut de l'image (y=0, Nord) doit être à +Y
                vertex.addData3(x * self.scale, (h - 1 - y) * self.scale, z)
                # Normale simplifiée (vers le haut)
                normal.addData3(0, 0, 1)
                # v=1.0 pour le haut (y=0), v=0 pour le bas (y=h-1)
                texcoord.addData2(x / (w - 1), 1.0 - y / (h - 1))
        
        # Création des triangles
        tris = GeomTriangles(Geom.UHStatic)
        for y in range(h - 1):
            for x in range(w - 1):
                v0 = y * w + x
                v1 = v0 + 1
                v2 = (y + 1) * w + x
                v3 = v2 + 1
                
                # Winding CCW
                tris.addVertices(v0, v2, v1)
                tris.addVertices(v1, v2, v3)
        
        geom = Geom(vdata)
        geom.addPrimitive(tris)
        
        node = GeomNode('terrain_node')
        node.addGeom(geom)
        
        return node
