from pyproj import Transformer

class GeoUtils:
    def __init__(self):
        # Transormateur WGS84 (Lat/Lon) vers Web Mercator (EPSG:3857)
        # On utilise Web Mercator car c'est le standard pour les tuiles (OSM, Google, etc.)
        self.transformer = Transformer.from_crs("EPSG:4326", "EPSG:3857", always_xy=True)
    
    def latlon_to_meters(self, lat, lon):
        """Convertit Latitude/Longitude en mètres (Web Mercator X, Y)."""
        x, y = self.transformer.transform(lon, lat)
        return x, y

    def get_tile_coords(self, lat, lon, zoom):
        """Calcule les indices de tuile OSM (x, y) pour une position donnée et un zoom."""
        import math
        lat_rad = math.radians(lat)
        n = 2.0 ** zoom
        xtile = int((lon + 180.0) / 360.0 * n)
        ytile = int((1.0 - math.log(math.tan(lat_rad) + (1 / math.cos(lat_rad))) / math.pi) / 2.0 * n)
        return xtile, ytile

    def destination_point(self, lat, lon, bearing, distance):
        """
        Calcule la nouvelle position (Lat, Lon) à partir d'un point, d'un cap et d'une distance.
        lat, lon: degrés
        bearing: degrés (cap)
        distance: mètres
        R: rayon de la Terre (6371000 m)
        """
        import math
        R = 6371000.0
        phi1 = math.radians(lat)
        lam1 = math.radians(lon)
        theta = math.radians(bearing)
        d_R = distance / R

        phi2 = math.asin(math.sin(phi1) * math.cos(d_R) +
                         math.cos(phi1) * math.sin(d_R) * math.cos(theta))
        lam2 = lam1 + math.atan2(math.sin(theta) * math.sin(d_R) * math.cos(phi1),
                                 math.cos(d_R) - math.sin(phi1) * math.sin(phi2))

        return math.degrees(phi2), math.degrees(lam2)
    def tile_to_latlon(self, x, y, z):
        """Convertit les indices de tuile en Lat/Lon (coin Nord-Ouest)."""
        import math
        n = 2.0 ** z
        lon_deg = x / n * 360.0 - 180.0
        lat_rad = math.atan(math.sinh(math.pi * (1 - 2 * y / n)))
        lat_deg = math.degrees(lat_rad)
        return lat_deg, lon_deg

    def get_tile_bbox(self, x, y, z):
        """Retourne la BBox (S, W, N, E) d'une tuile."""
        lat_n, lon_w = self.tile_to_latlon(x, y, z)
        lat_s, lon_e = self.tile_to_latlon(x + 1, y + 1, z)
        return lat_s, lon_w, lat_n, lon_e
