import os
import requests
import math
import numpy as np
from PIL import Image
from io import BytesIO

class TileManager:
    def __init__(self, cache_dir="cache"):
        self.cache_dir = cache_dir
        self.elevation_cache = os.path.join(cache_dir, "elevation")
        self.texture_cache = os.path.join(cache_dir, "textures")
        
        # URL des serveurs de tuiles
        self.osm_url_template = "https://tile.openstreetmap.org/{z}/{x}/{y}.png"
        self.sat_url_template = "https://server.arcgisonline.com/ArcGIS/rest/services/World_Imagery/MapServer/tile/{z}/{y}/{x}"
        
        # Note: Pour l'élévation, on utiliseMapzen terrarium
        self.ele_url_template = "https://s3.amazonaws.com/elevation-tiles-prod/terrarium/{z}/{x}/{y}.png"

    def get_texture_tile(self, x, y, z, base=None, map_type="osm"):
        """Récupère une tuile de texture (depuis le cache ou le web)."""
        # Séparation du cache par type de carte
        sub_cache = "osm" if map_type == "osm" else "sat"
        cache_path = os.path.join(self.texture_cache, sub_cache, f"{z}_{x}_{y}.png")
        os.makedirs(os.path.dirname(cache_path), exist_ok=True)
        
        if os.path.exists(cache_path):
            return cache_path
            
        if map_type == "osm":
            url = self.osm_url_template.format(z=z, x=x, y=y)
        else:
            url = self.sat_url_template.format(z=z, x=x, y=y)
            
        headers = {"User-Agent": "AeroSim/1.0 (Python Simulation)"}
        
        if base: base.active_requests += 1
        try:
            response = requests.get(url, headers=headers, timeout=10)
            if response.status_code == 200:
                with open(cache_path, "wb") as f:
                    f.write(response.content)
                return cache_path
        except Exception as e:
            print(f"Erreur téléchargement texture: {e}")
        finally:
            if base: base.active_requests -= 1
            
        return None

    def get_elevation_data(self, x, y, z, base=None):
        """Récupère les données d'élévation d'une tuile (format Terrarium)."""
        cache_path = os.path.join(self.elevation_cache, f"{z}_{x}_{y}.png")
        
        if not os.path.exists(cache_path):
            url = self.ele_url_template.format(z=z, x=x, y=y)
            if base: base.active_requests += 1
            try:
                response = requests.get(url, timeout=10)
                if response.status_code == 200:
                    with open(cache_path, "wb") as f:
                        f.write(response.content)
                else:
                    return None
            except Exception as e:
                print(f"Erreur téléchargement élévation: {e}")
                return None
            finally:
                if base: base.active_requests -= 1
        
        # Décodage de l'élévation (Format Terrarium: elevation = (R * 256 + G + B / 256) - 32768)
        img = Image.open(cache_path).convert("RGB")
        data = np.array(img, dtype=np.float32)
        r, g, b = data[:,:,0], data[:,:,1], data[:,:,2]
        elevation = (r * 256 + g + b / 256.0) - 32768.0
        return elevation
