from direct.gui.OnscreenText import OnscreenText
from direct.gui.DirectFrame import DirectFrame
from panda3d.core import TextNode

class InfoOverlay:
    def __init__(self, base):
        self.base = base
        self.texts = {}
        
        # Écran de chargement
        self.loading_screen = self.base.aspect2d.attachNewNode("loading_screen")
        self.loading_bg = OnscreenText(text="", pos=(0,0), scale=10, bg=(0,0,0,1), parent=self.loading_screen)
        self.loading_text = OnscreenText(text="Initialisation du vol...\nChargement des tuiles SIG", 
                                        pos=(0, 0), scale=0.1, fg=(1, 1, 1, 1),
                                        align=TextNode.ACenter, parent=self.loading_screen)
        
        # Création du panneau de contrôle en bas
        self.panel = DirectFrame(frameColor=(0.1, 0.1, 0.15, 0.9),
                                frameSize=(-1, 1, -1.0, -0.6),
                                parent=self.base.aspect2d)
        
        self.header_bg = DirectFrame(frameColor=(0.2, 0.2, 0.3, 1),
                                    frameSize=(-1, 1, -0.62, -0.6),
                                    parent=self.panel)
        
        # Zoom et Status (Bas Gauche)
        self.status_dot = DirectFrame(frameColor=(0.2, 0.8, 0.2, 1),
                                     frameSize=(-0.02, 0.02, -0.02, 0.02),
                                     pos=(-1.2, 0, -0.96),
                                     parent=self.panel)
        
        self.zoom_text = OnscreenText(text="ZOOM: --", pos=(-1.15, -0.97), scale=0.035, 
                                     fg=(1, 1, 1, 1), align=TextNode.ALeft, 
                                     parent=self.panel, font=self.base.loader.loadFont('cmss12.egg'))
        
        self.adjust_panel_size()
        self.setup_overlay()
        self.base.taskMgr.add(self.update_task, "update_overlay_task")
        self.base.accept("window-event", self.on_window_event)

    def on_window_event(self, window):
        self.adjust_panel_size()

    def adjust_panel_size(self):
        aspect = self.base.getAspectRatio()
        self.panel['frameSize'] = (-aspect, aspect, -1.0, -0.6)
        self.header_bg['frameSize'] = (-aspect, aspect, -0.62, -0.6)

    def hide_loading(self):
        if self.loading_screen:
            self.loading_screen.hide()

    def setup_overlay(self):
        # Indicateurs de vol (Colonne 1)
        col1_x = -1.2
        self.texts["pos"] = OnscreenText(text="LAT: -- | LON: -- | ALT: --", 
                                        pos=(col1_x, -0.72), scale=0.04, fg=(0.9, 0.9, 1, 1), 
                                        align=TextNode.ALeft, parent=self.panel, font=self.base.loader.loadFont('cmss12.egg'))
        self.texts["speed"] = OnscreenText(text="VITESSE: 0 km/h", 
                                          pos=(col1_x, -0.80), scale=0.04, fg=(1, 1, 1, 1), 
                                          align=TextNode.ALeft, parent=self.panel)
        self.texts["mode"] = OnscreenText(text="PILOTAGE: STATIQUE", 
                                         pos=(col1_x, -0.88), scale=0.04, fg=(1, 1, 0, 1), 
                                         align=TextNode.ALeft, parent=self.panel)
        self.texts["cap"] = OnscreenText(text="CAP: 000°", 
                                        pos=(col1_x + 0.45, -0.80), scale=0.04, fg=(0.4, 1, 0.4, 1), 
                                        align=TextNode.ALeft, parent=self.panel)
        self.texts["b_stats"] = OnscreenText(text="BATIMENTS: -- visibles | -- en attente", 
                                            pos=(-0.8, -0.97), scale=0.03, fg=(0.8, 0.8, 0.8, 1), 
                                            align=TextNode.ALeft, parent=self.panel)

        # Mnémos des touches (Colonne 2 & 3)
        col2_x = -0.1
        commands = [
            "Z/S : Avancer/Reculer",
            "Q/D : Rotation G/D",
            "A/E : Inclinaison Haut/Bas",
            "Space/Shift : Monter/Descendre"
        ]
        for i, cmd in enumerate(commands):
            OnscreenText(text=cmd, pos=(col2_x, -0.68 - i*0.06), scale=0.035, 
                         fg=(0.8, 0.8, 0.8, 1), align=TextNode.ALeft, parent=self.panel)

        col3_x = 0.6
        commands2 = [
            "M : Changer Mode (Statique/Dyn)",
            "+/- : Regler Vitesse",
            "I/O/K : Zoom / Type Carte",
            "V : Bordures | ESC : Quitter"
        ]
        for i, cmd in enumerate(commands2):
            OnscreenText(text=cmd, pos=(col3_x, -0.68 - i*0.06), scale=0.035, 
                         fg=(0.8, 0.8, 0.8, 1), align=TextNode.ALeft, parent=self.panel)

    def update_task(self, task):
        flight_cam = self.base.flight_cam
        lat, lon, alt = flight_cam.lat, flight_cam.lon, flight_cam.alt
        
        # Format fixe pour éviter les tremblements (signe inclus, padding 0)
        lat_str = f"{lat:+.5f}".rjust(10)
        lon_str = f"{lon:+.5f}".rjust(11)
        alt_str = f"{alt:6.1f}".rjust(8)
        
        self.texts["pos"].setText(f"LAT: {lat_str}  |  LON: {lon_str}  |  ALT: {alt_str} m")
        self.texts["speed"].setText(f"VITESSE: {abs(flight_cam.speed * 3.6):5.1f} km/h")
        self.texts["mode"].setText(f"PILOTAGE: {'DYNAMIQUE (Continu)' if flight_cam.is_dynamic else 'STATIQUE (Manuel)'}")
        
        # Calcul du Cap (Normalisé 0-359)
        heading = (self.base.camera.getH() % 360 + 360) % 360
        self.texts["cap"].setText(f"CAP: {int(heading):03d}°")
        
        # Stats bâtiments
        visibles = self.base.building_mgr.root.getNumChildren()
        attente = len(self.base.building_mgr.waiting_for_terrain)
        self.texts["b_stats"].setText(f"BATIMENTS: {visibles} visibles | {attente} en attente")
        
        # Zoom actuel
        self.zoom_text.setText(f"ZOOM: {self.base.scene_mgr.zoom}")
        
        # Indicateur réseau (Vrai trafic HTTP)
        if hasattr(self.base, 'active_requests') and self.base.active_requests > 0:
            self.status_dot['frameColor'] = (0.8, 0.2, 0.2, 1) # Rouge (Actif)
        else:
            self.status_dot['frameColor'] = (0.2, 0.8, 0.2, 1) # Vert (Repos)
            
        return task.cont
