
from enum import Enum
from dataclasses import dataclass, field
from typing import List, Optional, Tuple
import uuid

# Constantes de conversion
DPI_SCREEN = 96
MM_TO_INCH = 1 / 25.4
MM_TO_PX = MM_TO_INCH * DPI_SCREEN

class PageFormat(Enum):
    A4 = (210, 297)
    A3 = (297, 420)
    US_LETTER = (215.9, 279.4)

    def to_pixels(self) -> Tuple[float, float]:
        w_mm, h_mm = self.value
        return (w_mm * MM_TO_PX, h_mm * MM_TO_PX)

def mm_to_px(mm: float) -> float:
    return mm * MM_TO_PX

def px_to_mm(px: float) -> float:
    return px / MM_TO_PX

@dataclass
class ZoneModel:
    """Modèle de données pour une zone (texte ou image)."""
    id: str = field(default_factory=lambda: str(uuid.uuid4()))
    x: float = 0.0  # En pixels
    y: float = 0.0  # En pixels
    width: float = 100.0
    height: float = 100.0
    type: str = "generic" # 'text' ou 'image'
    content: str = "" # Texte brut ou chemin image
    style: dict = field(default_factory=dict) # Police, taille, etc.
    locked: bool = False
    z_index: int = 0
    role: str = "body" # "body", "heading1", "heading2", "toc"

@dataclass
class PageModel:
    """Modèle de données pour une page."""
    id: str = field(default_factory=lambda: str(uuid.uuid4()))
    width: float = 0.0
    height: float = 0.0
    zones: List[ZoneModel] = field(default_factory=list)
    margin_top: float = 0.0
    margin_bottom: float = 0.0
    margin_left: float = 0.0
    margin_right: float = 0.0
    background_color: str = "#FFFFFF"

@dataclass
class DocumentModel:
    """Modèle racine du document."""
    title: str = "Sans titre"
    pages: List[PageModel] = field(default_factory=list)
    active_page_index: int = 0
    numbering_style: str = "current_total" # "current_total" or "current_only"
    show_numbers_from: int = 1 # Start showing numbers from this page index (1-based)
    
    def create_page(self, format: PageFormat = PageFormat.A4) -> PageModel:
        w, h = format.to_pixels()
        page = PageModel(width=w, height=h)
        # Marges par défaut (10mm)
        margin_px = mm_to_px(10)
        page.margin_top = margin_px
        page.margin_bottom = margin_px
        page.margin_left = margin_px
        page.margin_right = margin_px
        
        self.pages.append(page)
        return page

    def move_page(self, old_index: int, new_index: int):
        if 0 <= old_index < len(self.pages) and 0 <= new_index < len(self.pages):
             page = self.pages.pop(old_index)
             self.pages.insert(new_index, page)
             # Adjust active index if needed
             if self.active_page_index == old_index:
                 self.active_page_index = new_index
             elif old_index < self.active_page_index <= new_index:
                 self.active_page_index -= 1
             elif new_index <= self.active_page_index < old_index:
                 self.active_page_index += 1

