
import json
import os
import html
from PySide6.QtGui import QPdfWriter, QPainter, QPageLayout, QPageSize, QImage, QFont, QColor
from PySide6.QtCore import QSizeF, Qt, QRectF, QPointF, QMarginsF
from core.document import DocumentModel, PageModel, ZoneModel, PageFormat, mm_to_px, px_to_mm

class ProjectSerializer:
    @staticmethod
    def to_json(document: DocumentModel) -> dict:
        """Convertit le modèle Document en dictionnaire JSON."""
        return {
            "title": document.title,
            "active_page_index": document.active_page_index,
            "numbering_style": document.numbering_style,
            "show_numbers_from": document.show_numbers_from,
            "pages": [ProjectSerializer._page_to_dict(p) for p in document.pages]
        }

    @staticmethod
    def _page_to_dict(page: PageModel) -> dict:
        return {
            "id": page.id,
            "width": page.width,
            "height": page.height,
            "margins": {
                "top": page.margin_top,
                "bottom": page.margin_bottom,
                "left": page.margin_left,
                "right": page.margin_right
            },
            "background_color": page.background_color,
            "zones": [ProjectSerializer._zone_to_dict(z) for z in page.zones]
        }

    @staticmethod
    def _zone_to_dict(zone: ZoneModel) -> dict:
        return {
            "id": zone.id,
            "type": zone.type,
            "x": zone.x,
            "y": zone.y,
            "width": zone.width,
            "height": zone.height,
            "content": zone.content,
            "style": zone.style,
            "locked": zone.locked,
            "z_index": zone.z_index,
            "role": zone.role
        }

    @staticmethod
    def from_json(data: dict) -> DocumentModel:
        """Reconstruit un DocumentModel depuis un dictionnaire."""
        doc = DocumentModel()
        doc.title = data.get("title", "Sans titre")
        doc.active_page_index = data.get("active_page_index", 0)
        doc.numbering_style = data.get("numbering_style", "current_total")
        doc.show_numbers_from = data.get("show_numbers_from", 1)
        doc.pages = [ProjectSerializer._dict_to_page(p) for p in data.get("pages", [])]
        return doc

    @staticmethod
    def _dict_to_page(data: dict) -> PageModel:
        page = PageModel(
            width=data.get("width", 0),
            height=data.get("height", 0)
        )
        page.id = data.get("id", "")
        margins = data.get("margins", {})
        page.margin_top = margins.get("top", 0)
        page.margin_bottom = margins.get("bottom", 0)
        page.margin_left = margins.get("left", 0)
        page.margin_right = margins.get("right", 0)
        page.background_color = data.get("background_color", "#FFFFFF")
        
        page.zones = [ProjectSerializer._dict_to_zone(z) for z in data.get("zones", [])]
        return page

    @staticmethod
    def _dict_to_zone(data: dict) -> ZoneModel:
        zone = ZoneModel()
        zone.id = data.get("id", "")
        zone.type = data.get("type", "generic")
        zone.x = data.get("x", 0)
        zone.y = data.get("y", 0)
        zone.width = data.get("width", 100)
        zone.height = data.get("height", 100)
        zone.content = data.get("content", "")
        zone.style = data.get("style", {})
        zone.locked = data.get("locked", False)
        zone.z_index = data.get("z_index", 0)
        zone.role = data.get("role", "body")
        return zone

    @staticmethod
    def save_to_file(document: DocumentModel, filepath: str):
        data = ProjectSerializer.to_json(document)
        with open(filepath, 'w', encoding='utf-8') as f:
            json.dump(data, f, indent=4)

    @staticmethod
    def load_from_file(filepath: str) -> DocumentModel:
        with open(filepath, 'r', encoding='utf-8') as f:
            data = json.load(f)
        return ProjectSerializer.from_json(data)


class PDFExporter:
    @staticmethod
    def export(document: DocumentModel, filepath: str, project_dir: str = ""):
        """Exporte le document en PDF haute résolution (300 DPI)."""
        writer = QPdfWriter(filepath)
        writer.setCreator("PAO Zone-Based v0.2")
        writer.setResolution(300) # Force 300 DPI
        
        # Configuration Page A4 par défaut
        page_layout = QPageLayout(
            QPageSize(QPageSize.A4),
            QPageLayout.Portrait,
            QMarginsF(0, 0, 0, 0),
            QPageLayout.Millimeter
        )
        writer.setPageLayout(page_layout) 
        
        painter = QPainter(writer)
        
        # Scale factor for coordinates
        scale = 300 / 96.0
        
        for i, page in enumerate(document.pages):
            if i > 0:
                writer.newPage()
            
            # Isolate this page's rendering completely
            painter.save()
            
            # --- Background ---
            if hasattr(page, 'background_color') and page.background_color != "#FFFFFF":
                w = page.width * scale
                h = page.height * scale
                painter.fillRect(QRectF(0, 0, w, h), QColor(page.background_color))
            
            # --- Zones ---
            for zone in page.zones:
                x = zone.x * scale
                y = zone.y * scale
                w = zone.width * scale
                h = zone.height * scale
                rect = QRectF(x, y, w, h)
                
                if zone.type == "text":
                    font = QFont(zone.style.get("font_family", "Arial"))
                    # Font points stay physical - QPdfWriter handles DPI internally
                    font.setPointSizeF(zone.style.get("font_size", 12))
                    painter.setFont(font)
                    painter.setPen(QColor(zone.style.get("text_color", "#000000")))
                    painter.drawText(rect, Qt.AlignLeft | Qt.TextWordWrap, zone.content)
                    
                elif zone.type == "image":
                    img_path = zone.content
                    if img_path and not os.path.isabs(img_path) and project_dir:
                        img_path = os.path.join(project_dir, img_path)
                    if img_path and os.path.exists(img_path):
                        img = QImage(img_path)
                        if not img.isNull():
                            painter.drawImage(rect, img)
            
            # --- Page Number ---
            if (i + 1) >= document.show_numbers_from:
                num_text = f"{i + 1}" if document.numbering_style == "current_only" else f"{i + 1} / {len(document.pages)}"
                font = QFont("Arial")
                font.setPointSizeF(10)
                painter.setFont(font)
                painter.setPen(QColor("gray"))
                
                page_h = page.height * scale
                page_w = page.width * scale
                margin = 15 * (300 / 25.4)  # 15mm in PDF pixels
                
                painter.drawText(QRectF(0, page_h - margin, page_w, margin), Qt.AlignCenter, num_text)
            
            painter.restore()
        
        painter.end()


class HTMLExporter:
    @staticmethod
    def export(document: DocumentModel, filepath: str, project_dir: str = ""):
        """Exporte le document en HTML (une page longue ou fichiers séparés)."""
        html_content = """<!DOCTYPE html>
<html>
<head>
    <meta charset="utf-8">
    <title>Export PAO</title>
    <style>
        body { background-color: #e0e0e0; margin: 0; padding: 20px; font-family: sans-serif; }
        .page { 
            position: relative; 
            background-color: white; 
            margin: 0 auto 20px auto; 
            box-shadow: 0 0 10px rgba(0,0,0,0.1); 
            overflow: hidden;
        }
        .zone { position: absolute; overflow: hidden; }
        .zone-text { word-wrap: break-word; }
        .zone-image img { width: 100%; height: 100%; object-fit: cover; }
    </style>
</head>
<body>
"""
        
        for i, page in enumerate(document.pages):
           width = page.width
           height = page.height
           bg_color = page.background_color if hasattr(page, 'background_color') else "#FFFFFF"
           
           html_content += f'<div class="page" style="width: {width}px; height: {height}px; background-color: {bg_color};">\n'
           
           for zone in page.zones:
               style = f"left: {zone.x}px; top: {zone.y}px; width: {zone.width}px; height: {zone.height}px; z-index: {zone.z_index};"
               
               if zone.type == "text":
                   font_family = zone.style.get("font_family", "Arial")
                   font_size = zone.style.get("font_size", 12)
                   text_color = zone.style.get("text_color", "#000000")
                   
                   bold = "font-weight: bold;" if zone.style.get("bold") else ""
                   italic = "font-style: italic;" if zone.style.get("italic") else ""
                   underline = "text-decoration: underline;" if zone.style.get("underline") else ""
                   
                   text_style = f"font-family: '{font_family}', sans-serif; font-size: {font_size}pt; color: {text_color}; {bold} {italic} {underline}"
                   
                   content_html = html.escape(zone.content).replace("\n", "<br>")
                   html_content += f'<div class="zone zone-text" style="{style} {text_style}">{content_html}</div>\n'
                   
               elif zone.type == "image":
                   img_path = zone.content
                   if img_path:
                       if not os.path.isabs(img_path) and project_dir:
                           img_path = os.path.join(project_dir, img_path)
                       # Convert path for HTML
                       img_src = f"file:///{img_path.replace(os.sep, '/')}"
                       html_content += f'<div class="zone zone-image" style="{style}"><img src="{img_src}"></div>\n'

           html_content += '</div>\n'
           
        html_content += "</body></html>"
        
        with open(filepath, "w", encoding="utf-8") as f:
            f.write(html_content)


class MarkdownExporter:
    @staticmethod
    def export(document: DocumentModel, filepath: str, project_dir: str = ""):
        """Exporte le document en Markdown (linéarisé)."""
        md_content = f"# {document.title}\n\n"
        
        for i, page in enumerate(document.pages):
            if len(document.pages) > 1:
                md_content += f"## Page {i+1}\n\n"
            
            # Tri spatial : Haut -> Bas, Gauche -> Droite
            sorted_zones = sorted(page.zones, key=lambda z: (z.y, z.x))
            
            for zone in sorted_zones:
                if zone.type == "text":
                    content = zone.content
                    role = getattr(zone, 'role', 'body')
                    
                    if role == "heading1": 
                        md_content += f"# {content}\n\n"
                    elif role == "heading2": 
                        md_content += f"## {content}\n\n"
                    else:
                        # Apply B/I if present
                        prefix = ""
                        suffix = ""
                        if zone.style.get("bold"):
                            prefix += "**"
                            suffix = "**" + suffix
                        if zone.style.get("italic"):
                            prefix += "*"
                            suffix = "*" + suffix
                            
                        md_content += f"{prefix}{content}{suffix}\n\n"
                    
                elif zone.type == "image":
                    img_path = zone.content
                    # Markdown prefers forward slashes
                    img_path = img_path.replace(os.sep, "/")
                    md_content += f"![Image]({img_path})\n\n"
            
            if i < len(document.pages) - 1:
                md_content += "---\n\n"
            
        with open(filepath, "w", encoding="utf-8") as f:
            f.write(md_content)

