
from typing import List, Optional

class Command:
    """Base class for undo/redo commands"""
    def execute(self):
        raise NotImplementedError
    
    def undo(self):
        raise NotImplementedError

class UndoRedoManager:
    def __init__(self):
        self.undo_stack: List[Command] = []
        self.redo_stack: List[Command] = []
        self.max_stack_size = 50  # Limite pour éviter surcharge mémoire
    
    def execute_command(self, command: Command):
        """Execute a command and add it to undo stack"""
        command.execute()
        self.undo_stack.append(command)
        
        # Clear redo stack when new command is executed
        self.redo_stack.clear()
        
        # Limit stack size
        if len(self.undo_stack) > self.max_stack_size:
            self.undo_stack.pop(0)
    
    def undo(self):
        """Undo the last command"""
        if not self.undo_stack:
            return False
        
        command = self.undo_stack.pop()
        command.undo()
        self.redo_stack.append(command)
        return True
    
    def redo(self):
        """Redo the last undone command"""
        if not self.redo_stack:
            return False
        
        command = self.redo_stack.pop()
        command.execute()
        self.undo_stack.append(command)
        return True
    
    def can_undo(self) -> bool:
        return len(self.undo_stack) > 0
    
    def can_redo(self) -> bool:
        return len(self.redo_stack) > 0
    
    def clear(self):
        """Clear all undo/redo history"""
        self.undo_stack.clear()
        self.redo_stack.clear()


# Concrete command implementations

class MoveItemCommand(Command):
    """Command for moving a zone item"""
    def __init__(self, item, old_pos, new_pos):
        self.item = item
        self.old_pos = old_pos
        self.new_pos = new_pos
    
    def execute(self):
        self.item.setPos(self.new_pos)
    
    def undo(self):
        self.item.setPos(self.old_pos)


class ResizeItemCommand(Command):
    """Command for resizing a zone item"""
    def __init__(self, item, old_rect, new_rect):
        self.item = item
        self.old_rect = old_rect
        self.new_rect = new_rect
    
    def execute(self):
        self.item.setRect(self.new_rect)
    
    def undo(self):
        self.item.setRect(self.old_rect)


class AddItemCommand(Command):
    """Command for adding a zone item"""
    def __init__(self, scene, item):
        self.scene = scene
        self.item = item
    
    def execute(self):
        self.scene.addItem(self.item)
    
    def undo(self):
        self.scene.removeItem(self.item)


class DeleteItemCommand(Command):
    """Command for deleting a zone item"""
    def __init__(self, scene, item, page_zones, zone_model):
        self.scene = scene
        self.item = item
        self.page_zones = page_zones
        self.zone_model = zone_model
    
    def execute(self):
        if self.zone_model in self.page_zones:
            self.page_zones.remove(self.zone_model)
        self.scene.removeItem(self.item)
    
    def undo(self):
        self.page_zones.append(self.zone_model)
        self.scene.addItem(self.item)


class ChangeTextCommand(Command):
    """Command for changing text content"""
    def __init__(self, text_item, old_text, new_text):
        self.text_item = text_item
        self.old_text = old_text
        self.new_text = new_text
    
    def execute(self):
        self.text_item.setPlainText(self.new_text)
    
    def undo(self):
        self.text_item.setPlainText(self.old_text)
