
from PySide6.QtWidgets import QGraphicsScene, QGraphicsView
from PySide6.QtCore import Qt, QLineF, QRectF, Signal, QPointF
from PySide6.QtGui import QPen, QColor, QPainter, QBrush, QTransform

class PAOGraphicsScene(QGraphicsScene):
    def __init__(self, parent=None):
        super().__init__(parent)
        self.grid_size = 20  # Pixels
        self.snap_to_grid = True
        self.show_grid = True  # Afficher la grille
        self.setBackgroundBrush(QBrush(QColor(30, 30, 30))) # Fond gris foncé pour l'éditeur

    def drawBackground(self, painter, rect):
        """Dessine une grille de fond."""
        super().drawBackground(painter, rect)
        
        # Ne dessiner la grille que si activée
        if not self.show_grid:
            return
        
        # Optimisation : ne dessiner que ce qui est visible
        left = int(rect.left()) - (int(rect.left()) % self.grid_size)
        top = int(rect.top()) - (int(rect.top()) % self.grid_size)
        
        lines = []
        
        # Grille verticale
        x = left
        while x < rect.right():
            lines.append(QLineF(x, rect.top(), x, rect.bottom()))
            x += self.grid_size

        # Grille horizontale
        y = top
        while y < rect.bottom():
            lines.append(QLineF(rect.left(), y, rect.right(), y))
            y += self.grid_size

        pen = QPen(QColor(80, 80, 80))  # Gris plus clair pour meilleure visibilité
        pen.setWidth(0) # Cosmetic pen (toujours 1px)
        painter.setPen(pen)
        painter.drawLines(lines)

    def snap_point(self, point: QPointF) -> QPointF:
        """Ajuste un point sur la grille si le snap est activé."""
        if not self.snap_to_grid:
            return point
        
        x = round(point.x() / self.grid_size) * self.grid_size
        y = round(point.y() / self.grid_size) * self.grid_size
        return QPointF(x, y)

class PAOGraphicsView(QGraphicsView):
    def __init__(self, scene, parent=None):
        super().__init__(scene, parent)
        self.setRenderHint(QPainter.Antialiasing)
        self.setRenderHint(QPainter.SmoothPixmapTransform)
        self.setDragMode(QGraphicsView.RubberBandDrag)
        self.setTransformationAnchor(QGraphicsView.AnchorUnderMouse)
        self.setResizeAnchor(QGraphicsView.AnchorUnderMouse)
        self.zoom_factor = 1.15
        
    def wheelEvent(self, event):
        """Gestion du zoom avec Ctrl + Molette."""
        if event.modifiers() & Qt.ControlModifier:
            if event.angleDelta().y() > 0:
                self.scale(self.zoom_factor, self.zoom_factor)
            else:
                self.scale(1 / self.zoom_factor, 1 / self.zoom_factor)
        else:
            super().wheelEvent(event)

    def dragEnterEvent(self, event):
        if event.mimeData().hasUrls():
            event.acceptProposedAction()
        else:
            super().dragEnterEvent(event)

    def dragMoveEvent(self, event):
        if event.mimeData().hasUrls():
            event.acceptProposedAction()
        else:
            super().dragMoveEvent(event)

    def dropEvent(self, event):
        if event.mimeData().hasUrls():
            # Delegate to main window or handle here?
            # Better to handle here for coordinates, but logic is in MainWindow (adding items).
            # We will emit a signal or call a method if we want to keep logic separated, 
            # but for simplicity let's rely on MainWindow having a reference or signal.
            # Actually, `PAOGraphicsScene` is self.scene().
            # Let's emit a signal.
            pass # We'll implement the actual handling via signal in next step to avoid circular deps or complex passing
            
            # For now, let's signal the parent.
            # The MainWindow has the view.
            # We can define a signal `filesDropped(list, QPointF)`
            event.acceptProposedAction()
            self.filesDropped.emit(event.mimeData().urls(), self.mapToScene(event.position().toPoint()))
        else:
            super().dropEvent(event)

    filesDropped = Signal(list, QPointF)
