
from PySide6.QtGui import QIcon, QPixmap
import base64

class Icons:
    # SVGs as simple strings
    
    _NEW_SVG = """<svg viewBox="0 0 24 24" fill="none" stroke="white" stroke-width="2"><path d="M12 5v14M5 12h14"/></svg>"""
    _OPEN_SVG = """<svg viewBox="0 0 24 24" fill="none" stroke="white" stroke-width="2"><path d="M22 19a2 2 0 0 1-2 2H4a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h5l2 3h9a2 2 0 0 1 2 2z"/></svg>"""
    _SAVE_SVG = """<svg viewBox="0 0 24 24" fill="none" stroke="white" stroke-width="2"><path d="M19 21H5a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h11l5 5v11a2 2 0 0 1-2 2z"/><polyline points="17 21 17 13 7 13 7 21"/><polyline points="7 3 7 8 15 8"/></svg>"""
    _PDF_SVG = """<svg viewBox="0 0 24 24" fill="none" stroke="white" stroke-width="2"><path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/><polyline points="14 2 14 8 20 8"/><line x1="16" y1="13" x2="8" y2="13"/><line x1="16" y1="17" x2="8" y2="17"/><polyline points="10 9 9 9 8 9"/></svg>"""
    _TEXT_SVG = """<svg viewBox="0 0 24 24" fill="none" stroke="white" stroke-width="2"><polyline points="4 7 4 4 20 4 20 7"/><line x1="9" y1="20" x2="15" y2="20"/><line x1="12" y1="4" x2="12" y2="20"/></svg>"""
    _IMAGE_SVG = """<svg viewBox="0 0 24 24" fill="none" stroke="white" stroke-width="2"><rect x="3" y="3" width="18" height="18" rx="2" ry="2"/><circle cx="8.5" cy="8.5" r="1.5"/><polyline points="21 15 16 10 5 21"/></svg>"""
    
    @staticmethod
    def _create_icon(svg_str):
        # Trick: embed SVG in data URL or save temp?
        # PySide6 supports SVG loading from memory via QByteArray usually, but QIcon logic can be finicky.
        # Simple approach: render to pixmap via QSvgRenderer or just save/load?
        # Simpler: Use QPixmap from file-like?
        # Actually, QIcon can take a pixmap. 
        # Let's use a temporary approach or valid SVG support.
        # Since we might not have QSvg modules easily, we can check.
        # If QIcon(file_path) with .svg works, we need a file.
        # Alternatively, we can assume standard theme icons if available, but user wants "beautiful icons".
        # Let's try standard theme icons as fallback or specific paths if defined.
        # Wait, the user wants "belles icônes". 
        pass

    # Better: return dict of pixmaps or icons generated at runtime
    # To keep it single-file-ish without deps:
    # We will use QPainter to draw simple vector-like shapes onto QPixmap if SVG is hard,
    # OR we rely on standard Theme icons primarily, but the user said "sad".
    
    # Let's try to just return QIcon.fromTheme if available, else text.
    # BUT, actually, creating QIcon from byte array of SVG is possible.
    
    @staticmethod
    def get_icon(name):
        # Placeholder implementation for now if we want to avoid complex inline SVG handling
        # Using standard keys
        mapping = {
            "new": "document-new",
            "open": "document-open",
            "save": "document-save",
            "pdf": "application-pdf",
            "text": "format-text-bold",
            "image": "image-x-generic"
        }
        
        if QIcon.hasThemeIcon(mapping.get(name, "")):
            return QIcon.fromTheme(mapping[name])
        
        # Fallback: Colored blocks created via pixmap
        pm = QPixmap(32, 32)
        pm.fill(0) # Transparent-ish or black?
        # return QIcon(pm)
        return QIcon() # Returns empty icon

# Revisiting: The user explicitly asked for "beautiful icons". 
# Standard QIcon.fromTheme depends on system. On Windows, often empty or ugly.
# Best approach for "Self-contained python script" is to not have external assets if possible.
# But 'write_to_file' can write .svg files!
# So I will write 6 svg files to a 'resources' folder and load them.

