
from PySide6.QtWidgets import QGraphicsRectItem, QGraphicsItem
from PySide6.QtCore import Qt, QRectF, QPointF, QObject, Signal
from PySide6.QtGui import QPen, QColor, QBrush
from core.geometry import Rect, check_collision
from gui.items.handle_item import HandleItem

class ZoneSignals(QObject):
    geometryChanged = Signal()
    contentChanged = Signal()

class ZoneItem(QGraphicsRectItem):
    def __init__(self, x, y, w, h, zone_model=None):
        super().__init__(0, 0, w, h)
        self.zone_model = zone_model
        self.setPos(x, y)
        
        # Signals proxy since QGraphicsRectItem is not a QObject
        self.signals = ZoneSignals()
        
        # Flags
        self.setFlag(QGraphicsItem.ItemIsMovable, True)
        self.setFlag(QGraphicsItem.ItemIsSelectable, True)
        self.setFlag(QGraphicsItem.ItemSendsGeometryChanges, True)
        
        # Apparence
        self.setPen(QPen(Qt.NoPen))
        self.setBrush(QBrush(QColor(50, 50, 150, 100))) # Bleu semi-transparent par défaut

        # Handles
        self.handles = []
        self._create_handles()
        self.update_handles()
        self.hide_handles()
        self.is_resizing = False

    def _create_handles(self):
        self.handles.append(HandleItem(HandleItem.top_left, self))
        self.handles.append(HandleItem(HandleItem.top_right, self))
        self.handles.append(HandleItem(HandleItem.bottom_left, self))
        self.handles.append(HandleItem(HandleItem.bottom_right, self))
    
    def update_handles(self):
        rect = self.rect()
        for handle in self.handles:
            handle.update_position(rect)

    def hide_handles(self):
        for handle in self.handles:
            handle.hide()

    def show_handles(self):
        for handle in self.handles:
            handle.show()

    def itemChange(self, change, value):
        if change == QGraphicsItem.ItemPositionChange and self.scene():
            if self.is_resizing:
                return value

            new_pos = value
            # Snap to grid
            if hasattr(self.scene(), 'snap_point'):
                new_pos = self.scene().snap_point(new_pos)
            
            # Collision Check
            new_rect = Rect(new_pos.x(), new_pos.y(), self.rect().width(), self.rect().height())
            if self.collides_with_others(new_rect):
                # Don't revert position, just let it overlap. 
                # Ideally, show visual feedback (red border), but user priority is "don't get stuck".
                pass 
                # return self.pos() # Revert to current position (Block movement) - REMOVED
            
            # Emit signal (deferred to avoid recursion issues if needed, but direct is ok here)
            self.signals.geometryChanged.emit()
            return new_pos

        if change == QGraphicsItem.ItemSelectedChange:
            if value:
                self.show_handles()
                self.setPen(QPen(QColor("#4282da"), 2))
            else:
                self.hide_handles()
                self.setPen(QPen(Qt.NoPen))

        return super().itemChange(change, value)

    def collides_with_others(self, my_rect_struct: Rect) -> bool:
        """Vérifie si le rect proposé entre en collision avec d'autres ZoneItem de la scène."""
        if not self.scene():
            return False
            
        for item in self.scene().items():
            if isinstance(item, ZoneItem) and item != self:
                other_pos = item.pos()
                other_rect = item.rect()
                other_rect_struct = Rect(other_pos.x(), other_rect.y(), other_rect.width(), other_rect.height()) # Assuming local 0,0
                # Note: item.rect() est en local coords (0,0,w,h). item.pos() est la position dans la scène.
                # Donc geometry global = pos.x + rect.x (0), pos.y + rect.y (0)
                
                # Correction sur le rect de l'autre item
                # QGraphicsRectItem.rect() renvoie les coords locales.
                # Si l'item a été créé avec (0,0,w,h), alors x,y sont 0,0.
                
                real_other_struct = Rect(other_pos.x() + other_rect.x(), 
                                         other_pos.y() + other_rect.y(), 
                                         other_rect.width(), 
                                         other_rect.height())

                if check_collision(my_rect_struct, real_other_struct):
                    return True
        return False
        
    def interactive_resize(self, handle_type, mouse_pos_scene):
        """
        Gère le redimensionnement interactif via les poignées.
        Calcule le nouveau rect candidat et applique si pas de collision.
        """
        # Snap souris
        if hasattr(self.scene(), 'snap_point'):
            mouse_pos_scene = self.scene().snap_point(mouse_pos_scene)

        current_rect = self.rect() # Local 0,0,w,h
        current_pos = self.pos()   # Scene coords
        
        # Limites actuelles en coordonnées scène
        left = current_pos.x()
        top = current_pos.y()
        right = left + current_rect.width()
        bottom = top + current_rect.height()

        # Calcul des nouvelles limites en fonction du handle
        if handle_type == HandleItem.top_left:
            left = min(mouse_pos_scene.x(), right - 10)
            top = min(mouse_pos_scene.y(), bottom - 10)
        elif handle_type == HandleItem.top_right:
            right = max(mouse_pos_scene.x(), left + 10)
            top = min(mouse_pos_scene.y(), bottom - 10)
        elif handle_type == HandleItem.bottom_left:
            left = min(mouse_pos_scene.x(), right - 10)
            bottom = max(mouse_pos_scene.y(), top + 10)
        elif handle_type == HandleItem.bottom_right:
            right = max(mouse_pos_scene.x(), left + 10)
            bottom = max(mouse_pos_scene.y(), top + 10)
        
        # Reconstruction geometry
        new_x = left
        new_y = top
        new_w = right - left
        new_h = bottom - top
        
        # Test Collision (avec geometry globale)
        # On ne bloque plus le resize sur collision
        # if not self.collides_with_others(candidate_global_rect):
        self.is_resizing = True
        self.setPos(new_x, new_y)
        self.setRect(0, 0, new_w, new_h)
        self.is_resizing = False

    def setRect(self, x, y, w, h):
        super().setRect(x, y, w, h)
        self.update_handles()
        self.signals.geometryChanged.emit()
