
from PySide6.QtWidgets import QMainWindow, QToolBar, QStatusBar, QWidget, QHBoxLayout, QVBoxLayout, QLabel, QFileDialog, QMessageBox, QInputDialog, QSizePolicy
from PySide6.QtCore import Qt, QSettings, QDir, QMimeData, QFileInfo, QSize, QPoint
from PySide6.QtGui import QAction, QIcon, QPen, QBrush, QColor
import os
import shutil
from gui.canvas import PAOGraphicsScene, PAOGraphicsView
from core.document import DocumentModel, PageFormat, PageModel, ZoneModel
from core.geometry import Rect, check_collision
from gui.items.text_item import TextZoneItem
from gui.items.image_item import ImageZoneItem
from gui.items.zone_item import ZoneItem
from gui.panels.properties import PropertiesPanel
from core.serializers import ProjectSerializer, PDFExporter, HTMLExporter, MarkdownExporter
from core.undo_redo import UndoRedoManager

class MainWindow(QMainWindow):
    def __init__(self):
        super().__init__()
        self.setWindowTitle("PAO Zone-Based - Python")
        self.resize(1200, 800)
        
        # Modèle de données (Document vide par défaut)
        self.document = DocumentModel()
        self.current_page = self.document.create_page(PageFormat.A4)
        self.current_filename = None
        self.is_dirty = False # Suivi des modifications
        self.undo_manager = UndoRedoManager()  # Gestionnaire undo/redo

        # Initialisation de l'UI
        self.init_ui()
        self.setup_scene()
        
        # Configuration Drag & Drop
        self.setAcceptDrops(True)
        
        # Restauration des paramètres (fenêtre)
        self.read_settings()

    def init_ui(self):
        # Widget central
        self.central_widget = QWidget()
        self.setCentralWidget(self.central_widget)
        self.layout = QHBoxLayout(self.central_widget)
        self.layout.setContentsMargins(0, 0, 0, 0)
        self.layout.setSpacing(0)

        # Zone Canvas (Centre)
        self.view_container = QWidget()
        self.view_layout = QVBoxLayout(self.view_container)
        self.view_layout.setContentsMargins(0, 0, 0, 0)
        self.layout.addWidget(self.view_container, stretch=1)

        # Panneau latéral droit (Propriétés)
        self.properties_panel = PropertiesPanel()
        self.properties_panel.setFixedWidth(280)
        self.properties_panel.page_color_changed.connect(self.on_page_color_changed)
        self.properties_panel.page_color_all_changed.connect(self.on_page_color_all_changed) # New
        self.layout.addWidget(self.properties_panel)
        
        # Initialiser le panneau avec la page courante
        self.properties_panel.set_page(self.current_page)

        # Barre d'outils
        self.toolbar = QToolBar("Outils")
        self.toolbar.setIconSize(QSize(24, 24))
        self.toolbar.setToolButtonStyle(Qt.ToolButtonIconOnly) # Icônes seulement
        self.addToolBar(self.toolbar)
        
        # Helper icon loader
        def load_icon(name):
            path = os.path.join(os.getcwd(), "resources", f"icon_{name}.svg")
            if os.path.exists(path):
                return QIcon(path)
            return QIcon() # Fallback

        # Actions Fichier
        new_act = QAction(load_icon("new"), "", self)
        new_act.setToolTip("Nouveau Projet")
        new_act.triggered.connect(self.new_project)
        self.toolbar.addAction(new_act)
        
        open_act = QAction(load_icon("open"), "", self)
        open_act.setToolTip("Ouvrir un Projet")
        open_act.triggered.connect(self.open_project)
        self.toolbar.addAction(open_act)
        
        save_act = QAction(load_icon("save"), "", self)
        save_act.setToolTip("Sauvegarder le Projet")
        save_act.triggered.connect(self.save_project)
        self.toolbar.addAction(save_act)

        export_act = QAction(load_icon("pdf"), "", self)
        export_act.setToolTip("Exporter en PDF")
        export_act.triggered.connect(self.export_pdf)
        self.toolbar.addAction(export_act)
        
        export_html_act = QAction(load_icon("html"), "", self)
        export_html_act.setToolTip("Exporter en HTML")
        export_html_act.triggered.connect(self.export_html)
        self.toolbar.addAction(export_html_act)

        export_md_act = QAction(load_icon("markdown"), "", self)
        export_md_act.setToolTip("Exporter en Markdown")
        export_md_act.triggered.connect(self.export_markdown)
        self.toolbar.addAction(export_md_act)
        
        self.toolbar.addSeparator()
        
        # Actions Undo/Redo
        undo_act = QAction(load_icon("undo"), "", self)
        undo_act.setToolTip("Annuler (Ctrl+Z)")
        undo_act.setShortcut("Ctrl+Z")
        undo_act.triggered.connect(self.undo)
        self.toolbar.addAction(undo_act)
        self.undo_action = undo_act  # Store reference to update enabled state
        
        redo_act = QAction(load_icon("redo"), "", self)
        redo_act.setToolTip("Refaire (Ctrl+Y)")
        redo_act.setShortcut("Ctrl+Y")
        redo_act.triggered.connect(self.redo)
        self.toolbar.addAction(redo_act)
        self.redo_action = redo_act
        
        self.toolbar.addSeparator()
        
        # Actions Outils
        add_text_act = QAction(load_icon("text"), "", self)
        add_text_act.setToolTip("Ajouter une Zone de Texte")
        add_text_act.triggered.connect(self.add_text_zone)
        self.toolbar.addAction(add_text_act)
        
        add_img_act = QAction(load_icon("image"), "", self)
        add_img_act.setToolTip("Ajouter une Image")
        add_img_act.triggered.connect(self.add_image_zone)
        self.toolbar.addAction(add_img_act)
        
        del_item_act = QAction(load_icon("trash"), "", self)
        del_item_act.setToolTip("Supprimer l'élément sélectionné")
        del_item_act.triggered.connect(self.delete_selected_item)
        self.toolbar.addAction(del_item_act)
        
        self.toolbar.addSeparator()
        
        # Actions Zoom et Grille
        zoom_in_act = QAction(load_icon("zoom_in"), "", self)
        zoom_in_act.setToolTip("Zoomer (Ctrl+Molette)")
        zoom_in_act.triggered.connect(self.zoom_in)
        self.toolbar.addAction(zoom_in_act)
        
        zoom_out_act = QAction(load_icon("zoom_out"), "", self)
        zoom_out_act.setToolTip("Dézoomer")
        zoom_out_act.triggered.connect(self.zoom_out)
        self.toolbar.addAction(zoom_out_act)
        
        zoom_reset_act = QAction(load_icon("zoom_reset"), "", self)
        zoom_reset_act.setToolTip("Réinitialiser le zoom (100%)")
        zoom_reset_act.triggered.connect(self.zoom_reset)
        self.toolbar.addAction(zoom_reset_act)
        
        grid_toggle_act = QAction(load_icon("grid"), "", self)
        grid_toggle_act.setToolTip("Grille : Afficher/Masquer (actuellement activée)")
        grid_toggle_act.setCheckable(True)
        grid_toggle_act.setChecked(True)  # Grille activée par défaut
        grid_toggle_act.triggered.connect(self.toggle_grid)
        self.toolbar.addAction(grid_toggle_act)
        self.grid_action = grid_toggle_act
        
        self.toolbar.addSeparator()
        
        # Actions Alignement
        align_left_act = QAction(load_icon("align_left"), "", self)
        align_left_act.setToolTip("Aligner à gauche (2+ zones)")
        align_left_act.triggered.connect(self.align_left)
        self.toolbar.addAction(align_left_act)
        
        align_center_act = QAction(load_icon("align_center"), "", self)
        align_center_act.setToolTip("Centrer horizontalement (2+ zones)")
        align_center_act.triggered.connect(self.align_center_h)
        self.toolbar.addAction(align_center_act)
        
        align_right_act = QAction(load_icon("align_right"), "", self)
        align_right_act.setToolTip("Aligner à droite (2+ zones)")
        align_right_act.triggered.connect(self.align_right)
        self.toolbar.addAction(align_right_act)
        
        align_top_act = QAction(load_icon("align_top"), "", self)
        align_top_act.setToolTip("Aligner en haut (2+ zones)")
        align_top_act.triggered.connect(self.align_top)
        self.toolbar.addAction(align_top_act)
        
        align_middle_act = QAction(load_icon("align_middle"), "", self)
        align_middle_act.setToolTip("Centrer verticalement (2+ zones)")
        align_middle_act.triggered.connect(self.align_middle)
        self.toolbar.addAction(align_middle_act)
        
        align_bottom_act = QAction(load_icon("align_bottom"), "", self)
        align_bottom_act.setToolTip("Aligner en bas (2+ zones)")
        align_bottom_act.triggered.connect(self.align_bottom)
        self.toolbar.addAction(align_bottom_act)
        
        # Spacer
        empty = QWidget()
        empty.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Preferred)
        self.toolbar.addWidget(empty)
        
        # Page Navigation Toolbar (Bottom ?) or just add to main toolbar
        # Let's add a separate toolbar for Pages at the bottom or top
        self.page_toolbar = QToolBar("Navigation Pages")
        self.page_toolbar.setIconSize(QSize(24, 24))
        self.addToolBar(Qt.BottomToolBarArea, self.page_toolbar)
        
        # Actions Page
        self.act_prev = QAction(load_icon("prev"), "Précédent", self)
        self.act_prev.triggered.connect(self.prev_page)
        self.page_toolbar.addAction(self.act_prev)
        
        self.lbl_page_num = QLabel(" Page 1 / 1 ")
        self.page_toolbar.addWidget(self.lbl_page_num)
        
        self.act_next = QAction(load_icon("next"), "Suivant", self)
        self.act_next.triggered.connect(self.next_page)
        self.page_toolbar.addAction(self.act_next)
        
        self.page_toolbar.addSeparator()
        
        act_add_page = QAction(load_icon("add_page"), "Ajouter Page", self)
        act_add_page.triggered.connect(self.add_page)
        self.page_toolbar.addAction(act_add_page)
        
        act_del_page = QAction(load_icon("del_page"), "Supprimer Page", self)
        act_del_page.triggered.connect(self.delete_page)
        self.page_toolbar.addAction(act_del_page)
        
        self.page_toolbar.addSeparator()
        
        act_mv_left = QAction(load_icon("move_left"), "Déplacer Gauche", self)
        act_mv_left.triggered.connect(self.move_page_left)
        self.page_toolbar.addAction(act_mv_left)
        
        act_mv_right = QAction(load_icon("move_right"), "Déplacer Droite", self)
        act_mv_right.triggered.connect(self.move_page_right)
        self.page_toolbar.addAction(act_mv_right)
        
        self.page_toolbar.addSeparator()
        
        act_toc = QAction("Insérer TDM", self) # No icon yet, text is fine or use generic
        act_toc.triggered.connect(self.insert_toc)
        self.page_toolbar.addAction(act_toc)

        # Barre d'état
        self.setStatusBar(QStatusBar())

    def setup_scene(self):
        self.scene = PAOGraphicsScene(self)
        self.view = PAOGraphicsView(self.scene)
        self.view.filesDropped.connect(self.on_files_dropped)
        self.view_layout.addWidget(self.view)
        
        # Connexion sélection
        self.scene.selectionChanged.connect(self.on_selection_changed)
        
        self.refresh_view()

    def refresh_view(self):
        """Redessine la scène en fonction du document courant."""
        self.scene.clear()
        
        # Définir la zone de la page
        bg_color = QColor(self.current_page.background_color) if hasattr(self.current_page, 'background_color') else Qt.white
        self.page_rect = self.scene.addRect(0, 0, self.current_page.width, self.current_page.height, 
                                            QPen(Qt.black), QBrush(bg_color))
        
        # Indicateur de numéro de page (Visuel seulement)
        page_num_txt = f"Page {self.document.active_page_index + 1} / {len(self.document.pages)}"
        page_lbl = self.scene.addText(page_num_txt)
        page_lbl.setDefaultTextColor(Qt.gray)
        # Positionner en bas au centre
        txt_rect = page_lbl.boundingRect()
        page_lbl.setPos((self.current_page.width - txt_rect.width()) / 2, self.current_page.height + 10) # En dessous de la page
        
        # Ajuster la vue
        self.scene.setSceneRect(-50, -50, self.current_page.width + 100, self.current_page.height + 100)
        
        # Recharger les items depuis le modèle
        for zone_model in self.current_page.zones:
            item = None
            if zone_model.type == "text":
                item = TextZoneItem(zone_model.x, zone_model.y, zone_model.width, zone_model.height, zone_model.content)
                item.zone_model = zone_model
                # Rétablissement du Style
                if zone_model.style:
                    font = item.text_item.font()
                    family = zone_model.style.get("font_family")
                    size = zone_model.style.get("font_size")
                    text_color = zone_model.style.get("text_color")
                    
                    if family:
                        font.setFamily(family)
                    if size:
                        font.setPointSize(size)
                    if text_color:
                        item.text_item.setDefaultTextColor(QColor(text_color))
                        
                    item.text_item.setFont(font)
                self.scene.addItem(item)
            elif zone_model.type == "image":
                # Gestion chemin relatif
                if zone_model.content and not os.path.isabs(zone_model.content) and self.current_filename:
                    # Reconstruire le chemin absolu
                    project_dir = os.path.dirname(self.current_filename)
                    abs_path = os.path.join(project_dir, zone_model.content)
                    item = ImageZoneItem(zone_model.x, zone_model.y, zone_model.width, zone_model.height, abs_path)
                else:
                    item = ImageZoneItem(zone_model.x, zone_model.y, zone_model.width, zone_model.height, zone_model.content)
                
                item.zone_model = zone_model
                self.scene.addItem(item)
                
            # Connexion pour le suivi des modifs
            # Note: `item` is now the newly created item
            if isinstance(item, ZoneItem):
                item.signals.geometryChanged.connect(self.mark_dirty)
                # item.signals.contentChanged.connect(self.mark_dirty) # TODO implémenter contentChanged dans TextItem
            
    def on_page_color_changed(self, color_hex):
        """Callback changement couleur de fond."""
        if self.current_page:
            self.current_page.background_color = color_hex
            self.is_dirty = True
            self.refresh_view() # Re-dessine le fond

    def mark_dirty(self):
        self.is_dirty = True
        self.setWindowTitle(f"PAO Zone-Based - {self.get_window_title_suffix()} *")

    def get_window_title_suffix(self):
        if self.current_filename:
            return os.path.basename(self.current_filename)
        return "Nouveau"

    def closeEvent(self, event):
        if self.maybe_save():
            self.write_settings()
            event.accept()
        else:
            event.ignore()
            
    def keyPressEvent(self, event):
        if event.key() == Qt.Key_Delete:
            self.delete_selected_item()
        else:
            super().keyPressEvent(event)

    def read_settings(self):
        settings = QSettings()
        pos = settings.value("pos", QPoint(200, 200))
        size = settings.value("size", QSize(1200, 800))
        self.resize(size)
        self.move(pos)
        
        # Auto-reopen last project
        last_project = settings.value("last_project_path", "")
        if last_project and os.path.exists(last_project):
            # Defer slightly to ensure UI is ready? No need usually.
            self.open_project_file(last_project)

    def write_settings(self):
        settings = QSettings()
        settings.setValue("pos", self.pos())
        settings.setValue("size", self.size())
        if self.current_filename:
            settings.setValue("last_project_path", self.current_filename)

    def maybe_save(self, force_save_as=False):
        if not self.is_dirty and not force_save_as:
            return True
        
        ret = QMessageBox.question(self, "Sauvegarder ?",
                                   "Le projet contient des modifications non sauvegardées.\nVoulez-vous le sauvegarder ?",
                                   QMessageBox.Save | QMessageBox.Discard | QMessageBox.Cancel)
        
        if ret == QMessageBox.Save:
            return self.save_project(force_save_as)
        elif ret == QMessageBox.Cancel:
            return False
        return True
    
    # Drag & Drop Events
    # Drag & Drop Handler (Signal from View)
    def on_files_dropped(self, urls, scene_pos):
        for url in urls:
            file_path = url.toLocalFile()
            ext = os.path.splitext(file_path)[1].lower()
            
            if ext == ".pao":
                if self.maybe_save():
                    self.open_project_file(file_path)
            elif ext in [".png", ".jpg", ".jpeg", ".bmp"]:
                # Copie image dans projet
                final_path = self.import_image(file_path)
                
                item = ImageZoneItem(scene_pos.x(), scene_pos.y(), 200, 200, final_path)
                item.signals.geometryChanged.connect(self.mark_dirty)
                self.scene.addItem(item)
                self.scene.clearSelection()
                item.setSelected(True)
                self.mark_dirty()

    def import_image(self, source_path):
        """Copie l'image dans le dossier 'images' du projet et retourne le chemin absolu."""
        if not self.current_filename:
            return source_path # Pas de projet sauvegardé, on garde le lien absolu temporairement
            
        project_dir = os.path.dirname(self.current_filename)
        images_dir = os.path.join(project_dir, "images")
        
        if not os.path.exists(images_dir):
            os.makedirs(images_dir)
            
        filename = os.path.basename(source_path)
        dest_path = os.path.join(images_dir, filename)
        
        # Eviter d'écraser si existe déjà (ou alors on écrase, selon préférence) -> ici on écrase pour simplifier
        shutil.copy2(source_path, dest_path)
        
        return dest_path

    def new_project(self):
        if not self.maybe_save():
            return

        name, ok = QInputDialog.getText(self, "Nouveau Projet", "Nom du projet :")
        if ok and name:
            # Nettoyage nom (basique)
            name = "".join(c for c in name if c.isalnum() or c in (' ', '-', '_')).strip()
            if not name:
                return

            base_dir = os.path.join(os.getcwd(), "projets")
            if not os.path.exists(base_dir):
                os.makedirs(base_dir)
            
            project_dir = os.path.join(base_dir, name)
            if not os.path.exists(project_dir):
                os.makedirs(project_dir)
            else:
                 QMessageBox.warning(self, "Attention", f"Le projet '{name}' existe déjà. Il sera ouvert/écrasé.")
            
            self.document = DocumentModel()
            self.properties_panel.set_document(self.document)
            self.current_page = self.document.create_page(PageFormat.A4)
            self.current_filename = os.path.join(project_dir, name + ".pao")
            
            # Sauvegarder immédiatement le fichier vide pour initier la structure
            ProjectSerializer.save_to_file(self.document, self.current_filename)
            
            self.refresh_view()
            self.is_dirty = False
            self.setWindowTitle(f"PAO Zone-Based - {self.get_window_title_suffix()}")

    def save_project(self, force_save_as=False):
        # Sync Scene -> Model
        self.update_model_from_view()
        
        if not self.current_filename or force_save_as:
            return self.save_as_project()
                
        ProjectSerializer.save_to_file(self.document, self.current_filename)
        self.is_dirty = False
        self.setWindowTitle(f"PAO Zone-Based - {self.get_window_title_suffix()}")
        self.statusBar().showMessage(f"Projet sauvegardé : {self.current_filename}", 3000)
        return True
        
    def save_as_project(self):
        # Cas où on veut "Enregistrer sous" ou si pas de filename
        base_dir = os.path.join(os.getcwd(), "projets")
        if not os.path.exists(base_dir):
            os.makedirs(base_dir)
            
        path, _ = QFileDialog.getSaveFileName(self, "Sauvegarder le projet", base_dir, "PAO Files (*.pao)")
        if not path:
            return False
        
        # Même logique dossier...
        file_info = QFileInfo(path)
        project_name = file_info.baseName()
        parent_dir_name = os.path.basename(file_info.absolutePath())
        
        final_path = path
        if parent_dir_name != project_name:
             new_dir = os.path.join(file_info.absolutePath(), project_name)
             if not os.path.exists(new_dir):
                 os.makedirs(new_dir)
             final_path = os.path.join(new_dir, project_name + ".pao")
        
        self.current_filename = final_path
        return self.save_project()

    def open_project(self):
        if not self.maybe_save():
            return
            
        base_dir = os.path.join(os.getcwd(), "projets")
        if not os.path.exists(base_dir):
            os.makedirs(base_dir) # Créer si existe pas pour éviter erreur
            
        # Lister les dossiers
        projects = []
        for entry in os.scandir(base_dir):
            if entry.is_dir():
                projects.append(entry.name)
        
        if not projects:
            QMessageBox.information(self, "Ouvrir", "Aucun projet trouvé dans le dossier 'projets'.")
            return

        item, ok = QInputDialog.getItem(self, "Ouvrir un projet", "Sélectionner un projet :", projects, 0, False)
        if ok and item:
            # Chercher le .pao
            possible_path = os.path.join(base_dir, item, item + ".pao")
            if os.path.exists(possible_path):
                self.open_project_file(possible_path)
            else:
                # Fallback : chercher n'importe quel .pao
                found = False
                for f in os.listdir(os.path.join(base_dir, item)):
                    if f.endswith(".pao"):
                        self.open_project_file(os.path.join(base_dir, item, f))
                        found = True
                        break
                if not found:
                     QMessageBox.warning(self, "Erreur", f"Aucun fichier .pao trouvé dans le projet '{item}'.")

    def open_project_file(self, path):
        try:
            self.document = ProjectSerializer.load_from_file(path)
            # Validate/Sanitize active_page_index
            if not self.document.pages:
                 self.document.create_page()
            
            if self.document.pages:
                 self.document.pages[0] # Just check
            
            # Pass document to properties for numbering
            self.properties_panel.set_document(self.document)
            
            if self.document.active_page_index >= len(self.document.pages):
                self.document.active_page_index = 0
                
            self.current_filename = path
            
            # Update View and Label
            self.refresh_view_for_page()
            
            self.is_dirty = False
            self.setWindowTitle(f"PAO Zone-Based - {self.get_window_title_suffix()}")
        except Exception as e:
            QMessageBox.critical(self, "Erreur", f"Impossible d'ouvrir le fichier : {e}")

    def export_pdf(self):
        self.update_model_from_view()
        
        # Default directory: Project folder if saved, else generic
        default_dir = ""
        if self.current_filename:
            default_dir = os.path.dirname(self.current_filename)
        
        path, _ = QFileDialog.getSaveFileName(self, "Exporter PDF", default_dir, "PDF Files (*.pdf)")
        if path:
            PDFExporter.export(self.document, path, default_dir)
            self.statusBar().showMessage("PDF exporté avec succès", 3000)

    def export_html(self):
        """Exporter en HTML"""
        self.update_model_from_view()
        
        default_dir = ""
        if self.current_filename:
            default_dir = os.path.dirname(self.current_filename)

        file_path, _ = QFileDialog.getSaveFileName(self, "Exporter en HTML", default_dir, "HTML Files (*.html)")
        if not file_path:
            return

        try:
            HTMLExporter.export(self.document, file_path, default_dir)
            QMessageBox.information(self, "Export HTML", "Export HTML réussi !")
        except Exception as e:
            QMessageBox.critical(self, "Erreur Export", f"Erreur lors de l'export HTML: {str(e)}")

    def export_markdown(self):
        """Exporter en Markdown"""
        self.update_model_from_view()
        
        default_dir = ""
        if self.current_filename:
            default_dir = os.path.dirname(self.current_filename)

        file_path, _ = QFileDialog.getSaveFileName(self, "Exporter en Markdown", default_dir, "Markdown Files (*.md)")
        if not file_path:
            return

        try:
            MarkdownExporter.export(self.document, file_path, default_dir)
            QMessageBox.information(self, "Export Markdown", "Export Markdown réussi !")
        except Exception as e:
            QMessageBox.critical(self, "Erreur Export", f"Erreur lors de l'export Markdown: {str(e)}")

    def update_model_from_view(self):
        """Met à jour le modèle de données à partir des items de la scène."""
        new_zones = []
        
        # Récupérer les items triés par z-value (stack order)
        # items() retourne en ordre descendant (top first). On veut sauvegarder l'ordre de rendu.
        # Donc pour la liste, si on veut que le premier soit le fond, on doit inverser.
        # Ou alors on stocke le z-index explicitement. Les items Qt ont une zValue().
        
        items = [i for i in self.scene.items() if isinstance(i, ZoneItem)]
        # Tri des items par zValue croissante pour cohérence (même si items() a un ordre)
        # items.sort(key=lambda i: i.zValue()) # Optional, items() usually z-order sorted
        
        # items() return items in descending stacking order (topmost first).
        # We want to save them so that loading restores them correcty.
        # If we append to list, index 0 is bottom-most usually preferable for simple rendering loops.
        # So we should reverse `items`.
        items.reverse() # Now bottom-most first
        
        for item in items:
            if isinstance(item, TextZoneItem):
                if item.zone_model:
                     zm = item.zone_model
                else:
                     zm = ZoneModel(type="text")
                
                rect = item.rect()
                pos = item.pos()
                zm.x = pos.x()
                zm.y = pos.y()
                zm.width = rect.width()
                zm.height = rect.height()
                zm.content = item.text_item.toPlainText()
                # Style font
                zm.style = {
                    "font_family": item.text_item.font().family(),
                    "font_size": item.text_item.font().pointSize(),
                    "text_color": item.text_item.defaultTextColor().name(),
                    "bold": item.text_item.font().bold(),
                    "italic": item.text_item.font().italic(),
                    "underline": item.text_item.font().underline()
                }
                new_zones.append(zm)
                
            elif isinstance(item, ImageZoneItem):
                if item.zone_model:
                     zm = item.zone_model
                else:
                     zm = ZoneModel(type="image")
                
                rect = item.rect()
                pos = item.pos()
                zm.x = pos.x()
                zm.y = pos.y()
                zm.width = rect.width()
                zm.height = rect.height()
                
                # Relativisation du chemin si dans le projet
                if self.current_filename and item.image_path:
                    project_dir = os.path.dirname(self.current_filename)
                    try:
                        zm.content = os.path.relpath(item.image_path, project_dir)
                    except ValueError:
                        zm.content = item.image_path # Pas sur le même drive
                else:
                    zm.content = item.image_path
                    
                new_zones.append(zm)
        
        self.current_page.zones = new_zones

    def on_selection_changed(self):
        items = self.scene.selectedItems()
        if items:
            # On prend le premier item (sélection unique préférée pour l'instant)
            self.properties_panel.set_item(items[0])
        else:
            self.properties_panel.set_item(None)

    # --- Page Management ---

    def update_page_label(self):
        total = len(self.document.pages)
        current = self.document.active_page_index + 1
        self.lbl_page_num.setText(f" Page {current} / {total} ")

    def prev_page(self):
        if self.document.active_page_index > 0:
            self.update_model_from_view() # Save current page state
            self.document.active_page_index -= 1
            self.refresh_view_for_page()

    def next_page(self):
        if self.document.active_page_index < len(self.document.pages) - 1:
            self.update_model_from_view() # Save current page state
            self.document.active_page_index += 1
            self.refresh_view_for_page()

    def add_page(self):
        self.update_model_from_view() # Save current
        self.document.create_page()
        self.document.active_page_index = len(self.document.pages) - 1
        self.refresh_view()
        self.mark_dirty()

    def on_page_color_all_changed(self, color_hex):
        """Callback changement couleur de fond pour TOUTES les pages."""
        self.update_model_from_view()
        
        # Apply to all pages
        for page in self.document.pages:
            page.background_color = color_hex
            
        # Global Contrast Check
        bg = QColor(color_hex)
        conflict = False
        for page in self.document.pages:
            for zone in page.zones:
                if zone.type == "text":
                    txt_col = QColor(zone.style.get("text_color", "#000000"))
                    if txt_col == bg:
                        conflict = True
                        break
            if conflict: break
        
        if conflict:
            QMessageBox.warning(self, "Attention", "La couleur de fond choisie masque certains textes sur une ou plusieurs pages !")
            
        self.refresh_view()
        self.mark_dirty()

    def delete_page(self):
        if len(self.document.pages) <= 1:
            QMessageBox.warning(self, "Attention", "Impossible de supprimer la dernière page.")
            return

        reply = QMessageBox.question(self, "Confirmation", "Voulez-vous vraiment supprimer cette page ?",
                                     QMessageBox.Yes | QMessageBox.No, QMessageBox.No)
        
        if reply == QMessageBox.Yes:
            idx = self.document.active_page_index
            self.document.pages.pop(idx)
            if self.document.active_page_index >= len(self.document.pages):
                self.document.active_page_index = len(self.document.pages) - 1
            self.refresh_view_for_page()
            self.mark_dirty()

            self.refresh_view_for_page()
            self.mark_dirty()

    def delete_selected_item(self):
        items = self.scene.selectedItems()
        if not items:
            return
            
        # Confirm?
        # reply = QMessageBox.question(self, "Supprimer", "Supprimer l'élément sélectionné ?", QMessageBox.Yes | QMessageBox.No)
        # if reply != QMessageBox.Yes: return
        
        for item in items:
            if isinstance(item, ZoneItem):
                # Remove from model
                # But item.zone_model is the link
                # We need to find it in current_page.zones
                if item.zone_model in self.current_page.zones:
                    self.current_page.zones.remove(item.zone_model)
                self.scene.removeItem(item)
                
        self.mark_dirty()
        self.properties_panel.set_item(None)

    def move_page_left(self):
        idx = self.document.active_page_index
        if idx > 0:
            self.update_model_from_view()
            self.document.move_page(idx, idx - 1)
            self.refresh_view_for_page()
            self.mark_dirty()

    def move_page_right(self):
        idx = self.document.active_page_index
        if idx < len(self.document.pages) - 1:
            self.update_model_from_view()
            self.document.move_page(idx, idx + 1)
            self.refresh_view_for_page()
            self.mark_dirty()
            
    def insert_toc(self):
        # Insert a special text zone for TOC
        w, h = 400, 300
        x, y = self.find_free_position(w, h)
        
        toc_content = "Table des Matières\n\n(Cliquez sur 'Actualiser TDM')"
        
        zone_model = ZoneModel(type="text", x=x, y=y, width=w, height=h, content=toc_content)
        zone_model.role = "toc" # Mark as TOC
        zone_model.style = {"font_size": 14, "font_family": "Arial"}
        
        # Ensure we attach to current page (model sync will handle it if we add to scene item too)
        # But best to add to page directly then create Item
        self.current_page.zones.append(zone_model)
        
        item = TextZoneItem(x, y, w, h, toc_content)
        item.zone_model = zone_model
        
        self.scene.addItem(item)
        self.mark_dirty()
        self.refresh_toc() # Auto-refresh immediately

    def refresh_toc(self):
        # Sync current page first to ensure latest edits are captured in model
        self.update_model_from_view()
        
        # Scan pages for headers
        toc_lines = ["Table des Matières\n"]
        
        for i, page in enumerate(self.document.pages):
            page_num = i + 1
            for zone in page.zones:
                if zone.type == "text" and zone.role in ["heading1", "heading2"]:
                     # Clean content (remove newlines)
                     title = zone.content.replace('\n', ' ').strip()
                     prefix = "- " if zone.role == "heading2" else ""
                     toc_lines.append(f"{prefix}{title} ...... p.{page_num}")
        
        new_content = "\n".join(toc_lines)
        
        # Update ALL TOC zones
        updated = False
        for page in self.document.pages:
            for zone in page.zones:
                if zone.type == "text" and zone.role == "toc":
                    zone.content = new_content
                    updated = True
        
        if updated:
            self.refresh_view() # Reload to show new text
            self.settings_dirty = True # Mark saved settings dirty or project dirty? PROJECT
            self.mark_dirty()
            self.statusBar().showMessage("Table des matières actualisée", 3000)

    def refresh_view_for_page(self):
        # Update current_page based on index
        self.current_page = self.document.pages[self.document.active_page_index]
        self.update_page_label()
        self.refresh_view()

    def find_free_position(self, width, height, start_x=50, start_y=50, step=20):
        """Cherche une position libre en décalant progressivement."""
        x, y = start_x, start_y
        max_attempts = 50
        
        for _ in range(max_attempts):
            candidate_rect = Rect(x, y, width, height)
            collision = False
            for item in self.scene.items():
                if isinstance(item, ZoneItem):
                    # Check global collision
                    other_pos = item.pos()
                    other_rect = item.rect()
                    # Attention: item.rect() local vs global pos
                    # rect global = pos + rect local
                    other_global = Rect(other_pos.x() + other_rect.x(), 
                                        other_pos.y() + other_rect.y(), 
                                        other_rect.width(), other_rect.height())
                    
                    if check_collision(candidate_rect, other_global):
                        collision = True
                        break
            
            if not collision:
                return x, y
            
            x += step
            y += step
            
        return start_x + 20, start_y + 20 # Fallback simple offset

    def add_text_zone(self):
        w, h = 200, 100
        x, y = self.find_free_position(w, h)
        
        item = TextZoneItem(x, y, w, h)
        item.signals.geometryChanged.connect(self.mark_dirty)
        self.scene.addItem(item)
        
        # Sélectionner
        self.scene.clearSelection()
        item.setSelected(True)

    def add_image_zone(self):
        file_path, _ = QFileDialog.getOpenFileName(self, "Choisir une image", "", "Images (*.png *.jpg *.jpeg *.bmp)")
        if file_path:
            # Import image
            final_path = self.import_image(file_path)
            
            w, h = 200, 200
            x, y = self.find_free_position(w, h, 100, 100)
            
            item = ImageZoneItem(x, y, w, h, final_path)
            item.signals.geometryChanged.connect(self.mark_dirty)
            self.scene.addItem(item)
            self.scene.clearSelection()
            item.setSelected(True)
            self.mark_dirty()

    def align_left(self):
        """Aligner les zones sélectionnées à gauche"""
        items = [item for item in self.scene.selectedItems() if isinstance(item, ZoneItem)]
        if len(items) < 2:
            return
        
        # Trouver la coordonnée X minimale (bord gauche le plus à gauche)
        min_x = min(item.pos().x() for item in items)
        
        for item in items:
            item.setPos(min_x, item.pos().y())
        self.mark_dirty()
    
    def align_right(self):
        """Aligner les zones sélectionnées à droite"""
        items = [item for item in self.scene.selectedItems() if isinstance(item, ZoneItem)]
        if len(items) < 2:
            return
        
        # Trouver la coordonnée X maximale (bord droit le plus à droite)
        # pos().x() + rect().width() assume que item est positionné par pos() et sa taille est dans rect()
        max_right = max(item.pos().x() + item.rect().width() for item in items)
        
        for item in items:
            new_x = max_right - item.rect().width()
            item.setPos(new_x, item.pos().y())
        self.mark_dirty()
    
    def align_center_h(self):
        """Centrer les zones sélectionnées horizontalement"""
        items = [item for item in self.scene.selectedItems() if isinstance(item, ZoneItem)]
        if len(items) < 2:
            return
        
        min_x = min(item.pos().x() for item in items)
        max_right = max(item.pos().x() + item.rect().width() for item in items)
        center_x = (min_x + max_right) / 2
        
        for item in items:
            new_x = center_x - item.rect().width() / 2
            item.setPos(new_x, item.pos().y())
        self.mark_dirty()
    
    def align_top(self):
        """Aligner les zones sélectionnées en haut"""
        items = [item for item in self.scene.selectedItems() if isinstance(item, ZoneItem)]
        if len(items) < 2:
            return
        
        min_y = min(item.pos().y() for item in items)
        
        for item in items:
            item.setPos(item.pos().x(), min_y)
        self.mark_dirty()
    
    def align_bottom(self):
        """Aligner les zones sélectionnées en bas"""
        items = [item for item in self.scene.selectedItems() if isinstance(item, ZoneItem)]
        if len(items) < 2:
            return
        
        max_bottom = max(item.pos().y() + item.rect().height() for item in items)
        
        for item in items:
            new_y = max_bottom - item.rect().height()
            item.setPos(item.pos().x(), new_y)
        self.mark_dirty()
    
    def align_middle(self):
        """Centrer les zones sélectionnées verticalement"""
        items = [item for item in self.scene.selectedItems() if isinstance(item, ZoneItem)]
        if len(items) < 2:
            return
        
        min_y = min(item.pos().y() for item in items)
        max_bottom = max(item.pos().y() + item.rect().height() for item in items)
        center_y = (min_y + max_bottom) / 2
        
        for item in items:
            new_y = center_y - item.rect().height() / 2
            item.setPos(item.pos().x(), new_y)
        self.mark_dirty()

    def zoom_in(self):
        """Zoomer dans la vue"""
        self.view.scale(1.2, 1.2)
    
    def zoom_out(self):
        """Dézoomer dans la vue"""
        self.view.scale(1/1.2, 1/1.2)
    
    def zoom_reset(self):
        """Réinitialiser le zoom à 100%"""
        self.view.resetTransform()
    
    def toggle_grid(self):
        """Afficher/masquer la grille"""
        is_checked = self.grid_action.isChecked()
        self.scene.show_grid = is_checked
        self.scene.snap_to_grid = is_checked
        self.scene.update()  # Rafraîchir la scène pour redessiner la grille

    def undo(self):
        """Annuler la dernière action"""
        if self.undo_manager.undo():
            self.mark_dirty()
            self.update_undo_redo_actions()
            self.statusBar().showMessage("Action annulée", 2000)

    def redo(self):
        """Refaire l'action annulée"""
        if self.undo_manager.redo():
            self.mark_dirty()
            self.update_undo_redo_actions()
            self.statusBar().showMessage("Action refaite", 2000)

    def update_undo_redo_actions(self):
        """Met à jour l'état des boutons undo/redo"""
        self.undo_action.setEnabled(self.undo_manager.can_undo())
        self.redo_action.setEnabled(self.undo_manager.can_redo())

