
from PySide6.QtWidgets import (QWidget, QVBoxLayout, QFormLayout, QDoubleSpinBox, 
                               QLabel, QGroupBox, QFontComboBox, QSpinBox, 
                               QPushButton, QFileDialog, QColorDialog, QComboBox)
from PySide6.QtGui import QColor, QPalette
from PySide6.QtCore import Qt, Signal
from gui.items.zone_item import ZoneItem
from gui.items.text_item import TextZoneItem
from gui.items.image_item import ImageZoneItem
from gui.utils.spell_checker import SpellHighlighter
from gui.utils.spell_text_edit import SpellCheckTextEdit


class PropertiesPanel(QWidget):
    page_color_changed = Signal(str) # Emits new hex color
    page_color_all_changed = Signal(str) # New signal

    def __init__(self, parent=None):
        super().__init__(parent)
        self.current_item = None
        self.current_item = None
        self.current_page = None # Reference to page model
        self.document = None # Reference to document model
        self.init_ui()

    def init_ui(self):
        self.layout = QVBoxLayout(self)
        self.layout.setAlignment(Qt.AlignTop)

        # Groupe Document / Page
        self.page_group = QGroupBox("Propriétés Document")
        self.page_layout = QFormLayout()
        
        self.btn_page_color = QPushButton("Couleur de Fond (Page Actuelle)")
        self.btn_page_color.clicked.connect(self.on_page_color_clicked)
        
        self.btn_page_color_all = QPushButton("Couleur de Fond (Tout le Doc)")
        self.btn_page_color_all.clicked.connect(self.on_page_color_all_clicked)
        
        # Numérotation
        self.combo_num_style = QComboBox()
        self.combo_num_style.addItems(["1 / Total", "1"])
        self.combo_num_style.currentTextChanged.connect(self.on_num_style_changed)
        
        self.spin_num_start = QSpinBox()
        self.spin_num_start.setRange(1, 999)
        self.spin_num_start.setValue(1)
        self.spin_num_start.setPrefix("Dès la page ")
        self.spin_num_start.valueChanged.connect(self.on_num_start_changed)
        
        self.page_layout.addRow(self.btn_page_color)
        self.page_layout.addRow(self.btn_page_color_all)
        self.page_layout.addRow("Style Numéro:", self.combo_num_style)
        self.page_layout.addRow("Affichage:", self.spin_num_start)
        
        self.page_group.setLayout(self.page_layout)
        self.layout.addWidget(self.page_group)

        # Groupe Géométrie
        self.geo_group = QGroupBox("Géométrie")
        self.geo_layout = QFormLayout()
        
        self.spin_x = self._create_spinbox(self.on_geo_changed)
        self.spin_y = self._create_spinbox(self.on_geo_changed)
        self.spin_w = self._create_spinbox(self.on_geo_changed)
        self.spin_h = self._create_spinbox(self.on_geo_changed)

        self.geo_layout.addRow("X:", self.spin_x)
        self.geo_layout.addRow("Y:", self.spin_y)
        self.geo_layout.addRow("Largeur:", self.spin_w)
        self.geo_layout.addRow("Hauteur:", self.spin_h)
        self.geo_group.setLayout(self.geo_layout)
        self.layout.addWidget(self.geo_group)

        # Groupe Texte
        self.text_group = QGroupBox("Texte")
        self.text_layout = QFormLayout()
        
        self.font_combo = QFontComboBox()
        self.font_combo.currentFontChanged.connect(self.on_text_changed)
        
        self.size_spin = QSpinBox()
        self.size_spin.setRange(6, 144)
        self.size_spin.setValue(12)
        self.size_spin.valueChanged.connect(self.on_text_changed)
        
        # Rôle (Normal, H1, H2)
        self.role_combo = QComboBox()
        self.role_combo.addItems(["Normal", "Titre 1", "Titre 2"])
        self.role_combo.currentTextChanged.connect(self.on_role_changed)

        # Formatage (Gras, Italique, Souligné)
        from PySide6.QtWidgets import QCheckBox, QHBoxLayout
        
        format_widget = QWidget()
        format_layout = QHBoxLayout(format_widget)
        format_layout.setContentsMargins(0, 0, 0, 0)
        
        self.bold_check = QCheckBox("Gras")
        self.bold_check.stateChanged.connect(self.on_text_format_changed)
        format_layout.addWidget(self.bold_check)
        
        self.italic_check = QCheckBox("Italique")
        self.italic_check.stateChanged.connect(self.on_text_format_changed)
        format_layout.addWidget(self.italic_check)
        
        self.underline_check = QCheckBox("Souligné")
        self.underline_check.stateChanged.connect(self.on_text_format_changed)
        format_layout.addWidget(self.underline_check)

        self.btn_text_color = QPushButton("Couleur Texte")
        self.btn_text_color.clicked.connect(self.on_text_color_clicked)

        # Contenu texte simple pour l'éditer ici aussi
        self.content_edit = SpellCheckTextEdit()
        self.content_edit.setMaximumHeight(100)
        self.content_edit.textChanged.connect(self.on_text_content_changed)
        
        # Correcteur orthographique
        self.spell_highlighter = SpellHighlighter(self.content_edit.document())

        self.text_layout.addRow("Police:", self.font_combo)
        self.text_layout.addRow("Taille:", self.size_spin)
        self.text_layout.addRow("Rôle:", self.role_combo)
        self.text_layout.addRow("Style:", format_widget)
        self.text_layout.addRow("Couleur:", self.btn_text_color)
        self.text_layout.addRow("Contenu:", self.content_edit)
        self.text_group.setLayout(self.text_layout)
        self.layout.addWidget(self.text_group)
        self.text_group.hide()

        # Groupe Image
        self.img_group = QGroupBox("Image")
        self.img_layout = QFormLayout()
        self.btn_change_img = QPushButton("Changer Image...")
        self.btn_change_img.clicked.connect(self.on_change_image)
        self.img_layout.addRow(self.btn_change_img)
        self.img_group.setLayout(self.img_layout)
        self.layout.addWidget(self.img_group)
        self.img_group.hide()
        
        # Initial state
        self.geo_group.hide()

    def _create_spinbox(self, callback):
        sb = QDoubleSpinBox()
        sb.setRange(-10000, 10000)
        sb.valueChanged.connect(callback)
        return sb
        
    def set_document(self, doc_model):
        self.document = doc_model
        if self.document:
            # Init values
            is_simple = self.document.numbering_style == "current_only"
            self.combo_num_style.setCurrentIndex(1 if is_simple else 0)
            self.spin_num_start.setValue(self.document.show_numbers_from)

    def set_page(self, page_model):
        self.current_page = page_model
        # If no item selected, ensure page group is shown/updated
        if not self.current_item:
             self.page_group.show()

    def set_item(self, item):
        """Met à jour le panneau avec l'item sélectionné."""
        # Déconnexion ancien item
        if self.current_item and hasattr(self.current_item, 'signals'):
            try:
                self.current_item.signals.geometryChanged.disconnect(self.update_from_item)
            except:
                pass

        self.current_item = item
        self.block_signals(True)

        if not item:
            self.geo_group.hide()
            self.text_group.hide()
            self.img_group.hide()
            
            # Show Page Properties
            self.page_group.show()
            
            self.block_signals(False)
            return
        
        # Hide Page Properties when item selected
        self.page_group.hide()
        
        # Connexion nouveau item
        if hasattr(self.current_item, 'signals'):
             self.current_item.signals.geometryChanged.connect(self.update_from_item)

        self.geo_group.show()
        self.update_from_item()

        # Type spécifique (reste identique)
        if isinstance(item, TextZoneItem):
            self.text_group.show()
            self.img_group.hide()
            
            # Block signals again just to be safe (though already blocked)
            self.font_combo.blockSignals(True)
            self.size_spin.blockSignals(True)
            
            # Use currentFont from text item
            font = item.text_item.font()
            self.font_combo.setCurrentFont(font)
            
            # Handle float size from pointSizeF, cast to int for spinbox
            size_f = font.pointSizeF()
            if size_f <= 0: # Fallback if pointSizeF is -1
                 size_f = font.pointSize()
            
            size = int(size_f)
            self.size_spin.setValue(size)
            
            # Set Text Color Button Style
            self._update_text_color_btn(item.text_item.defaultTextColor())
            
            # Set Role
            role_map = {"heading1": "Titre 1", "heading2": "Titre 2", "body": "Normal", "toc": "Normal"}
            role_inv = {v: k for k, v in role_map.items()}
            
            current_role = "body"
            if hasattr(item, "zone_model") and hasattr(item.zone_model, "role"):
                 current_role = item.zone_model.role
            
            display_role = role_map.get(current_role, "Normal")
            self.role_combo.setCurrentText(display_role)
            
            # Set B/I/U checkboxes
            self.bold_check.setChecked(font.bold())
            self.italic_check.setChecked(font.italic())
            self.underline_check.setChecked(font.underline())
            
            self.content_edit.setPlainText(item.text_item.toPlainText())
        elif isinstance(item, ImageZoneItem):
            self.text_group.hide()
            self.img_group.show()
        else:
            self.text_group.hide()
            self.img_group.hide()

        self.block_signals(False)
    
    def _update_text_color_btn(self, color):
        if color.isValid():
            self.btn_text_color.setStyleSheet(f"background-color: {color.name()}; color: {'black' if color.lightness() > 128 else 'white'}")
            self.btn_text_color.setText(color.name())
    
    def update_from_item(self):
        """Callback pour mettre à jour l'UI depuis l'item."""
        if not self.current_item:
            return
        
        self.block_signals(True)
        rect = self.current_item.rect()
        pos = self.current_item.pos()
        self.spin_x.setValue(pos.x())
        self.spin_y.setValue(pos.y())
        self.spin_w.setValue(rect.width())
        self.spin_h.setValue(rect.height())
        self.block_signals(False)

    def block_signals(self, block):
        self.spin_x.blockSignals(block)
        self.spin_y.blockSignals(block)
        self.spin_w.blockSignals(block)
        self.spin_h.blockSignals(block)
        self.font_combo.blockSignals(block)
        self.size_spin.blockSignals(block)
        self.role_combo.blockSignals(block)
        self.content_edit.blockSignals(block)

    def on_geo_changed(self):
        if not self.current_item:
            return
        
        x = self.spin_x.value()
        y = self.spin_y.value()
        w = self.spin_w.value()
        h = self.spin_h.value()
        
        self.current_item.setPos(x, y)
        self.current_item.setRect(0, 0, w, h)

    def on_text_changed(self):
        if isinstance(self.current_item, TextZoneItem):
            font = self.font_combo.currentFont()
            font.setPointSize(self.size_spin.value())
            # Apply B/I/U from checkboxes
            font.setBold(self.bold_check.isChecked())
            font.setItalic(self.italic_check.isChecked())
            font.setUnderline(self.underline_check.isChecked())
            self.current_item.text_item.setFont(font)
            
            # Update Model
            if self.current_item.zone_model:
                self.current_item.zone_model.style["font_family"] = font.family()
                self.current_item.zone_model.style["font_size"] = font.pointSize()
                self.current_item.zone_model.style["bold"] = font.bold()
                self.current_item.zone_model.style["italic"] = font.italic()
                self.current_item.zone_model.style["underline"] = font.underline()
    
    def on_text_format_changed(self):
        """Handler for bold/italic/underline checkbox changes"""
        if isinstance(self.current_item, TextZoneItem):
            font = self.current_item.text_item.font()
            font.setBold(self.bold_check.isChecked())
            font.setItalic(self.italic_check.isChecked())
            font.setUnderline(self.underline_check.isChecked())
            self.current_item.text_item.setFont(font)
            
            # Update Model
            if self.current_item.zone_model:
                self.current_item.zone_model.style["bold"] = font.bold()
                self.current_item.zone_model.style["italic"] = font.italic()
                self.current_item.zone_model.style["underline"] = font.underline()

    def on_role_changed(self, text):
        if isinstance(self.current_item, TextZoneItem) and self.current_item.zone_model:
            role_map = {"Titre 1": "heading1", "Titre 2": "heading2", "Normal": "body"}
            self.current_item.zone_model.role = role_map.get(text, "body")

    def on_text_content_changed(self):
         if isinstance(self.current_item, TextZoneItem):
             self.current_item.text_item.setPlainText(self.content_edit.toPlainText())

    def on_change_image(self):
        if isinstance(self.current_item, ImageZoneItem):
             file_path, _ = QFileDialog.getOpenFileName(self, "Choisir une image", "", "Images (*.png *.jpg *.jpeg *.bmp)")
             if file_path:
                 self.current_item.load_image(file_path)
                 
    def on_page_color_clicked(self):
        if self.current_page:
            color = QColorDialog.getColor(QColor(self.current_page.background_color), self, "Choisir la couleur de fond")
            if color.isValid():
                self.page_color_changed.emit(color.name())

    def on_page_color_all_clicked(self):
        if self.document:
            # Default to current page color or white
            init_col = QColor(self.current_page.background_color) if self.current_page else Qt.white
            color = QColorDialog.getColor(init_col, self, "Choisir la couleur de fond (Tout le document)")
            if color.isValid():
                self.page_color_all_changed.emit(color.name())

    def on_num_style_changed(self, text):
        if self.document:
            # "1 / Total" or "1"
            self.document.numbering_style = "current_only" if text == "1" else "current_total"

    def on_num_start_changed(self, val):
        if self.document:
            self.document.show_numbers_from = val

    def on_text_color_clicked(self):
        if isinstance(self.current_item, TextZoneItem):
            current_color = self.current_item.text_item.defaultTextColor()
            color = QColorDialog.getColor(current_color, self, "Choisir la couleur du texte")
            if color.isValid():
                self.current_item.text_item.setDefaultTextColor(color)
                # Store in style via MainWindow sync usually, but we can set it here if we want immediate feedback
                # But MainWindow handles Document sync. We should update the Item's model property if possible or rely on sync.
                # Actually ZoneItem.zone_model is updated in Main Window 'update_model_from_view'.
                # But we should update buttons style
                self._update_text_color_btn(color)
