
from PySide6.QtWidgets import QTextEdit, QMenu
from PySide6.QtGui import QTextCursor
import enchant

class SpellCheckTextEdit(QTextEdit):
    """QTextEdit avec menu contextuel pour corrections orthographiques"""
    
    def __init__(self, parent=None):
        super().__init__(parent)
        
        # Dictionary for suggestions
        try:
            self.dictionary = enchant.Dict("fr_FR")
        except:
            try:
                self.dictionary = enchant.Dict("fr")
            except:
                self.dictionary = None
    
    def contextMenuEvent(self, event):
        """Affiche menu avec suggestions orthographiques"""
        menu = self.createStandardContextMenu()
        
        if not self.dictionary:
            menu.exec_(event.globalPos())
            return
        
        # Get cursor at mouse position
        cursor = self.cursorForPosition(event.pos())
        cursor.select(QTextCursor.WordUnderCursor)
        word = cursor.selectedText()
        
        if word and not any(char.isdigit() for char in word):
            if not self.dictionary.check(word):
                # Word is misspelled, add suggestions
                suggestions = self.dictionary.suggest(word)[:5]  # Max 5 suggestions
                
                if suggestions:
                    menu.insertSeparator(menu.actions()[0])
                    
                    for suggestion in suggestions:
                        action = menu.addAction(f"→ {suggestion}")
                        action.triggered.connect(lambda checked, s=suggestion, c=cursor: self.replace_word(c, s))
                    
                    menu.insertSeparator(menu.actions()[0])
        
        menu.exec_(event.globalPos())
    
    def replace_word(self, cursor, new_word):
        """Replace misspelled word with suggestion"""
        cursor.beginEditBlock()
        cursor.removeSelectedText()
        cursor.insertText(new_word)
        cursor.endEditBlock()
