"""
Effets audio pour le synthétiseur : Chorus et Reverb.
"""

import numpy as np

class ChorusEffect:
    """
    Effet Chorus simple utilisant des delay lines modulées.
    """
    
    def __init__(self, sample_rate=44100):
        """
        Initialise l'effet Chorus.
        
        Args:
            sample_rate: Taux d'échantillonnage
        """
        self.sample_rate = sample_rate
        self.enabled = False
        
        # Paramètres
        self.rate = 1.5  # Hz - vitesse de modulation du LFO
        self.depth = 0.003  # secondes - profondeur de modulation du delay
        self.mix = 0.5  # 0.0 à 1.0 - mix wet/dry
        
        # Buffer de delay (50ms max)
        max_delay_samples = int(0.05 * sample_rate)
        self.delay_buffer = np.zeros(max_delay_samples)
        self.write_pos = 0
        
        # LFO phase
        self.lfo_phase = 0.0
    
    def set_enabled(self, enabled):
        """Active/désactive l'effet."""
        self.enabled = enabled
        if not enabled:
            # Reset le buffer
            self.delay_buffer.fill(0)
            self.write_pos = 0
            self.lfo_phase = 0.0
    
    def set_rate(self, rate):
        """Définit la vitesse du LFO (Hz)."""
        self.rate = np.clip(rate, 0.1, 10.0)
    
    def set_depth(self, depth):
        """Définit la profondeur du chorus (0.0 à 1.0)."""
        self.depth = np.clip(depth, 0.0, 1.0) * 0.01  # Max 10ms
    
    def set_mix(self, mix):
        """Définit le mix wet/dry (0.0 à 1.0)."""
        self.mix = np.clip(mix, 0.0, 1.0)
    
    def process(self, audio):
        """
        Applique l'effet chorus.
        
        Args:
            audio: Tableau numpy d'échantillons
        
        Returns:
            Audio traité
        """
        if not self.enabled:
            return audio
        
        output = np.zeros_like(audio)
        buffer_size = len(self.delay_buffer)
        
        for i in range(len(audio)):
            # Écrire dans le buffer
            self.delay_buffer[self.write_pos] = audio[i]
            
            # Calculer le delay modulé par le LFO
            lfo = np.sin(2.0 * np.pi * self.lfo_phase)
            delay_samples = self.depth * self.sample_rate * (1.0 + lfo)
            
            # Position de lecture
            read_pos = (self.write_pos - delay_samples) % buffer_size
            read_pos_int = int(read_pos)
            frac = read_pos - read_pos_int
            
            # Interpolation linéaire
            sample1 = self.delay_buffer[read_pos_int]
            sample2 = self.delay_buffer[(read_pos_int + 1) % buffer_size]
            delayed_sample = sample1 + frac * (sample2 - sample1)
            
            # Mix wet/dry
            output[i] = (1.0 - self.mix) * audio[i] + self.mix * delayed_sample
            
            # Avancer les positions
            self.write_pos = (self.write_pos + 1) % buffer_size
            self.lfo_phase += self.rate / self.sample_rate
            if self.lfo_phase >= 1.0:
                self.lfo_phase -= 1.0
        
        return output


class ReverbEffect:
    """
    Effet de réverbération simple (Schroeder reverb simplifié).
    Utilise des comb filters et all-pass filters.
    """
    
    def __init__(self, sample_rate=44100):
        """
        Initialise l'effet Reverb.
        
        Args:
            sample_rate: Taux d'échantillonnage
        """
        self.sample_rate = sample_rate
        self.enabled = False
        
        # Paramètres
        self.room_size = 0.5  # 0.0 à 1.0
        self.damping = 0.5  # 0.0 à 1.0
        self.mix = 0.3  # 0.0 à 1.0
        
        # Comb filters (4 filtres en parallèle)
        comb_delays = [1557, 1617, 1491, 1422]  # Samples (environ 35-37ms à 44.1kHz)
        self.comb_buffers = [np.zeros(d) for d in comb_delays]
        self.comb_positions = [0] * len(comb_delays)
        self.comb_filter_state = [0.0] * len(comb_delays)
        
        # All-pass filters (2 filtres en série)
        allpass_delays = [225, 556]  # Samples
        self.allpass_buffers = [np.zeros(d) for d in allpass_delays]
        self.allpass_positions = [0] * len(allpass_delays)
    
    def set_enabled(self, enabled):
        """Active/désactive l'effet."""
        self.enabled = enabled
        if not enabled:
            # Reset les buffers
            for buf in self.comb_buffers:
                buf.fill(0)
            for buf in self.allpass_buffers:
                buf.fill(0)
            self.comb_positions = [0] * len(self.comb_buffers)
            self.allpass_positions = [0] * len(self.allpass_buffers)
            self.comb_filter_state = [0.0] * len(self.comb_buffers)
    
    def set_room_size(self, room_size):
        """Définit la taille de la pièce (0.0 à 1.0)."""
        self.room_size = np.clip(room_size, 0.0, 1.0)
    
    def set_damping(self, damping):
        """Définit l'amortissement (0.0 à 1.0)."""
        self.damping = np.clip(damping, 0.0, 1.0)
    
    def set_mix(self, mix):
        """Définit le mix wet/dry (0.0 à 1.0)."""
        self.mix = np.clip(mix, 0.0, 1.0)
    
    def _process_comb_filter(self, input_sample, buffer, position, filter_state_idx):
        """Traite un comb filter."""
        # Feedback calculé à partir de room_size
        feedback = 0.7 + self.room_size * 0.28  # 0.7 à 0.98
        
        # Lire de la buffer
        output = buffer[position]
        
        # Lowpass filter pour damping
        damp = 0.4 * self.damping
        filtered = output * (1.0 - damp) + self.comb_filter_state[filter_state_idx] * damp
        self.comb_filter_state[filter_state_idx] = filtered
        
        # Écrire dans la buffer avec feedback
        buffer[position] = input_sample + filtered * feedback
        
        return output
    
    def _process_allpass_filter(self, input_sample, buffer, position):
        """Traite un all-pass filter."""
        feedback = 0.5
        
        # Lire de la buffer
        buffered = buffer[position]
        output = -input_sample + buffered
        
        # Écrire dans la buffer
        buffer[position] = input_sample + buffered * feedback
        
        return output
    
    def process(self, audio):
        """
        Applique l'effet de réverbération.
        
        Args:
            audio: Tableau numpy d'échantillons
        
        Returns:
            Audio traité
        """
        if not self.enabled:
            return audio
        
        output = np.zeros_like(audio)
        
        for i in range(len(audio)):
            # Comb filters en parallèle
            comb_sum = 0.0
            for j in range(len(self.comb_buffers)):
                comb_output = self._process_comb_filter(
                    audio[i],
                    self.comb_buffers[j],
                    self.comb_positions[j],
                    j
                )
                comb_sum += comb_output
                
                # Avancer la position
                self.comb_positions[j] = (self.comb_positions[j] + 1) % len(self.comb_buffers[j])
            
            comb_sum *= 0.25  # Moyenne des 4 combs
            
            # All-pass filters en série
            allpass_output = comb_sum
            for j in range(len(self.allpass_buffers)):
                allpass_output = self._process_allpass_filter(
                    allpass_output,
                    self.allpass_buffers[j],
                    self.allpass_positions[j]
                )
                
                # Avancer la position
                self.allpass_positions[j] = (self.allpass_positions[j] + 1) % len(self.allpass_buffers[j])
            
            # Mix wet/dry
            output[i] = (1.0 - self.mix) * audio[i] + self.mix * allpass_output
        
        return output
