"""
Gestion de la sortie audio en temps réel avec sounddevice.
"""

import numpy as np
import sounddevice as sd
import threading
import queue

class AudioOutput:
    """
    Gère la sortie audio en temps réel avec un callback.
    """
    
    def __init__(self, synth_engine, sf2_engine=None, sample_rate=44100, buffer_size=512):
        """
        Initialise la sortie audio.
        
        Args:
            synth_engine: Instance de SynthEngine
            sf2_engine: Instance de FluidSynthEngine (optionnel)
            sample_rate: Taux d'échantillonnage
            buffer_size: Taille du buffer audio (plus petit = moins de latence)
        """
        self.synth_engine = synth_engine
        self.sf2_engine = sf2_engine
        self.sample_rate = sample_rate
        self.buffer_size = buffer_size
        self.stream = None
        self.is_running = False
    
    def _audio_callback(self, outdata, frames, time_info, status):
        """
        Callback appelé par sounddevice pour remplir le buffer audio.
        
        Args:
            outdata: Buffer de sortie à remplir
            frames: Nombre de frames demandées
            time_info: Informations temporelles
            status: Statut du stream
        """
        if status:
            print(f"Audio callback status: {status}")
        
        try:
            # Générer les échantillons audio du synthétiseur
            audio = self.synth_engine.generate_audio(frames)
            
            # Ajouter les échantillons de la SoundFont si disponible
            if self.sf2_engine:
                audio_sf2 = self.sf2_engine.generate_audio(frames)
                if audio_sf2 is not None:
                    # S'assurer que les deux signaux ont la même longueur (devrait être frames)
                    if len(audio_sf2) == len(audio):
                        audio = audio + audio_sf2
            
            # Clipping final pour éviter la saturation après mixage
            audio = np.clip(audio, -1.0, 1.0)
            
            # Convertir en stéréo (dupliquer le signal mono si nécessaire)
            # FluidSynthEngine produit déjà du stéréo si configuré, mais ici on gère le retour mono -> stéréo
            if len(outdata.shape) == 2:  # Sortie stéréo
                outdata[:, 0] = audio
                outdata[:, 1] = audio
            else:  # Sortie mono
                outdata[:] = audio.reshape(-1, 1)
                
        except Exception as e:
            print(f"Erreur dans le callback audio: {e}")
            # En cas d'erreur, remplir avec du silence
            outdata.fill(0)
    
    def start(self):
        """Démarre le flux audio."""
        if self.is_running:
            print("Le flux audio est déjà actif.")
            return
        
        try:
            # Créer et démarrer le flux audio
            self.stream = sd.OutputStream(
                samplerate=self.sample_rate,
                channels=2,  # Stéréo
                callback=self._audio_callback,
                blocksize=self.buffer_size,
                dtype='float32'
            )
            
            self.stream.start()
            self.is_running = True
            print(f"Flux audio démarré (sample rate: {self.sample_rate} Hz, buffer: {self.buffer_size})")
            
        except Exception as e:
            print(f"Erreur lors du démarrage du flux audio: {e}")
            self.is_running = False
    
    def stop(self):
        """Arrête le flux audio."""
        if not self.is_running:
            return
        
        try:
            if self.stream:
                self.stream.stop()
                self.stream.close()
                self.stream = None
            self.is_running = False
            print("Flux audio arrêté.")
            
        except Exception as e:
            print(f"Erreur lors de l'arrêt du flux audio: {e}")
    
    def get_latency(self):
        """
        Retourne la latence estimée en millisecondes.
        
        Returns:
            Latence en ms
        """
        if self.stream:
            latency = self.stream.latency
            # latency est un tuple (input, output)
            return latency[1] * 1000 if isinstance(latency, tuple) else latency * 1000
        return 0
