"""
Gestionnaire de configuration pour mémoriser les préférences utilisateur.
"""

import json
import os
from pathlib import Path

class ConfigManager:
    """
    Gère la configuration du synthétiseur (port MIDI, preset, position fenêtre, etc.).
    """
    
    DEFAULT_CONFIG = {
        "last_midi_port": None,
        "last_category": "🎵 Classiques",
        "last_preset": "Lead Synth",
        "window_position": {"x": 100, "y": 100},
        "window_size": {"width": 800, "height": 700},
        "theme": "dark",
        "soundfont_path": None,  # Chemin de la dernière SoundFont chargée
        "soundfont_instrument": {  # Dernier instrument SoundFont sélectionné
            "name": None,
            "bank": 0,
            "preset": 0
        },
        "engine_mode": "Synthétiseur (Soustractif)",  # Dernier moteur utilisé
        "instruments_per_row": 8  # Nombre d'instruments affichés par ligne dans le sélecteur
    }
    
    def __init__(self, config_file="config.json"):
        """
        Initialise le gestionnaire de configuration.
        
        Args:
            config_file: Chemin du fichier de configuration
        """
        self.config_file = config_file
        self.config = self.DEFAULT_CONFIG.copy()
        self.load_config()
    
    def load_config(self):
        """Charge la configuration depuis le fichier."""
        if os.path.exists(self.config_file):
            try:
                with open(self.config_file, 'r', encoding='utf-8') as f:
                    loaded_config = json.load(f)
                    # Fusionner avec les valeurs par défaut (pour gérer les nouvelles clés)
                    self.config.update(loaded_config)
                print(f"✓ Configuration chargée depuis {self.config_file}")
            except Exception as e:
                print(f"Erreur lors du chargement de la configuration: {e}")
                print("Utilisation de la configuration par défaut.")
        else:
            print("Aucun fichier de configuration trouvé. Utilisation des valeurs par défaut.")
    
    def save_config(self):
        """Sauvegarde la configuration dans le fichier."""
        try:
            with open(self.config_file, 'w', encoding='utf-8') as f:
                json.dump(self.config, f, indent=2, ensure_ascii=False)
            print(f"✓ Configuration sauvegardée dans {self.config_file}")
        except Exception as e:
            print(f"Erreur lors de la sauvegarde de la configuration: {e}")
    
    # Getters
    def get_last_midi_port(self):
        """Retourne le dernier port MIDI utilisé."""
        return self.config.get("last_midi_port")
    
    def get_last_preset(self):
        """Retourne le dernier preset utilisé."""
        return self.config.get("last_preset", "Lead Synth")
    
    def get_last_category(self):
        """Retourne la dernière catégorie utilisée."""
        return self.config.get("last_category", "🎵 Classiques")
    
    def get_window_position(self):
        """Retourne la position de la fenêtre."""
        return self.config.get("window_position", {"x": 100, "y": 100})
    
    def get_window_size(self):
        """Retourne la taille de la fenêtre."""
        return self.config.get("window_size", {"width": 800, "height": 700})
    
    def get_theme(self):
        """Retourne le thème de l'interface."""
        return self.config.get("theme", "dark")
    
    # Setters
    def set_last_midi_port(self, port_name):
        """Définit le dernier port MIDI utilisé."""
        self.config["last_midi_port"] = port_name
    
    def set_last_category(self, category):
        """Définit la dernière catégorie utilisée."""
        self.config["last_category"] = category
    
    def set_last_preset(self, preset_name):
        """Définit le dernier preset utilisé."""
        self.config["last_preset"] = preset_name
    
    def set_window_position(self, x, y):
        """Définit la position de la fenêtre."""
        self.config["window_position"] = {"x": x, "y": y}
    
    def set_window_size(self, width, height):
        """Définit la taille de la fenêtre."""
        self.config["window_size"] = {"width": width, "height": height}
    
    def set_theme(self, theme):
        """Définit le thème de l'interface."""
        self.config["theme"] = theme
    
    def get_soundfont_path(self):
        """Retourne le chemin de la dernière SoundFont chargée."""
        return self.config.get("soundfont_path")
    
    def set_soundfont_path(self, path):
        """Définit le chemin de la dernière SoundFont chargée."""
        self.config["soundfont_path"] = path
    
    def get_soundfont_instrument(self):
        """Retourne le dernier instrument SoundFont sélectionné."""
        return self.config.get("soundfont_instrument", {"name": None, "bank": 0, "preset": 0})
    
    def set_soundfont_instrument(self, name, bank, preset):
        """Définit le dernier instrument SoundFont sélectionné."""
        self.config["soundfont_instrument"] = {
            "name": name,
            "bank": bank,
            "preset": preset
        }
    
    def get_engine_mode(self):
        """Retourne le dernier moteur utilisé."""
        return self.config.get("engine_mode", "Synthétiseur (Soustractif)")
    
    def set_engine_mode(self, mode):
        """Définit le dernier moteur utilisé."""
        self.config["engine_mode"] = mode
    
    def get(self, key, default=None):
        """Récupère une valeur de configuration arbitraire."""
        return self.config.get(key, default)
    
    def set(self, key, value):
        """Définit une valeur de configuration arbitraire."""
        self.config[key] = value
    
    def get_instruments_per_row(self):
        """Retourne le nombre d'instruments par ligne dans le sélecteur."""
        return self.config.get("instruments_per_row", 8)
    
    def set_instruments_per_row(self, count):
        """Définit le nombre d'instruments par ligne dans le sélecteur."""
        self.config["instruments_per_row"] = max(1, min(count, 20))  # Entre 1 et 20
