"""
Module d'enveloppes ADSR pour contrôler l'amplitude dans le temps.
"""

import numpy as np

class ADSREnvelope:
    """
    Enveloppe ADSR (Attack, Decay, Sustain, Release) pour contrôler
    l'évolution de l'amplitude d'une note dans le temps.
    """
    
    # États de l'enveloppe
    IDLE = 0
    ATTACK = 1
    DECAY = 2
    SUSTAIN = 3
    RELEASE = 4
    
    def __init__(self, sample_rate=44100):
        """
        Initialise l'enveloppe ADSR.
        
        Args:
            sample_rate: Taux d'échantillonnage
        """
        self.sample_rate = sample_rate
        
        # Paramètres ADSR (en secondes, sauf sustain)
        self.attack_time = 0.01   # 10ms
        self.decay_time = 0.1     # 100ms
        self.sustain_level = 0.7  # 70% du niveau max
        self.release_time = 0.3   # 300ms
        
        # État interne
        self.state = self.IDLE
        self.current_level = 0.0
        self.release_level = 0.0  # Niveau au début du release
        self.sample_count = 0
    
    def set_attack(self, time_seconds):
        """Définit le temps d'attaque en secondes."""
        self.attack_time = max(0.001, time_seconds)  # Minimum 1ms
    
    def set_decay(self, time_seconds):
        """Définit le temps de decay en secondes."""
        self.decay_time = max(0.001, time_seconds)
    
    def set_sustain(self, level):
        """Définit le niveau de sustain (0.0 à 1.0)."""
        self.sustain_level = np.clip(level, 0.0, 1.0)
    
    def set_release(self, time_seconds):
        """Définit le temps de release en secondes."""
        self.release_time = max(0.001, time_seconds)
    
    def note_on(self):
        """Démarre l'enveloppe (note jouée)."""
        self.state = self.ATTACK
        self.sample_count = 0
        # Si on était en release, partir du niveau actuel
        if self.current_level < 0.0001:
            self.current_level = 0.0
    
    def note_off(self):
        """Commence la phase de release (note relâchée)."""
        if self.state != self.IDLE:
            self.state = self.RELEASE
            self.release_level = self.current_level
            self.sample_count = 0
    
    def is_active(self):
        """Retourne True si l'enveloppe est active (pas IDLE)."""
        return self.state != self.IDLE
    
    def reset(self):
        """Réinitialise l'enveloppe."""
        self.state = self.IDLE
        self.current_level = 0.0
        self.sample_count = 0
    
    def process(self, num_samples):
        """
        Génère les valeurs d'enveloppe pour un nombre d'échantillons.
        
        Args:
            num_samples: Nombre d'échantillons à générer
        
        Returns:
            numpy array des valeurs d'enveloppe (0.0 à 1.0)
        """
        output = np.zeros(num_samples)
        
        for i in range(num_samples):
            if self.state == self.IDLE:
                output[i] = 0.0
                
            elif self.state == self.ATTACK:
                # Phase d'attaque : montée linéaire de 0 à 1
                attack_samples = int(self.attack_time * self.sample_rate)
                if attack_samples == 0:
                    self.current_level = 1.0
                    self.state = self.DECAY
                    self.sample_count = 0
                else:
                    progress = self.sample_count / attack_samples
                    self.current_level = min(1.0, progress)
                    
                    if self.sample_count >= attack_samples:
                        self.current_level = 1.0
                        self.state = self.DECAY
                        self.sample_count = 0
                
                output[i] = self.current_level
                self.sample_count += 1
                
            elif self.state == self.DECAY:
                # Phase de decay : descente de 1 vers sustain_level
                decay_samples = int(self.decay_time * self.sample_rate)
                if decay_samples == 0:
                    self.current_level = self.sustain_level
                    self.state = self.SUSTAIN
                else:
                    progress = self.sample_count / decay_samples
                    self.current_level = 1.0 - progress * (1.0 - self.sustain_level)
                    self.current_level = max(self.sustain_level, self.current_level)
                    
                    if self.sample_count >= decay_samples:
                        self.current_level = self.sustain_level
                        self.state = self.SUSTAIN
                
                output[i] = self.current_level
                self.sample_count += 1
                
            elif self.state == self.SUSTAIN:
                # Phase de sustain : maintien du niveau
                self.current_level = self.sustain_level
                output[i] = self.current_level
                
            elif self.state == self.RELEASE:
                # Phase de release : descente vers 0
                release_samples = int(self.release_time * self.sample_rate)
                if release_samples == 0:
                    self.current_level = 0.0
                    self.state = self.IDLE
                else:
                    progress = self.sample_count / release_samples
                    self.current_level = self.release_level * (1.0 - progress)
                    self.current_level = max(0.0, self.current_level)
                    
                    if self.sample_count >= release_samples or self.current_level < 0.0001:
                        self.current_level = 0.0
                        self.state = self.IDLE
                
                output[i] = self.current_level
                self.sample_count += 1
        
        return output
