"""
Module de filtres audio pour sculpter le timbre.
"""

import numpy as np
from scipy import signal

class LowPassFilter:
    """
    Filtre passe-bas résonant pour supprimer les hautes fréquences.
    Implémentation d'un filtre biquad.
    """
    
    def __init__(self, sample_rate=44100):
        """
        Initialise le filtre passe-bas.
        
        Args:
            sample_rate: Taux d'échantillonnage
        """
        self.sample_rate = sample_rate
        self.cutoff_freq = 1000.0  # Hz
        self.resonance = 0.7  # Q factor
        
        # État du filtre (pour continuité entre les appels)
        self.x1 = 0.0
        self.x2 = 0.0
        self.y1 = 0.0
        self.y2 = 0.0
        
        # Coefficients du filtre
        self._update_coefficients()
    
    def set_cutoff(self, frequency):
        """
        Définit la fréquence de coupure.
        
        Args:
            frequency: Fréquence de coupure en Hz
        """
        self.cutoff_freq = np.clip(frequency, 20.0, self.sample_rate / 2.0 - 1.0)
        self._update_coefficients()
    
    def set_resonance(self, q):
        """
        Définit la résonance (Q factor).
        
        Args:
            q: Facteur de qualité (0.1 à 20.0, typiquement 0.7)
        """
        self.resonance = np.clip(q, 0.1, 20.0)
        self._update_coefficients()
    
    def _update_coefficients(self):
        """Calcule les coefficients du filtre biquad."""
        # Fréquence normalisée
        omega = 2.0 * np.pi * self.cutoff_freq / self.sample_rate
        
        # Calculs des coefficients
        alpha = np.sin(omega) / (2.0 * self.resonance)
        cos_omega = np.cos(omega)
        
        # Coefficients du numérateur
        b0 = (1.0 - cos_omega) / 2.0
        b1 = 1.0 - cos_omega
        b2 = (1.0 - cos_omega) / 2.0
        
        # Coefficients du dénominateur
        a0 = 1.0 + alpha
        a1 = -2.0 * cos_omega
        a2 = 1.0 - alpha
        
        # Normalisation
        self.b0 = b0 / a0
        self.b1 = b1 / a0
        self.b2 = b2 / a0
        self.a1 = a1 / a0
        self.a2 = a2 / a0
    
    def process(self, input_samples):
        """
        Applique le filtre aux échantillons d'entrée.
        
        Args:
            input_samples: numpy array des échantillons à filtrer
        
        Returns:
            numpy array des échantillons filtrés
        """
        output = np.zeros_like(input_samples)
        
        for i in range(len(input_samples)):
            # Équation de différence du filtre biquad
            x0 = input_samples[i]
            y0 = (self.b0 * x0 + self.b1 * self.x1 + self.b2 * self.x2 
                  - self.a1 * self.y1 - self.a2 * self.y2)
            
            # Mise à jour de l'état
            self.x2 = self.x1
            self.x1 = x0
            self.y2 = self.y1
            self.y1 = y0
            
            output[i] = y0
        
        return output
    
    def reset(self):
        """Réinitialise l'état du filtre."""
        self.x1 = 0.0
        self.x2 = 0.0
        self.y1 = 0.0
        self.y2 = 0.0

class HighPassFilter:
    """
    Filtre passe-haut pour supprimer les basses fréquences.
    """
    
    def __init__(self, sample_rate=44100):
        """
        Initialise le filtre passe-haut.
        
        Args:
            sample_rate: Taux d'échantillonnage
        """
        self.sample_rate = sample_rate
        self.cutoff_freq = 100.0  # Hz
        self.resonance = 0.7
        
        # État du filtre
        self.x1 = 0.0
        self.x2 = 0.0
        self.y1 = 0.0
        self.y2 = 0.0
        
        self._update_coefficients()
    
    def set_cutoff(self, frequency):
        """Définit la fréquence de coupure."""
        self.cutoff_freq = np.clip(frequency, 20.0, self.sample_rate / 2.0 - 1.0)
        self._update_coefficients()
    
    def set_resonance(self, q):
        """Définit la résonance."""
        self.resonance = np.clip(q, 0.1, 20.0)
        self._update_coefficients()
    
    def _update_coefficients(self):
        """Calcule les coefficients du filtre biquad passe-haut."""
        omega = 2.0 * np.pi * self.cutoff_freq / self.sample_rate
        alpha = np.sin(omega) / (2.0 * self.resonance)
        cos_omega = np.cos(omega)
        
        # Coefficients pour passe-haut
        b0 = (1.0 + cos_omega) / 2.0
        b1 = -(1.0 + cos_omega)
        b2 = (1.0 + cos_omega) / 2.0
        
        a0 = 1.0 + alpha
        a1 = -2.0 * cos_omega
        a2 = 1.0 - alpha
        
        # Normalisation
        self.b0 = b0 / a0
        self.b1 = b1 / a0
        self.b2 = b2 / a0
        self.a1 = a1 / a0
        self.a2 = a2 / a0
    
    def process(self, input_samples):
        """Applique le filtre aux échantillons."""
        output = np.zeros_like(input_samples)
        
        for i in range(len(input_samples)):
            x0 = input_samples[i]
            y0 = (self.b0 * x0 + self.b1 * self.x1 + self.b2 * self.x2 
                  - self.a1 * self.y1 - self.a2 * self.y2)
            
            self.x2 = self.x1
            self.x1 = x0
            self.y2 = self.y1
            self.y1 = y0
            
            output[i] = y0
        
        return output
    
    def reset(self):
        """Réinitialise l'état du filtre."""
        self.x1 = 0.0
        self.x2 = 0.0
        self.y1 = 0.0
        self.y2 = 0.0
