"""
Application principale du synthétiseur MIDI.
Intègre la détection MIDI avec le moteur de synthèse.
"""

import mido
import time
import threading
from synth_engine import SynthEngine
from audio_output import AudioOutput

# Note names mapping
NOTE_NAMES = ['C', 'C#', 'D', 'D#', 'E', 'F', 'F#', 'G', 'G#', 'A', 'A#', 'B']

def get_note_name(note_number):
    """Convertit un numéro MIDI en nom de note."""
    octave = note_number // 12 - 1
    note = NOTE_NAMES[note_number % 12]
    return f"{note}{octave}"

class MIDISynthesizer:
    """
    Synthétiseur MIDI complet.
    """
    
    def __init__(self, sample_rate=44100):
        """
        Initialise le synthétiseur.
        
        Args:
            sample_rate: Taux d'échantillonnage
        """
        self.sample_rate = sample_rate
        
        # Initialiser le moteur de synthèse
        self.synth_engine = SynthEngine(sample_rate=sample_rate, num_voices=32)
        
        # Configurer les paramètres par défaut
        self.configure_default_preset()
        
        # Initialiser la sortie audio
        self.audio_output = AudioOutput(self.synth_engine, sample_rate=sample_rate, buffer_size=256)
        
        # État
        self.midi_port = None
        self.is_running = False
        self.midi_thread = None
    
    def configure_default_preset(self):
        """Configure un preset par défaut avec un son de synthé."""
        # Forme d'onde
        self.synth_engine.set_waveform('sawtooth')
        
        # ADSR : Attaque rapide, sustain élevé, release moyen
        self.synth_engine.set_adsr(
            attack=0.01,    # 10ms
            decay=0.1,      # 100ms
            sustain=0.7,    # 70%
            release=0.3     # 300ms
        )
        
        # Filtre : Cutoff à 2000 Hz, résonance modérée
        self.synth_engine.set_filter(cutoff=2000.0, resonance=1.0)
        
        # Volume master
        self.synth_engine.set_master_volume(0.6)
    
    def select_midi_port(self):
        """
        Permet à l'utilisateur de sélectionner un port MIDI.
        
        Returns:
            Nom du port MIDI sélectionné, ou None si aucun
        """
        print("\n=== Sélection du port MIDI ===")
        try:
            input_names = mido.get_input_names()
        except Exception as e:
            print(f"Erreur lors de la recherche des ports MIDI: {e}")
            return None
        
        if not input_names:
            print("Aucun port MIDI trouvé. Assurez-vous que votre clavier est connecté.")
            return None
        
        print(f"\nPorts MIDI disponibles ({len(input_names)}):")
        for i, name in enumerate(input_names):
            print(f"  {i}: {name}")
        
        # Sélection automatique ou manuelle
        if len(input_names) > 1:
            # Essayer de trouver un port "réel" (pas Midi Through)
            default_index = 0
            for i, name in enumerate(input_names):
                if "Midi Through" not in name and "Through" not in name:
                    default_index = i
                    break
            
            selection = input(f"\nEntrez le numéro du port [{default_index}]: ").strip()
            
            if selection:
                try:
                    index = int(selection)
                    if 0 <= index < len(input_names):
                        return input_names[index]
                    else:
                        print("Numéro invalide, utilisation du port par défaut.")
                        return input_names[default_index]
                except ValueError:
                    print("Entrée invalide, utilisation du port par défaut.")
                    return input_names[default_index]
            else:
                return input_names[default_index]
        else:
            return input_names[0]
    
    def midi_loop(self, port_name):
        """
        Boucle de traitement des messages MIDI.
        
        Args:
            port_name: Nom du port MIDI
        """
        try:
            with mido.open_input(port_name) as inport:
                print(f"✓ Connecté au port MIDI: {port_name}")
                print("\n🎹 Synthétiseur prêt ! Jouez des notes sur votre clavier.")
                print("   Appuyez sur Ctrl+C pour quitter.\n")
                
                for msg in inport:
                    if not self.is_running:
                        break
                    
                    if msg.type == 'note_on' and msg.velocity > 0:
                        # Note On
                        note_name = get_note_name(msg.note)
                        print(f"♪ {note_name} (vélocité: {msg.velocity})")
                        self.synth_engine.note_on(msg.note, msg.velocity)
                    
                    elif msg.type == 'note_off' or (msg.type == 'note_on' and msg.velocity == 0):
                        # Note Off
                        self.synth_engine.note_off(msg.note)
                    
                    # Autres messages MIDI pourraient être gérés ici
                    # (pitch bend, modulation, etc.)
                    
        except KeyboardInterrupt:
            print("\n\nInterruption détectée dans le thread MIDI.")
        except Exception as e:
            print(f"\nErreur dans la boucle MIDI: {e}")
        finally:
            self.is_running = False
    
    def start(self):
        """Démarre le synthétiseur."""
        # Sélectionner le port MIDI
        port_name = self.select_midi_port()
        if not port_name:
            print("Impossible de démarrer sans port MIDI.")
            return False
        
        # Démarrer le flux audio
        print("\n=== Démarrage du moteur audio ===")
        self.audio_output.start()
        
        latency = self.audio_output.get_latency()
        print(f"✓ Latence audio: {latency:.1f} ms")
        
        # Démarrer le thread MIDI
        self.is_running = True
        self.midi_thread = threading.Thread(target=self.midi_loop, args=(port_name,), daemon=True)
        self.midi_thread.start()
        
        return True
    
    def stop(self):
        """Arrête le synthétiseur."""
        print("\n=== Arrêt du synthétiseur ===")
        
        # Arrêter la boucle MIDI
        self.is_running = False
        if self.midi_thread:
            self.midi_thread.join(timeout=2.0)
        
        # Arrêter toutes les notes
        self.synth_engine.panic()
        
        # Arrêter le flux audio
        self.audio_output.stop()
        
        print("✓ Synthétiseur arrêté.")
    
    def run(self):
        """Exécute le synthétiseur jusqu'à interruption."""
        if not self.start():
            return
        
        try:
            # Attendre jusqu'à ce que l'utilisateur interrompe
            while self.is_running:
                time.sleep(0.1)
        except KeyboardInterrupt:
            print("\n\nInterruption clavier détectée.")
        finally:
            self.stop()

def main():
    """Point d'entrée principal."""
    import sys
    
    # Vérifier si le mode GUI est demandé
    if "--gui" in sys.argv:
        # Lancer l'interface graphique
        from synth_gui import main as gui_main
        gui_main()
    else:
        # Mode console (comportement par défaut)
        print("╔═══════════════════════════════════════╗")
        print("║   🎹 Synthétiseur MIDI Python 🎹   ║")
        print("╔═══════════════════════════════════════╗")
        print()
        print("Astuce: Utilisez 'python main_synth.py --gui' pour l'interface graphique")
        print()
        
        synth = MIDISynthesizer(sample_rate=44100)
        synth.run()
        
        print("\nMerci d'avoir utilisé le synthétiseur !")

if __name__ == "__main__":
    main()
