import mido
import time

# Note names mapping
NOTE_NAMES = ['C', 'C#', 'D', 'D#', 'E', 'F', 'F#', 'G', 'G#', 'A', 'A#', 'B']

def get_note_name(note_number):
    """Converts a MIDI note number to its name (e.g., 60 -> C4)."""
    octave = note_number // 12 - 1
    note = NOTE_NAMES[note_number % 12]
    return f"{note}{octave}"

def main():
    print("Recherche des ports MIDI disponibles...")
    try:
        input_names = mido.get_input_names()
    except Exception as e:
        print(f"Erreur lors de la recherche des ports MIDI: {e}")
        return

    if not input_names:
        print("Aucun port MIDI trouvé. Assurez-vous que votre clavier est connecté.")
        return

    print(f"Ports MIDI trouvés ({len(input_names)}) :")
    for i, name in enumerate(input_names):
        print(f"{i}: {name}")

    if len(input_names) > 1:
        # Try to find a "real" device (not Midi Through)
        default_index = 0
        for i, name in enumerate(input_names):
            if "Midi Through" not in name:
                default_index = i
                break
        
        selection = input(f"\nEntrez le numéro du port à utiliser (par défaut {default_index} - {input_names[default_index]}) : ")
        if selection.strip():
            try:
                index = int(selection)
                if 0 <= index < len(input_names):
                    port_name = input_names[index]
                else:
                    print("Numéro invalide, utilisation du port par défaut.")
                    port_name = input_names[default_index]
            except ValueError:
                print("Entrée invalide, utilisation du port par défaut.")
                port_name = input_names[default_index]
        else:
            port_name = input_names[default_index]
    else:
        port_name = input_names[0]

    print(f"\nConnexion au port : {port_name}")

    active_notes = {}  # Dictionary to store start time of active notes: {note_number: start_time}

    try:
        with mido.open_input(port_name) as inport:
            print("Prêt ! Jouez des notes sur votre clavier (Ctrl+C pour quitter).")
            
            for msg in inport:
                current_time = time.time()
                
                if msg.type == 'note_on' and msg.velocity > 0:
                    # Note On
                    note_name = get_note_name(msg.note)
                    print(f"Note ON : {note_name}")
                    active_notes[msg.note] = current_time
                
                elif msg.type == 'note_off' or (msg.type == 'note_on' and msg.velocity == 0):
                    # Note Off
                    if msg.note in active_notes:
                        start_time = active_notes.pop(msg.note)
                        duration = current_time - start_time
                        note_name = get_note_name(msg.note)
                        print(f"Note OFF: {note_name} | Durée: {duration:.2f} secondes")
                        
    except KeyboardInterrupt:
        print("\nArrêt du programme.")
    except Exception as e:
        print(f"\nUne erreur est survenue : {e}")

if __name__ == "__main__":
    main()
