"""
Fenêtre d'édition de presets pour le synthétiseur.
"""

import customtkinter as ctk
from tkinter import messagebox

class PresetEditorWindow(ctk.CTkToplevel):
    """
    Fenêtre popup pour éditer et créer des presets.
    """
    
    def __init__(self, parent, preset_manager, synth_engine, category_var, preset_var):
        super().__init__(parent)
        
        self.preset_manager = preset_manager
        self.synth_engine = synth_engine
        self.category_var = category_var  # Variable catégorie du dropdown principal
        self.preset_var = preset_var  # Variable preset du dropdown principal
        self.parent = parent
        
        # Configuration de la fenêtre
        self.title("✏️ Éditeur de Presets")
        self.geometry("500x700")
        self.resizable(False, False)
        
        # Variables pour les paramètres
        self._create_variables()
        
        # Créer l'interface
        self._create_widgets()
        
        # Charger le preset actuel
        current_category = self.category_var.get()
        current_preset = self.preset_var.get()
        self._load_preset_to_editor(current_category, current_preset)
        
        # Rester au-dessus de la fenêtre principale
        self.transient(parent)
        self.grab_set()
    
    def _create_variables(self):
        """Crée les variables pour tous les paramètres."""
        self.waveform_var = ctk.StringVar(value="sine")
        self.attack_var = ctk.DoubleVar(value=0.01)
        self.decay_var = ctk.DoubleVar(value=0.1)
        self.sustain_var = ctk.DoubleVar(value=0.7)
        self.release_var = ctk.DoubleVar(value=0.3)
        self.cutoff_var = ctk.DoubleVar(value=2000)
        self.resonance_var = ctk.DoubleVar(value=1.0)
        self.volume_var = ctk.DoubleVar(value=0.6)
    
    def _create_widgets(self):
        """Crée l'interface de l'éditeur."""
        
        # Header
        header = ctk.CTkLabel(
            self,
            text="✏️ Édition de Preset",
            font=("Arial", 20, "bold")
        )
        header.pack(pady=15)
        
        # Frame principal avec scroll
        main_frame = ctk.CTkScrollableFrame(self, width=460, height=450)
        main_frame.pack(padx=10, pady=5, fill="both", expand=True)
        
        # Catégorie du preset
        category_frame = ctk.CTkFrame(main_frame)
        category_frame.pack(padx=10, pady=5, fill="x")
        
        ctk.CTkLabel(category_frame, text="Catégorie:", font=("Arial", 12)).pack(side="left", padx=5)
        self.category_entry_var = ctk.StringVar(value=self.category_var.get())
        
        # Dropdown pour sélectionner une catégorie existante
        categories = self.preset_manager.get_categories()
        self.category_dropdown = ctk.CTkOptionMenu(
            category_frame,
            variable=self.category_entry_var,
            values=categories,
            width=250
        )
        self.category_dropdown.pack(side="left", padx=5, fill="x", expand=True)
        
        # Nom du preset
        name_frame = ctk.CTkFrame(main_frame)
        name_frame.pack(padx=10, pady=10, fill="x")
        
        ctk.CTkLabel(name_frame, text="Nom du preset:", font=("Arial", 12)).pack(side="left", padx=5)
        self.name_entry = ctk.CTkEntry(name_frame, width=250)
        self.name_entry.pack(side="left", padx=5, fill="x", expand=True)
        self.name_entry.insert(0, self.preset_var.get())
        
        # Oscillateur
        self._create_section(main_frame, "Oscillateur")
        osc_frame = ctk.CTkFrame(main_frame)
        osc_frame.pack(padx=10, pady=5, fill="x")
        
        ctk.CTkLabel(osc_frame, text="Forme d'onde:").pack(side="left", padx=5)
        ctk.CTkOptionMenu(
            osc_frame,
            variable=self.waveform_var,
            values=["sine", "square", "sawtooth", "triangle"],
            width=150
        ).pack(side="left", padx=5)
        
        # ADSR
        self._create_section(main_frame, "Enveloppe ADSR")
        
        self._create_parameter_slider(main_frame, "Attack (ms)", self.attack_var, 1, 2000, 
                                      lambda v: self.attack_var.set(v/1000))
        self._create_parameter_slider(main_frame, "Decay (ms)", self.decay_var, 1, 2000,
                                      lambda v: self.decay_var.set(v/1000))
        self._create_parameter_slider(main_frame, "Sustain (%)", self.sustain_var, 0, 100,
                                      lambda v: self.sustain_var.set(v/100))
        self._create_parameter_slider(main_frame, "Release (ms)", self.release_var, 1, 3000,
                                      lambda v: self.release_var.set(v/1000))
        
        # Filtre
        self._create_section(main_frame, "Filtre")
        
        self._create_parameter_slider(main_frame, "Cutoff (Hz)", self.cutoff_var, 20, 10000,
                                      lambda v: self.cutoff_var.set(v))
        self._create_parameter_slider(main_frame, "Resonance", self.resonance_var, 0.1, 10.0,
                                      lambda v: self.resonance_var.set(v))
        
        # Volume
        self._create_section(main_frame, "Volume")
        
        self._create_parameter_slider(main_frame, "Master (%)", self.volume_var, 0, 100,
                                      lambda v: self.volume_var.set(v/100))
        
        # Boutons d'action
        button_frame = ctk.CTkFrame(self)
        button_frame.pack(padx=10, pady=10, fill="x")
        
        ctk.CTkButton(
            button_frame,
            text="💾 Sauvegarder",
            command=self._save_preset,
            fg_color="green",
            hover_color="darkgreen",
            width=120
        ).pack(side="left", padx=5)
        
        ctk.CTkButton(
            button_frame,
            text="💾 Sauvegarder sous...",
            command=self._save_preset_as,
            width=150
        ).pack(side="left", padx=5)
        
        ctk.CTkButton(
            button_frame,
            text="🔊 Tester",
            command=self._test_preset,
            width=100
        ).pack(side="left", padx=5)
        
        ctk.CTkButton(
            button_frame,
            text="Annuler",
            command=self.destroy,
            fg_color="gray",
            width=100
        ).pack(side="right", padx=5)
    
    def _create_section(self, parent, title):
        """Crée un titre de section."""
        ctk.CTkLabel(
            parent,
            text=title,
            font=("Arial", 14, "bold")
        ).pack(pady=(15, 5))
    
    def _create_parameter_slider(self, parent, label, variable, min_val, max_val, callback):
        """Crée un slider horizontal pour un paramètre."""
        frame = ctk.CTkFrame(parent)
        frame.pack(padx=10, pady=5, fill="x")
        
        label_widget = ctk.CTkLabel(frame, text=f"{label}:", width=120)
        label_widget.pack(side="left", padx=5)
        
        # Calculer la valeur initiale pour le slider
        current = variable.get()
        if "ms" in label:
            display_val = current * 1000
        elif "%" in label:
            display_val = current * 100
        else:
            display_val = current
        
        value_label = ctk.CTkLabel(frame, text=f"{display_val:.1f}", width=60)
        value_label.pack(side="right", padx=5)
        
        def update_label(v):
            value_label.configure(text=f"{float(v):.1f}")
            callback(float(v))
        
        slider = ctk.CTkSlider(
            frame,
            from_=min_val,
            to=max_val,
            command=update_label
        )
        slider.set(display_val)
        slider.pack(side="left", fill="x", expand=True, padx=5)
    
    def _load_preset_to_editor(self, category, preset_name):
        """Charge un preset dans l'éditeur."""
        preset = self.preset_manager.get_preset(category, preset_name)
        if preset:
            self.waveform_var.set(preset.get('waveform', 'sine'))
            self.attack_var.set(preset.get('attack', 0.01))
            self.decay_var.set(preset.get('decay', 0.1))
            self.sustain_var.set(preset.get('sustain', 0.7))
            self.release_var.set(preset.get('release', 0.3))
            self.cutoff_var.set(preset.get('filter_cutoff', 2000))
            self.resonance_var.set(preset.get('filter_resonance', 1.0))
            self.volume_var.set(preset.get('master_volume', 0.6))
    
    def _get_current_parameters(self):
        """Récupère les paramètres actuels de l'éditeur."""
        return {
            "waveform": self.waveform_var.get(),
            "attack": self.attack_var.get(),
            "decay": self.decay_var.get(),
            "sustain": self.sustain_var.get(),
            "release": self.release_var.get(),
            "filter_cutoff": self.cutoff_var.get(),
            "filter_resonance": self.resonance_var.get(),
            "master_volume": self.volume_var.get()
        }
    
    def _save_preset(self):
        """Sauvegarde le preset actuel."""
        category = self.category_entry_var.get().strip()
        name = self.name_entry.get().strip()
        
        if not category:
            messagebox.showerror("Erreur", "La catégorie ne peut pas être vide")
            return
        if not name:
            messagebox.showerror("Erreur", "Le nom du preset ne peut pas être vide")
            return
        
        parameters = self._get_current_parameters()
        self.preset_manager.add_preset(category, name, parameters)
        
        # Mettre à jour les dropdowns principaux
        self._update_main_dropdowns(category, name)
        
        messagebox.showinfo("Succès", f"Preset '{name}' sauvegardé dans '{category}' avec succès!")
        self.destroy()
    
    def _save_preset_as(self):
        """Sauvegarde le preset sous un nouveau nom."""
        # Dialog pour demander un nouveau nom
        dialog = ctk.CTkInputDialog(
            text="Entrez le nom du nouveau preset:",
            title="Sauvegarder sous..."
        )
        new_name = dialog.get_input()
        
        if new_name and new_name.strip():
            category = self.category_entry_var.get().strip()
            if not category:
                messagebox.showerror("Erreur", "La catégorie ne peut pas être vide")
                return
            
            parameters = self._get_current_parameters()
            self.preset_manager.add_preset(category, new_name.strip(), parameters)
            
            # Mettre à jour les dropdowns principaux
            self._update_main_dropdowns(category, new_name.strip())
            
            messagebox.showinfo("Succès", f"Preset '{new_name}' créé dans '{category}' avec succès!")
            self.destroy()
    
    def _test_preset(self):
        """Applique le preset temporairement pour tester le son."""
        parameters = self._get_current_parameters()
        self.synth_engine.set_waveform(parameters['waveform'])
        self.synth_engine.set_adsr(
            attack=parameters['attack'],
            decay=parameters['decay'],
            sustain=parameters['sustain'],
            release=parameters['release']
        )
        self.synth_engine.set_filter(
            cutoff=parameters['filter_cutoff'],
            resonance=parameters['filter_resonance']
        )
        self.synth_engine.set_master_volume(parameters['master_volume'])
        
        print("🔊 Preset appliqué pour test (jouez une note pour entendre)")
    
    def _update_main_dropdowns(self, category, preset_name):
        """Met à jour les dropdowns de la fenêtre principale."""
        # Mettre à jour le dropdown de catégories (au cas où une nouvelle catégorie a été créée)
        categories = self.preset_manager.get_categories()
        self.parent.category_dropdown.configure(values=categories)
        
        # Sélectionner la catégorie
        self.category_var.set(category)
        
        # Mettre à jour le dropdown de presets pour cette catégorie
        preset_names = self.preset_manager.get_presets_in_category(category)
        self.parent.preset_dropdown.configure(values=preset_names)
        
        # Sélectionner le preset
        self.preset_var.set(preset_name)
