"""
Système de presets pour sauvegarder et charger des configurations de sons.
"""

import json
import os

class PresetManager:
    """
    Gère les presets du synthétiseur organisés par catégories.
    """
    
    # Presets organisés par catégories
    DEFAULT_PRESETS = {
        "🎵 Classiques": {
            "Lead Synth": {
                "waveform": "sawtooth",
                "attack": 0.01,
                "decay": 0.1,
                "sustain": 0.7,
                "release": 0.3,
                "filter_cutoff": 2000.0,
                "filter_resonance": 1.0,
                "master_volume": 0.6
            },
            "Soft Pad": {
                "waveform": "sine",
                "attack": 0.5,
                "decay": 0.3,
                "sustain": 0.8,
                "release": 1.0,
                "filter_cutoff": 1500.0,
                "filter_resonance": 0.5,
                "master_volume": 0.5
            },
            "Brass": {
                "waveform": "square",
                "attack": 0.05,
                "decay": 0.2,
                "sustain": 0.6,
                "release": 0.2,
                "filter_cutoff": 1800.0,
                "filter_resonance": 1.5,
                "master_volume": 0.6
            },
            "Bass": {
                "waveform": "triangle",
                "attack": 0.005,
                "decay": 0.05,
                "sustain": 0.5,
                "release": 0.15,
                "filter_cutoff": 800.0,
                "filter_resonance": 1.2,
                "master_volume": 0.7
            },
            "Organ": {
                "waveform": "sine",
                "attack": 0.001,
                "decay": 0.0,
                "sustain": 1.0,
                "release": 0.05,
                "filter_cutoff": 3000.0,
                "filter_resonance": 0.7,
                "master_volume": 0.5
            },
            "Pluck": {
                "waveform": "sawtooth",
                "attack": 0.001,
                "decay": 0.3,
                "sustain": 0.1,
                "release": 0.2,
                "filter_cutoff": 2500.0,
                "filter_resonance": 0.8,
                "master_volume": 0.6
            }
        },
        
        "🔊 Bass": {
            "Sub Bass": {
                "waveform": "sine",
                "attack": 0.001,
                "decay": 0.1,
                "sustain": 0.9,
                "release": 0.1,
                "filter_cutoff": 400.0,
                "filter_resonance": 0.5,
                "master_volume": 0.8
            },
            "Acid Bass": {
                "waveform": "square",
                "attack": 0.001,
                "decay": 0.15,
                "sustain": 0.3,
                "release": 0.1,
                "filter_cutoff": 1200.0,
                "filter_resonance": 5.0,
                "master_volume": 0.65
            },
            "Wobble Bass": {
                "waveform": "sawtooth",
                "attack": 0.005,
                "decay": 0.2,
                "sustain": 0.6,
                "release": 0.15,
                "filter_cutoff": 600.0,
                "filter_resonance": 3.5,
                "master_volume": 0.7
            }
        },
        
        "🎸 Leads": {
            "Supersaw": {
                "waveform": "sawtooth",
                "attack": 0.02,
                "decay": 0.15,
                "sustain": 0.75,
                "release": 0.4,
                "filter_cutoff": 3500.0,
                "filter_resonance": 1.2,
                "master_volume": 0.6
            },
            "Square Lead": {
                "waveform": "square",
                "attack": 0.01,
                "decay": 0.1,
                "sustain": 0.7,
                "release": 0.25,
                "filter_cutoff": 2800.0,
                "filter_resonance": 1.5,
                "master_volume": 0.6
            },
            "Synth Bells": {
                "waveform": "sine",
                "attack": 0.001,
                "decay": 0.8,
                "sustain": 0.2,
                "release": 0.6,
                "filter_cutoff": 4000.0,
                "filter_resonance": 0.8,
                "master_volume": 0.55
            }
        },
        
        "🌊 Pads": {
            "Warm Pad": {
                "waveform": "triangle",
                "attack": 0.8,
                "decay": 0.4,
                "sustain": 0.85,
                "release": 1.5,
                "filter_cutoff": 1800.0,
                "filter_resonance": 0.6,
                "master_volume": 0.5
            },
            "Dark Pad": {
                "waveform": "sawtooth",
                "attack": 1.0,
                "decay": 0.5,
                "sustain": 0.9,
                "release": 2.0,
                "filter_cutoff": 1200.0,
                "filter_resonance": 0.8,
                "master_volume": 0.45
            },
            "Ethereal": {
                "waveform": "sine",
                "attack": 1.2,
                "decay": 0.6,
                "sustain": 0.75,
                "release": 2.5,
                "filter_cutoff": 2500.0,
                "filter_resonance": 0.4,
                "master_volume": 0.48
            }
        },
        
        "🥁 Percussifs": {
            "Stab": {
                "waveform": "square",
                "attack": 0.001,
                "decay": 0.15,
                "sustain": 0.0,
                "release": 0.1,
                "filter_cutoff": 3000.0,
                "filter_resonance": 2.0,
                "master_volume": 0.7
            },
            "Kick Drum": {
                "waveform": "sine",
                "attack": 0.001,
                "decay": 0.25,
                "sustain": 0.0,
                "release": 0.05,
                "filter_cutoff": 200.0,
                "filter_resonance": 1.5,
                "master_volume": 0.8
            },
            "Zap": {
                "waveform": "square",
                "attack": 0.001,
                "decay": 0.08,
                "sustain": 0.0,
                "release": 0.05,
                "filter_cutoff": 5000.0,
                "filter_resonance": 8.0,
                "master_volume": 0.65
            }
        },
        
        "✨ Spéciaux": {
            "Hoover": {
                "waveform": "sawtooth",
                "attack": 0.02,
                "decay": 0.2,
                "sustain": 0.8,
                "release": 0.3,
                "filter_cutoff": 1500.0,
                "filter_resonance": 4.0,
                "master_volume": 0.6
            },
            "Rave Stab": {
                "waveform": "square",
                "attack": 0.001,
                "decay": 0.3,
                "sustain": 0.2,
                "release": 0.15,
                "filter_cutoff": 2500.0,
                "filter_resonance": 6.0,
                "master_volume": 0.65
            }
        },
        
        "🎹 Pianos": {
            "Piano à Queue": {
                "waveform": "triangle",
                "attack": 0.002,
                "decay": 0.4,
                "sustain": 0.3,
                "release": 0.8,
                "filter_cutoff": 5000.0,
                "filter_resonance": 0.5,
                "master_volume": 0.65
            },
            "Piano Électrique": {
                "waveform": "sine",
                "attack": 0.005,
                "decay": 0.6,
                "sustain": 0.2,
                "release": 1.0,
                "filter_cutoff": 3500.0,
                "filter_resonance": 1.2,
                "master_volume": 0.6
            },
            "Honky-Tonk": {
                "waveform": "sawtooth",
                "attack": 0.001,
                "decay": 0.3,
                "sustain": 0.4,
                "release": 0.5,
                "filter_cutoff": 4000.0,
                "filter_resonance": 2.0,
                "master_volume": 0.6
            },
            "Clavecin": {
                "waveform": "sawtooth",
                "attack": 0.001,
                "decay": 0.15,
                "sustain": 0.0,
                "release": 0.2,
                "filter_cutoff": 6000.0,
                "filter_resonance": 0.8,
                "master_volume": 0.55
            }
        },
        
        "🎻 Cordes": {
            "Violon": {
                "waveform": "sawtooth",
                "attack": 0.08,
                "decay": 0.2,
                "sustain": 0.75,
                "release": 0.4,
                "filter_cutoff": 3000.0,
                "filter_resonance": 1.8,
                "master_volume": 0.55
            },
            "Ensemble Cordes": {
                "waveform": "sawtooth",
                "attack": 0.15,
                "decay": 0.3,
                "sustain": 0.85,
                "release": 0.6,
                "filter_cutoff": 2500.0,
                "filter_resonance": 1.0,
                "master_volume": 0.5
            },
            "Pizzicato": {
                "waveform": "triangle",
                "attack": 0.001,
                "decay": 0.25,
                "sustain": 0.0,
                "release": 0.15,
                "filter_cutoff": 4000.0,
                "filter_resonance": 1.2,
                "master_volume": 0.6
            },
            "Harpe": {
                "waveform": "sine",
                "attack": 0.001,
                "decay": 0.8,
                "sustain": 0.1,
                "release": 1.2,
                "filter_cutoff": 5000.0,
                "filter_resonance": 0.6,
                "master_volume": 0.55
            },
            "Contrebasse": {
                "waveform": "triangle",
                "attack": 0.02,
                "decay": 0.15,
                "sustain": 0.7,
                "release": 0.3,
                "filter_cutoff": 600.0,
                "filter_resonance": 1.5,
                "master_volume": 0.7
            }
        },
        
        "🎺 Cuivres": {
            "Trompette": {
                "waveform": "square",
                "attack": 0.03,
                "decay": 0.1,
                "sustain": 0.8,
                "release": 0.2,
                "filter_cutoff": 2800.0,
                "filter_resonance": 2.5,
                "master_volume": 0.6
            },
            "Trombone": {
                "waveform": "square",
                "attack": 0.05,
                "decay": 0.15,
                "sustain": 0.75,
                "release": 0.25,
                "filter_cutoff": 1800.0,
                "filter_resonance": 2.2,
                "master_volume": 0.6
            },
            "Ensemble Cuivres": {
                "waveform": "sawtooth",
                "attack": 0.08,
                "decay": 0.2,
                "sustain": 0.8,
                "release": 0.3,
                "filter_cutoff": 2200.0,
                "filter_resonance": 2.0,
                "master_volume": 0.55
            },
            "Cor": {
                "waveform": "triangle",
                "attack": 0.06,
                "decay": 0.2,
                "sustain": 0.7,
                "release": 0.4,
                "filter_cutoff": 1500.0,
                "filter_resonance": 1.5,
                "master_volume": 0.6
            }
        },
        
        "🎷 Vents": {
            "Flûte": {
                "waveform": "sine",
                "attack": 0.05,
                "decay": 0.1,
                "sustain": 0.85,
                "release": 0.3,
                "filter_cutoff": 4500.0,
                "filter_resonance": 0.8,
                "master_volume": 0.55
            },
            "Saxophone": {
                "waveform": "sawtooth",
                "attack": 0.04,
                "decay": 0.15,
                "sustain": 0.75,
                "release": 0.25,
                "filter_cutoff": 2500.0,
                "filter_resonance": 2.5,
                "master_volume": 0.6
            },
            "Clarinette": {
                "waveform": "square",
                "attack": 0.03,
                "decay": 0.1,
                "sustain": 0.8,
                "release": 0.2,
                "filter_cutoff": 3000.0,
                "filter_resonance": 1.8,
                "master_volume": 0.55
            },
            "Hautbois": {
                "waveform": "sawtooth",
                "attack": 0.04,
                "decay": 0.12,
                "sustain": 0.75,
                "release": 0.22,
                "filter_cutoff": 2800.0,
                "filter_resonance": 2.2,
                "master_volume": 0.55
            }
        },
        
        "🎸 Guitares": {
            "Guitare Acoustique": {
                "waveform": "sawtooth",
                "attack": 0.002,
                "decay": 0.4,
                "sustain": 0.3,
                "release": 0.6,
                "filter_cutoff": 3500.0,
                "filter_resonance": 1.0,
                "master_volume": 0.6
            },
            "Guitare Électrique Clean": {
                "waveform": "sawtooth",
                "attack": 0.003,
                "decay": 0.3,
                "sustain": 0.5,
                "release": 0.4,
                "filter_cutoff": 4000.0,
                "filter_resonance": 1.2,
                "master_volume": 0.6
            },
            "Guitare Distorsion": {
                "waveform": "square",
                "attack": 0.002,
                "decay": 0.25,
                "sustain": 0.6,
                "release": 0.35,
                "filter_cutoff": 2500.0,
                "filter_resonance": 3.5,
                "master_volume": 0.65
            },
            "Guitare 12 Cordes": {
                "waveform": "sawtooth",
                "attack": 0.001,
                "decay": 0.5,
                "sustain": 0.4,
                "release": 0.8,
                "filter_cutoff": 4500.0,
                "filter_resonance": 1.5,
                "master_volume": 0.6
            }
        }
    }
    
    def __init__(self, presets_file="presets.json"):
        """
        Initialise le gestionnaire de presets.
        
        Args:
            presets_file: Chemin du fichier de presets
        """
        self.presets_file = presets_file
        self.presets = {}
        self.load_presets()
    
    def load_presets(self):
        """Charge les presets depuis le fichier, ou utilise les presets par défaut."""
        if os.path.exists(self.presets_file):
            try:
                with open(self.presets_file, 'r', encoding='utf-8') as f:
                    self.presets = json.load(f)
                print(f"✓ Presets chargés depuis {self.presets_file}")
            except Exception as e:
                print(f"Erreur lors du chargement des presets: {e}")
                self.presets = self.DEFAULT_PRESETS.copy()
        else:
            # Utiliser les presets par défaut
            import copy
            self.presets = copy.deepcopy(self.DEFAULT_PRESETS)
            self.save_presets()
    
    def save_presets(self):
        """Sauvegarde les presets dans le fichier."""
        try:
            with open(self.presets_file, 'w', encoding='utf-8') as f:
                json.dump(self.presets, f, indent=2, ensure_ascii=False)
            print(f"✓ Presets sauvegardés dans {self.presets_file}")
        except Exception as e:
            print(f"Erreur lors de la sauvegarde des presets: {e}")
    
    def get_categories(self):
        """Retourne la liste des catégories de presets."""
        return list(self.presets.keys())
    
    def get_presets_in_category(self, category):
        """Retourne la liste des presets dans une catégorie."""
        if category in self.presets:
            return list(self.presets[category].keys())
        return []
    
    def get_preset(self, category, preset_name):
        """
        Récupère un preset spécifique.
        
        Args:
            category: Catégorie du preset
            preset_name: Nom du preset
        
        Returns:
            Dictionnaire des paramètres du preset ou None
        """
        if category in self.presets and preset_name in self.presets[category]:
            return self.presets[category][preset_name].copy()
        return None
    
    def add_preset(self, category, preset_name, parameters):
        """
        Ajoute ou met à jour un preset.
        
        Args:
            category: Catégorie du preset
            preset_name: Nom du preset
            parameters: Dictionnaire des paramètres
        """
        if category not in self.presets:
            self.presets[category] = {}
        
        self.presets[category][preset_name] = parameters.copy()
        self.save_presets()
    
    def delete_preset(self, category, preset_name):
        """
        Supprime un preset.
        
        Args:
            category: Catégorie du preset
            preset_name: Nom du preset
        
        Returns:
            True si supprimé, False sinon
        """
        if category in self.presets and preset_name in self.presets[category]:
            del self.presets[category][preset_name]
            self.save_presets()
            return True
        return False
    
    def apply_preset(self, synth_engine, category, preset_name):
        """
        Applique un preset au moteur de synthèse.
        
        Args:
            synth_engine: Instance de SynthEngine
            category: Catégorie du preset
            preset_name: Nom du preset
        
        Returns:
            True si appliqué avec succès, False sinon
        """
        preset = self.get_preset(category, preset_name)
        if not preset:
            print(f"Preset '{preset_name}' dans la catégorie '{category}' non trouvé.")
            return False
        
        try:
            # Appliquer les paramètres
            synth_engine.set_waveform(preset.get('waveform', 'sine'))
            synth_engine.set_adsr(
                attack=preset.get('attack', 0.01),
                decay=preset.get('decay', 0.1),
                sustain=preset.get('sustain', 0.7),
                release=preset.get('release', 0.3)
            )
            synth_engine.set_filter(
                cutoff=preset.get('filter_cutoff', 2000.0),
                resonance=preset.get('filter_resonance', 1.0)
            )
            synth_engine.set_master_volume(preset.get('master_volume', 0.5))
            
            print(f"✓ Preset '{preset_name}' appliqué.")
            return True
            
        except Exception as e:
            print(f"Erreur lors de l'application du preset: {e}")
            return False
