"""
Gestionnaire de projets pour sauvegarder/charger des sessions de séquenceur.
"""

import json
import os
from datetime import datetime
from typing import Optional, Dict, Any
from sequencer_engine import SequencerEngine, Track, MidiEvent, NoteEventType


class ProjectManager:
    """
    Gère la sauvegarde et le chargement de projets de séquenceur.
    """
    
    def __init__(self, projects_dir: str = "projects"):
        """
        Initialise le gestionnaire de projets.
        
        Args:
            projects_dir: Répertoire de sauvegarde des projets
        """
        self.projects_dir = projects_dir
        os.makedirs(self.projects_dir, exist_ok=True)
    
    def save_project(self, sequencer: SequencerEngine, filepath: str, 
                    project_name: str = "Untitled") -> bool:
        """
        Sauvegarde un projet complet.
        
        Args:
            sequencer: Instance du séquenceur à sauvegarder
            filepath: Chemin du fichier (sans extension)
            project_name: Nom du projet
            
        Returns:
            True si succès, False sinon
        """
        try:
            # Ajouter l'extension si manquante
            if not filepath.endswith('.project.json'):
                filepath = filepath + '.project.json'
            
            # Créer le chemin complet
            full_path = os.path.join(self.projects_dir, os.path.basename(filepath))
            
            # Construire les données du projet
            project_data = {
                "version": "1.0",
                "name": project_name,
                "created_at": datetime.now().isoformat(),
                "bpm": sequencer.bpm,
                "ppqn": sequencer.ppqn,
                "loop_enabled": sequencer.loop_enabled,
                "loop_start": sequencer.loop_start,
                "loop_end": sequencer.loop_end,
                "tracks": []
            }
            
            # Sauvegarder chaque piste
            for track in sequencer.tracks:
                track_data = {
                    "name": track.name,
                    "instrument_type": track.instrument_type,
                    "instrument_category": getattr(track, 'instrument_category', 'Piano'),
                    "instrument_name": track.instrument_name,
                    "bank": track.bank,
                    "preset": track.preset,
                    "volume": track.volume,
                    "solo": track.solo,
                    "mute": track.mute,
                    "events": []
                }
                
                # Sauvegarder les événements
                for event in track.events:
                    event_data = {
                        "timestamp": event.timestamp,
                        "note": event.note,
                        "velocity": event.velocity,
                        "event_type": event.event_type.value,
                        "duration": event.duration
                    }
                    track_data["events"].append(event_data)
                
                project_data["tracks"].append(track_data)
            
            # Écrire le fichier JSON
            with open(full_path, 'w', encoding='utf-8') as f:
                json.dump(project_data, f, indent=2, ensure_ascii=False)
            
            print(f"✓ Projet sauvegardé : {full_path}")
            return True
            
        except Exception as e:
            print(f"Erreur lors de la sauvegarde du projet : {e}")
            return False
    
    def load_project(self, filepath: str, sequencer: SequencerEngine) -> bool:
        """
        Charge un projet.
        
        Args:
            filepath: Chemin du fichier projet
            sequencer: Instance du séquenceur à remplir
            
        Returns:
            True si succès, False sinon
        """
        try:
            # Ajouter l'extension si manquante
            if not filepath.endswith('.project.json'):
                filepath = filepath + '.project.json'
            
            # Créer le chemin complet
            full_path = os.path.join(self.projects_dir, os.path.basename(filepath))
            
            if not os.path.exists(full_path):
                print(f"Fichier projet introuvable : {full_path}")
                return False
            
            # Lire le fichier JSON
            with open(full_path, 'r', encoding='utf-8') as f:
                project_data = json.load(f)
            
            # Arrêter la lecture en cours
            sequencer.stop()
            
            # Restaurer les paramètres globaux
            sequencer.set_bpm(project_data.get("bpm", 120))
            sequencer.ppqn = project_data.get("ppqn", 96)
            sequencer.loop_enabled = project_data.get("loop_enabled", False)
            sequencer.loop_start = project_data.get("loop_start", 0.0)
            sequencer.loop_end = project_data.get("loop_end", 16.0)
            sequencer.set_position(0.0)
            
            # Effacer les pistes existantes
            sequencer.tracks.clear()
            
            # Charger les pistes
            for track_data in project_data.get("tracks", []):
                track = Track(
                    name=track_data.get("name", "Piste"),
                    instrument_type=track_data.get("instrument_type", "synth"),
                    instrument_category=track_data.get("instrument_category", "Piano"),
                    instrument_name=track_data.get("instrument_name", "Lead Synth"),
                    bank=track_data.get("bank", 0),
                    preset=track_data.get("preset", 0),
                    volume=track_data.get("volume", 0.8),
                    solo=track_data.get("solo", False),
                    mute=track_data.get("mute", False)
                )
                
                # Charger les événements
                for event_data in track_data.get("events", []):
                    event = MidiEvent(
                        timestamp=event_data["timestamp"],
                        note=event_data["note"],
                        velocity=event_data["velocity"],
                        event_type=NoteEventType(event_data["event_type"]),
                        duration=event_data.get("duration")
                    )
                    track.add_event(event)
                
                sequencer.tracks.append(track)
            
            print(f"✓ Projet chargé : {full_path}")
            print(f"  - {len(sequencer.tracks)} pistes")
            total_events = sum(len(t.events) for t in sequencer.tracks)
            print(f"  - {total_events} événements MIDI")
            return True
            
        except Exception as e:
            print(f"Erreur lors du chargement du projet : {e}")
            return False
    
    def list_projects(self) -> list[str]:
        """
        Liste tous les projets disponibles.
        
        Returns:
            Liste des noms de fichiers projet
        """
        try:
            files = os.listdir(self.projects_dir)
            return [f for f in files if f.endswith('.project.json')]
        except Exception as e:
            print(f"Erreur lors du listage des projets : {e}")
            return []
    
    def delete_project(self, filepath: str) -> bool:
        """
        Supprime un projet.
        
        Args:
            filepath: Chemin du fichier projet
            
        Returns:
            True si succès, False sinon
        """
        try:
            if not filepath.endswith('.project.json'):
                filepath = filepath + '.project.json'
            
            full_path = os.path.join(self.projects_dir, os.path.basename(filepath))
            
            if os.path.exists(full_path):
                os.remove(full_path)
                print(f"✓ Projet supprimé : {full_path}")
                return True
            else:
                print(f"Fichier projet introuvable : {full_path}")
                return False
                
        except Exception as e:
            print(f"Erreur lors de la suppression du projet : {e}")
            return False
    
    def export_midi(self, sequencer: SequencerEngine, filepath: str) -> bool:
        """
        Exporte le projet au format MIDI standard.
        
        Args:
            sequencer: Instance du séquenceur
            filepath: Chemin du fichier MIDI à créer
            
        Returns:
            True si succès, False sinon
        """
        try:
            import mido
            from mido import Message, MidiFile, MidiTrack, MetaMessage
            
            # Créer le fichier MIDI
            mid = MidiFile(ticks_per_beat=sequencer.ppqn)
            
            # Ajouter une piste de tempo
            tempo_track = MidiTrack()
            mid.tracks.append(tempo_track)
            tempo_track.append(MetaMessage('set_tempo', tempo=mido.bpm2tempo(sequencer.bpm)))
            
            # Ajouter chaque piste du séquenceur
            for track in sequencer.tracks:
                if track.mute or len(track.events) == 0:
                    continue  # Ignorer les pistes mutées ou vides
                
                midi_track = MidiTrack()
                mid.tracks.append(midi_track)
                midi_track.append(MetaMessage('track_name', name=track.name))
                
                # Convertir les événements en messages MIDI
                last_time = 0
                
                for event in sorted(track.events, key=lambda e: e.timestamp):
                    # Calculer le delta time en ticks
                    time_ticks = int(event.timestamp * sequencer.ppqn / 4.0)  # Convertir beats en ticks
                    delta_ticks = time_ticks - last_time
                    
                    # Note ON
                    midi_track.append(Message('note_on', 
                                            note=event.note,
                                            velocity=event.velocity,
                                            time=delta_ticks))
                    
                    # Note OFF (après la durée)
                    if event.duration:
                        duration_ticks = int(event.duration * sequencer.ppqn / 4.0)
                        midi_track.append(Message('note_off',
                                                note=event.note,
                                                velocity=0,
                                                time=duration_ticks))
                        last_time = time_ticks + duration_ticks
                    else:
                        last_time = time_ticks
            
            # Sauvegarder le fichier
            mid.save(filepath)
            print(f"✓ Fichier MIDI exporté : {filepath}")
            return True
            
        except ImportError:
            print("❌ Module 'mido' requis pour l'export MIDI. Installez-le avec: pip install mido")
            return False
        except Exception as e:
            print(f"Erreur lors de l'export MIDI : {e}")
            return False
