"""
Interface graphique du séquenceur multi-pistes avec CustomTkinter.
"""

import customtkinter as ctk
from typing import Optional
from sequencer_engine import SequencerEngine, Track, TransportState
from project_manager import ProjectManager


class SequencerGUI(ctk.CTkFrame):
    """
    Interface graphique complète du séquenceur.
    """
    
    def __init__(self, parent, sequencer: SequencerEngine, preset_manager=None, sf2_instruments_data=None, **kwargs):
        super().__init__(parent, **kwargs)
        
        self.sequencer = sequencer
        self.project_manager = ProjectManager()
        self.preset_manager = preset_manager
        self.sf2_instruments_data = sf2_instruments_data
        self.timeline_canvases = {}  # Stockage des canvas pour mise à jour optimisée
        
        # Configuration de la grille
        self.grid_rowconfigure(0, weight=0)  # Transport bar
        self.grid_rowconfigure(1, weight=1)  # Zone principale
        self.grid_columnconfigure(0, weight=1)
        
        # Créer les composants
        self._create_transport_bar()
        self._create_main_area()
        
        # Démarrer la mise à jour de l'UI
        self._start_ui_updates()
    
    def _create_transport_bar(self):
        """Crée la barre de transport (Play, Stop, Record, BPM, etc.)."""
        self.transport_frame = ctk.CTkFrame(self, fg_color="#2b2b2b")
        self.transport_frame.grid(row=0, column=0, padx=10, pady=(10, 5), sticky="ew")
        
        # --- Contrôles de transport ---
        controls_frame = ctk.CTkFrame(self.transport_frame, fg_color="transparent")
        controls_frame.pack(side="left", padx=10, pady=10)
        
        # Bouton Play
        self.play_button = ctk.CTkButton(
            controls_frame,
            text="▶ Play",
            command=self._on_play,
            width=80,
            fg_color="#1f6aa5",
            hover_color="#1a5a8f"
        )
        self.play_button.pack(side="left", padx=2)
        
        # Bouton Stop
        self.stop_button = ctk.CTkButton(
            controls_frame,
            text="⏹ Stop",
            command=self._on_stop,
            width=80,
            fg_color="#555555",
            hover_color="#444444"
        )
        self.stop_button.pack(side="left", padx=2)
        
        # Bouton Record
        self.record_button = ctk.CTkButton(
            controls_frame,
            text="⏺ Record",
            command=self._on_record,
            width=80,
            fg_color="#c93232",
            hover_color="#a82828"
        )
        self.record_button.pack(side="left", padx=2)
        
        # Bouton PANIC
        panic_button = ctk.CTkButton(
            controls_frame,
            text="⚠️ PANIC",
            command=self._on_panic,
            width=80,
            fg_color="#8b0000",
            hover_color="#6b0000"
        )
        panic_button.pack(side="left", padx=10)
        
        # --- BPM ---
        bpm_frame = ctk.CTkFrame(self.transport_frame, fg_color="transparent")
        bpm_frame.pack(side="left", padx=10, pady=10)
        
        ctk.CTkLabel(bpm_frame, text="BPM:", font=("Arial", 12)).pack(side="left", padx=5)
        
        self.bpm_var = ctk.StringVar(value=str(self.sequencer.bpm))
        self.bpm_entry = ctk.CTkEntry(
            bpm_frame,
            textvariable=self.bpm_var,
            width=60
        )
        self.bpm_entry.pack(side="left", padx=2)
        self.bpm_entry.bind("<Return>", lambda e: self._on_bpm_changed())
        
        # --- Position ---
        position_frame = ctk.CTkFrame(self.transport_frame, fg_color="transparent")
        position_frame.pack(side="left", padx=10, pady=10)
        
        ctk.CTkLabel(position_frame, text="Position:", font=("Arial", 12)).pack(side="left", padx=5)
        
        self.position_label = ctk.CTkLabel(
            position_frame,
            text="0.0",
            font=("Arial", 12, "bold"),
            width=80
        )
        self.position_label.pack(side="left", padx=2)
        
        # --- Métronome ---
        metronome_frame = ctk.CTkFrame(self.transport_frame, fg_color="transparent")
        metronome_frame.pack(side="left", padx=10, pady=10)
        
        self.metronome_var = ctk.BooleanVar(value=self.sequencer.metronome_enabled)
        self.metronome_checkbox = ctk.CTkCheckBox(
            metronome_frame,
            text="Métronome",
            variable=self.metronome_var,
            command=self._on_metronome_toggle
        )
        self.metronome_checkbox.pack(side="left", padx=5)
        
        # --- Quantification ---
        quant_frame = ctk.CTkFrame(self.transport_frame, fg_color="transparent")
        quant_frame.pack(side="left", padx=10, pady=10)
        
        self.quantize_var = ctk.BooleanVar(value=self.sequencer.quantize_enabled)
        self.quantize_checkbox = ctk.CTkCheckBox(
            quant_frame,
            text="Quantize",
            variable=self.quantize_var,
            command=self._on_quantize_toggle
        )
        self.quantize_checkbox.pack(side="left", padx=5)
        
        self.quantize_value_var = ctk.StringVar(value="16")
        self.quantize_menu = ctk.CTkOptionMenu(
            quant_frame,
            values=["4", "8", "16", "32"],
            variable=self.quantize_value_var,
            command=self._on_quantize_value_changed,
            width=60
        )
        self.quantize_menu.pack(side="left", padx=2)
    
    def _create_main_area(self):
        """Crée la zone principale (liste de pistes + piano roll)."""
        self.main_frame = ctk.CTkFrame(self, fg_color="transparent")
        self.main_frame.grid(row=1, column=0, padx=10, pady=5, sticky="nsew")
        
        self.main_frame.grid_rowconfigure(0, weight=1)
        self.main_frame.grid_columnconfigure(0, weight=0)  # Liste de pistes
        self.main_frame.grid_columnconfigure(1, weight=1)  # Piano roll
        
        # Liste de pistes à gauche
        self._create_track_list()
        
        # Piano roll à droite
        self._create_piano_roll()
    
    def _create_track_list(self):
        """Crée la liste des pistes avec contrôles."""
        self.track_list_frame = ctk.CTkScrollableFrame(
            self.main_frame,
            width=300,
            fg_color="#1a1a1a"
        )
        self.track_list_frame.grid(row=0, column=0, padx=(0, 5), pady=0, sticky="nsew")
        
        # En-tête
        header_frame = ctk.CTkFrame(self.track_list_frame, fg_color="#2b2b2b")
        header_frame.pack(fill="x", padx=5, pady=(5, 10))
        
        ctk.CTkLabel(
            header_frame,
            text="🎚️ PISTES",
            font=("Arial", 14, "bold")
        ).pack(side="left", padx=10, pady=8)
        
        # Bouton ajouter piste
        add_track_button = ctk.CTkButton(
            header_frame,
            text="+ Ajouter",
            command=self._on_add_track,
            width=80,
            height=28,
            fg_color="#1f6aa5"
        )
        add_track_button.pack(side="right", padx=10, pady=5)
        
        # Liste des pistes
        self.track_widgets = []
        self._refresh_track_list()
    
    def _refresh_track_list(self):
        """Rafraîchit l'affichage de la liste des pistes."""
        # Supprimer les widgets existants
        for widget in self.track_widgets:
            widget.destroy()
        self.track_widgets.clear()
        
        # Créer un widget pour chaque piste
        for idx, track in enumerate(self.sequencer.tracks):
            track_widget = self._create_track_widget(idx, track)
            track_widget.pack(fill="x", padx=5, pady=2)
            self.track_widgets.append(track_widget)
    
    def _create_track_widget(self, idx: int, track: Track):
        """Crée le widget pour une piste (contrôles uniquement)."""
        frame = ctk.CTkFrame(self.track_list_frame, fg_color="#2b2b2b")
        
        # Ligne 1: Nom et boutons
        top_row = ctk.CTkFrame(frame, fg_color="transparent")
        top_row.pack(fill="x", padx=5, pady=3)
        
        # Bouton Arm (enregistrement)
        arm_color = "#c93232" if track.armed else "#555555"
        arm_button = ctk.CTkButton(
            top_row,
            text="⏺",
            command=lambda: self._on_arm_track(idx),
            width=30,
            height=24,
            fg_color=arm_color,
            hover_color="#a82828" if track.armed else "#444444"
        )
        arm_button.pack(side="left", padx=2)
        
        # Nom de la piste
        name_label = ctk.CTkLabel(
            top_row,
            text=track.name,
            font=("Arial", 11, "bold"),
            anchor="w",
            width=100
        )
        name_label.pack(side="left", padx=5)
        
        # Bouton Solo
        solo_color = "#ffaa00" if track.solo else "#555555"
        solo_button = ctk.CTkButton(
            top_row,
            text="S",
            command=lambda: self._on_solo_track(idx),
            width=24,
            height=24,
            fg_color=solo_color
        )
        solo_button.pack(side="left", padx=1)
        
        # Bouton Mute
        mute_color = "#ff3333" if track.mute else "#555555"
        mute_button = ctk.CTkButton(
            top_row,
            text="M",
            command=lambda: self._on_mute_track(idx),
            width=24,
            height=24,
            fg_color=mute_color
        )
        mute_button.pack(side="left", padx=1)
        
        # Bouton Supprimer
        if len(self.sequencer.tracks) > 1:
            delete_button = ctk.CTkButton(
                top_row,
                text="🗑",
                command=lambda: self._on_delete_track(idx),
                width=24,
                height=24,
                fg_color="#8b0000",
                hover_color="#6b0000"
            )
            delete_button.pack(side="left", padx=1)
        
        # Ligne 2: Sélecteur d'instrument
        instrument_row = ctk.CTkFrame(frame, fg_color="transparent")
        instrument_row.pack(fill="x", padx=5, pady=(0, 3))
        
        ctk.CTkLabel(
            instrument_row,
            text="🎹",
            font=("Arial", 10)
        ).pack(side="left", padx=(5, 2))
        
        # Bouton pour choisir l'instrument
        instrument_button = ctk.CTkButton(
            instrument_row,
            text=track.instrument_name,
            command=lambda: self._show_instrument_selector(idx),
            width=200,
            height=20,
            font=("Arial", 9),
            fg_color="#3a3a3a",
            hover_color="#4a4a4a"
        )
        instrument_button.pack(side="left", padx=2)
        
        # Ligne 3: Volume
        volume_row = ctk.CTkFrame(frame, fg_color="transparent")
        volume_row.pack(fill="x", padx=5, pady=(0, 3))
        
        ctk.CTkLabel(volume_row, text="Vol:", font=("Arial", 8)).pack(side="left", padx=(5, 2))
        
        volume_slider = ctk.CTkSlider(
            volume_row,
            from_=0.0,
            to=1.0,
            command=lambda v: self._on_track_volume_changed(idx, v),
            height=12
        )
        volume_slider.set(track.volume)
        volume_slider.pack(side="left", fill="x", expand=True, padx=3)
        
        return frame
    
    def _draw_timeline(self, canvas, track: Track):
        """Dessine les notes sur la timeline."""
        canvas.delete("all")  # Effacer le contenu précédent
        
        if not track.events:
            # Afficher "vide" si pas de notes
            canvas.create_text(125, 20, text="(vide)", fill="#666666", font=("Arial", 9))
            return
        
        # Trouver la plage temporelle
        max_time = max(e.timestamp + (e.duration or 0) for e in track.events)
        max_time = max(max_time, 16.0)  # Au moins 4 mesures
        
        # Dessiner une grille de fond (mesures)
        canvas.create_line(0, 20, 250, 20, fill="#333333", dash=(2, 2))
        
        # Dessiner les notes
        for event in track.events:
            # Position horizontale proportionnelle au temps
            x_start = (event.timestamp / max_time) * 250
            duration = event.duration or 0.5
            x_end = ((event.timestamp + duration) / max_time) * 250
            width = max(2, x_end - x_start)
            
            # Position verticale basée sur la note (pitch)
            # Notes MIDI 0-127, afficher de 30 à 100 (Piano)
            note_normalized = (event.note - 30) / 70.0  # 0.0 = grave, 1.0 = aigu
            y = 35 - (note_normalized * 30)  # Inversé (grave en bas)
            y = max(5, min(35, y))  # Limiter à la hauteur du canvas
            
            # Couleur basée sur la vélocité
            velocity_color = int(127 + (event.velocity / 127) * 100)  # 127-227
            color = f"#{velocity_color:02x}{velocity_color//2:02x}55"
            
            # Dessiner la note
            canvas.create_rectangle(
                x_start, y-3, x_start + width, y+3,
                fill=color,
                outline="",
                tags="note"
            )
        
        # Texte: nombre de notes
        count_text = f"{len(track.events)} notes"
        canvas.create_text(5, 5, text=count_text, fill="#888888", font=("Arial", 8), anchor="nw")
    
    
    def _create_piano_roll(self):
        """Crée la vue timeline multi-pistes."""
        self.piano_roll_frame = ctk.CTkFrame(self.main_frame, fg_color="#0a0a0a")
        self.piano_roll_frame.grid(row=0, column=1, padx=0, pady=0, sticky="nsew")
        
        # En-tête
        header = ctk.CTkFrame(self.piano_roll_frame, fg_color="#2b2b2b", height=40)
        header.pack(fill="x", padx=0, pady=0)
        header.pack_propagate(False)
        
        ctk.CTkLabel(
            header,
            text="🎼 TIMELINE MULTI-PISTES",
            font=("Arial", 14, "bold")
        ).pack(side="left", padx=15, pady=10)
        
        # Frame scrollable pour les timelines
        self.timeline_scroll_frame =ctk.CTkScrollableFrame(
            self.piano_roll_frame,
            fg_color="#1a1a1a"
        )
        self.timeline_scroll_frame.pack(fill="both", expand=True, padx=5, pady=5)
        
        # Initialiser le dictionnaire des canvas
        self.timeline_canvases = {}
        
        # Créer les timelines initiales
        self._refresh_timeline_view()
    
    def _refresh_timeline_view(self):
        """Rafraîchit la vue timeline multi-pistes."""
        # Nettoyer les anciens canvas
        for widget in self.timeline_scroll_frame.winfo_children():
            widget.destroy()
        self.timeline_canvases.clear()
        
        # Créer une timeline pour chaque piste
        import tkinter as tk
        for idx, track in enumerate(self.sequencer.tracks):
            # Frame pour cette piste
            track_timeline_frame = ctk.CTkFrame(self.timeline_scroll_frame, fg_color="#2b2b2b")
            track_timeline_frame.pack(fill="x", padx=5, pady=3)
            
            # Nom de la piste à gauche
            name_label = ctk.CTkLabel(
                track_timeline_frame,
                text=f"{track.name}",
                font=("Arial", 10, "bold"),
                width=100,
                anchor="w"
            )
            name_label.pack(side="left", padx=10, pady=5)
            
            # Canvas timeline
            canvas = tk.Canvas(
                track_timeline_frame,
                height=60,
                bg="#1a1a1a",
                highlightthickness=1,
                highlightbackground="#444444"
            )
            canvas.pack(side="left", fill="both", expand=True, padx=5, pady=5)
            
            # Stocker la référence
            canvas.track_idx = idx
            self.timeline_canvases[idx] = canvas
            
            # Dessiner les notes
            self._draw_timeline(canvas, track)
            
            # Double-clic pour piano roll détaillé
            canvas.bind("<Double-Button-1>", lambda e, i=idx: self._on_track_double_click(i))
    
    def _show_instrument_selector(self, track_idx: int):
        """Affiche la popup de sélection d'instrument avec toggle Synth/SF2."""
        track = self.sequencer.tracks[track_idx]
        
        # Créer une fenêtre Toplevel
        selector_window = ctk.CTkToplevel(self)
        selector_window.title(f"Instrument : {track.name}")
        selector_window.geometry("950x700")
        selector_window.transient(self.master)
        selector_window.attributes('-topmost', True)
        
        # Frame principal
        main_frame = ctk.CTkFrame(selector_window, fg_color="transparent")
        main_frame.pack(fill="both", expand=True, padx=20, pady=20)
        
        # Header: Toggle Source (Synth vs SoundBank)
        source_frame = ctk.CTkFrame(main_frame, fg_color="#1a1a1a", height=60)
        source_frame.pack(fill="x", pady=(0, 20))
        source_frame.pack_propagate(False)
        
        source_var = ctk.StringVar(value="synth" if track.instrument_type == "synth" else "sf2")
        
        # Titre contextuel
        title_label = ctk.CTkLabel(
            source_frame, 
            text="SOURCE AUDIO :", 
            font=("Arial", 14, "bold"),
            text_color="#888888"
        )
        title_label.pack(side="left", padx=20)

        def switch_source(new_source):
            source_var.set(new_source)
            # Réinitialiser catégories si on passe au synthé
            if new_source == "synth":
                current_cat_var.set(list(self.preset_manager.presets.keys())[0] if self.preset_manager else "")
            refresh_ui()

        btn_synth = ctk.CTkButton(
            source_frame,
            text="🎹 SYNTHÉTISEUR",
            command=lambda: switch_source("synth"),
            fg_color="#1f6aa5" if source_var.get() == "synth" else "#333333",
            hover_color="#2b82c9",
            width=200,
            height=40,
            font=("Arial", 12, "bold")
        )
        btn_synth.pack(side="left", padx=10)

        btn_sf2 = ctk.CTkButton(
            source_frame,
            text="🎵 SOUNDBANK (SF2)",
            command=lambda: switch_source("sf2"),
            fg_color="#1f6aa5" if source_var.get() == "sf2" else "#333333",
            hover_color="#2b82c9",
            width=200,
            height=40,
            font=("Arial", 12, "bold"),
            state="normal" if self.sf2_instruments_data else "disabled"
        )
        btn_sf2.pack(side="left", padx=10)

        # Zone Recherche
        search_var = ctk.StringVar()
        search_entry = ctk.CTkEntry(
            source_frame,
            placeholder_text="🔍 Rechercher...",
            width=200,
            textvariable=search_var
        )
        search_entry.pack(side="right", padx=20)

        # Corps (Catégories + Grille)
        content_frame = ctk.CTkFrame(main_frame, fg_color="transparent")
        content_frame.pack(fill="both", expand=True)

        # Frame Catégories (visible seulement pour Synth)
        cat_scroll = ctk.CTkScrollableFrame(content_frame, width=150, fg_color="#252525")
        cat_scroll.pack(side="left", fill="y", padx=(0, 10))
        
        current_cat_var = ctk.StringVar()
        if track.instrument_type == "synth":
            current_cat_var.set(track.instrument_category)
        elif self.preset_manager:
            current_cat_var.set(list(self.preset_manager.presets.keys())[0])

        # Grille d'instruments
        grid_container = ctk.CTkScrollableFrame(content_frame, fg_color="#111111")
        grid_container.pack(side="right", fill="both", expand=True)

        def refresh_ui(*args):
            source = source_var.get()
            
            # Update buttons colors
            btn_synth.configure(fg_color="#1f6aa5" if source == "synth" else "#333333")
            btn_sf2.configure(fg_color="#1f6aa5" if source == "sf2" else "#333333")
            
            # Update categories list
            for widget in cat_scroll.winfo_children():
                widget.destroy()
            
            if source == "synth":
                cat_scroll.pack(side="left", fill="y", padx=(0, 10))
                if self.preset_manager:
                    for cat in self.preset_manager.presets.keys():
                        btn = ctk.CTkRadioButton(
                            cat_scroll, text=cat, variable=current_cat_var, value=cat,
                            command=refresh_grid, font=("Arial", 11)
                        )
                        btn.pack(pady=5, padx=5, anchor="w")
            else:
                cat_scroll.pack_forget() # Pas de catégories pour SF2 (tout en vrac ou par banque)
            
            refresh_grid()

        def refresh_grid(*args):
            for widget in grid_container.winfo_children():
                widget.destroy()
            
            source = source_var.get()
            search_query = search_var.get().lower()
            instruments = []

            if source == "sf2":
                for inst in self.sf2_instruments_data:
                    if search_query in inst['name'].lower():
                        instruments.append({**inst, 'type': 'sf2'})
            elif self.preset_manager:
                cat = current_cat_var.get()
                presets = self.preset_manager.get_presets_in_category(cat)
                for p_name in presets:
                    if search_query in p_name.lower():
                        instruments.append({'name': p_name, 'type': 'synth', 'cat': cat})
            
            for i, inst in enumerate(instruments):
                is_selected = (inst['name'] == track.instrument_name and inst['type'] == track.instrument_type)
                btn_color = "#1f6aa5" if is_selected else "#2a2a2a"
                
                def make_cmd(ins=inst):
                    if ins['type'] == 'sf2':
                        return lambda: select_sf2(ins['name'], ins['bank'], ins['preset'])
                    else:
                        return lambda: select_synth(ins['cat'], ins['name'])

                btn = ctk.CTkButton(
                    grid_container, text=inst['name'], fg_color=btn_color,
                    command=make_cmd(inst), height=35, font=("Arial", 11)
                )
                btn.grid(row=i // 3, column=i % 3, padx=5, pady=5, sticky="nsew")
            
            for i in range(3):
                grid_container.grid_columnconfigure(i, weight=1)

        def select_synth(cat, name):
            track.instrument_type = "synth"
            track.instrument_category = cat
            track.instrument_name = name
            finish()

        def select_sf2(name, bank, preset):
            track.instrument_type = "sf2"
            track.instrument_category = "SoundBank"
            track.instrument_name = name
            track.bank = bank
            track.preset = preset
            finish()

        def finish():
            self._refresh_track_list()
            selector_window.destroy()

        search_var.trace_add("write", refresh_grid)
        refresh_ui()
    
    # ===== CALLBACKS =====

    
    def _on_play(self):
        """Callback bouton Play."""
        if self.sequencer.state == TransportState.STOPPED or self.sequencer.state == TransportState.PAUSED:
            self.sequencer.play()
            self.play_button.configure(fg_color="#1a5a8f", text="▶ Playing")
    
    def _on_stop(self):
        """Callback bouton Stop."""
        self.sequencer.stop()
        self.sequencer.set_position(0.0)
        self.play_button.configure(fg_color="#1f6aa5", text="▶ Play")
        self.record_button.configure(fg_color="#c93232", text="⏺ Record")
        # Rafraîchir pour afficher l'état final des timelines
        self._refresh_timeline_view()
    
    def _on_record(self):
        """Callback bouton Record."""
        if self.sequencer.state == TransportState.STOPPED or self.sequencer.state == TransportState.PAUSED:
            if self.sequencer.start_recording():
                self.record_button.configure(fg_color="#ff4444", text="⏺ Recording")
                self.play_button.configure(fg_color="#1a5a8f", text="▶ Playing")
            else:
                # Afficher un message d'erreur si aucune piste armée
                pass
        else:
            self.sequencer.stop_recording()
            self.record_button.configure(fg_color="#c93232", text="⏺ Record")
            # Rafraîchir pour afficher toutes les notes enregistrées
            self._refresh_timeline_view()
    
    def _on_panic(self):
        """Callback bouton PANIC."""
        self.sequencer.all_notes_off()
    
    def _on_bpm_changed(self):
        """Callback changement de BPM."""
        try:
            bpm = int(self.bpm_var.get())
            self.sequencer.set_bpm(bpm)
        except ValueError:
            self.bpm_var.set(str(self.sequencer.bpm))
    
    def _on_metronome_toggle(self):
        """Callback toggle métronome."""
        self.sequencer.metronome_enabled = self.metronome_var.get()
    
    def _on_quantize_toggle(self):
        """Callback toggle quantification."""
        self.sequencer.quantize_enabled = self.quantize_var.get()
    
    def _on_quantize_value_changed(self, value):
        """Callback changement valeur de quantification."""
        self.sequencer.quantize_value = int(value)
    
    def _on_add_track(self):
        """Callback ajout de piste."""
        self.sequencer.add_track()
        self._refresh_track_list()
        self._refresh_timeline_view()
    
    def _on_delete_track(self, idx: int):
        """Callback suppression de piste."""
        self.sequencer.remove_track(idx)
        self._refresh_track_list()
        self._refresh_timeline_view()
    
    def _on_arm_track(self, idx: int):
        """Callback armer une piste pour enregistrement."""
        print(f"🎯 Tentative d'armement de la piste {idx+1}")
        # Désarmer toutes les pistes
        for track in self.sequencer.tracks:
            track.armed = False
        # Armer la piste sélectionnée
        if idx < len(self.sequencer.tracks):
            self.sequencer.tracks[idx].armed = True
            print(f"✅ Piste {idx+1} armée: {self.sequencer.tracks[idx].armed}")
        self._refresh_track_list()
    
    def _on_solo_track(self, idx: int):
        """Callback solo d'une piste."""
        if idx < len(self.sequencer.tracks):
            self.sequencer.tracks[idx].solo = not self.sequencer.tracks[idx].solo
            self._refresh_track_list()
    
    def _on_mute_track(self, idx: int):
        """Callback mute d'une piste."""
        if idx < len(self.sequencer.tracks):
            self.sequencer.tracks[idx].mute = not self.sequencer.tracks[idx].mute
            self._refresh_track_list()
    
    def _on_track_volume_changed(self, idx: int, volume: float):
        """Callback changement de volume d'une piste."""
        if idx < len(self.sequencer.tracks):
            self.sequencer.tracks[idx].volume = volume
    
    def _on_track_double_click(self, idx: int):
        """Callback double-clic sur timeline de piste."""
        print(f"🎹 Double-clic sur piste {idx+1} - Piano Roll à implémenter")
        # TODO: Ouvrir piano roll pour cette piste
    
    # ===== MISE À UPDATE UI =====
    
    def _start_ui_updates(self):
        """Démarre les mises à jour périodiques de l'UI."""
        self._update_position()
        self._update_timelines()
    
    def _update_position(self):
        """Met à jour l'affichage de la position."""
        if self.sequencer:
            position = self.sequencer.current_position
            self.position_label.configure(text=f"{position:.2f}")
        
        # Rappeler cette fonction après 50ms
        self.after(50, self._update_position)
    
    def _update_timelines(self):
        """Met à jour les timelines de toutes les pistes (optimisé)."""
        # Rafraîchir uniquement si en enregistrement
        if self.sequencer and self.sequencer.state.value == "recording":
            # Ne redessiner QUE les canvas, pas tout recréer!
            if hasattr(self, 'timeline_canvases'):
                for idx, canvas in self.timeline_canvases.items():
                    if idx < len(self.sequencer.tracks):
                        track = self.sequencer.tracks[idx]
                        self._draw_timeline(canvas, track)
        
        # Rappeler cette fonction après 1000ms (1 seconde, moins fréquent)
        self.after(1000, self._update_timelines)
    
    def set_note_callbacks(self, note_on_callback, note_off_callback):
        """Définit les callbacks pour envoyer les notes aux moteurs de synthèse."""
        self.sequencer.note_on_callback = note_on_callback
        self.sequencer.note_off_callback = note_off_callback
