"""
Moteur de synthèse basé sur FluidSynth pour le support des SoundFonts (.sf2).
"""

import numpy as np
import fluidsynth
import time

class FluidSynthEngine:
    """
    Moteur de synthèse utilisant FluidSynth.
    """
    
    def __init__(self, sample_rate=44100):
        """
        Initialise le moteur FluidSynth.
        
        Args:
            sample_rate: Taux d'échantillonnage
        """
        self.sample_rate = sample_rate
        self.fs = fluidsynth.Synth(samplerate=float(sample_rate))
        
        # État
        self.sf2_path = None
        self.sf_id = None
        self.current_preset = 0
        self.current_bank = 0
        self.master_volume = 0.6  # Volume par défaut identique au synthétiseur

        
        # FluidSynth n'a pas besoin d'une boucle audio ici car nous allons 
        # appeler get_samples manuellement dans AudioOutput.
        
    def load_soundfont(self, sf2_path):
        """
        Charge une banque de sons .sf2 ou .sf3.
        
        Args:
            sf2_path: Chemin vers le fichier .sf2 ou .sf3
            
        Returns:
            True si succès, False sinon
        """
        try:
            sf_id = self.fs.sfload(sf2_path)
            if sf_id == -1:
                print(f"Erreur : Impossible de charger le SoundFont {sf2_path}")
                return False
            
            self.sf2_path = sf2_path
            self.sf_id = sf_id
            
            # Sélectionner le premier instrument par défaut
            self.fs.program_select(0, sf_id, 0, 0)
            return True
        except Exception as e:
            print(f"Exception lors du chargement du SoundFont : {e}")
            return False
            
    def get_available_instruments(self):
        """
        Récupère la liste des instruments disponibles dans la banque chargée.
        Scanne le bank 0 (instruments) et le bank 128 (percussions).
        
        Returns:
            Liste de dictionnaires {bank, preset, name}
        """
        if self.sf_id is None:
            return []
            
        instruments = []
        
        # Scanner bank 0 (GM Instruments) et bank 128 (Drums)
        # On pourrait scanner plus, mais c'est le plus commun.
        for bank in [0, 128]:
            for preset in range(128):
                name = self.fs.sfpreset_name(self.sf_id, bank, preset)
                if name:
                    instruments.append({
                        'bank': bank,
                        'preset': preset,
                        'name': name
                    })
        
        return instruments

    def set_instrument(self, bank, preset):
        """
        Sélectionne un instrument.
        """
        if self.sf_id is not None:
            self.fs.program_select(0, self.sf_id, bank, preset)
            self.current_bank = bank
            self.current_preset = preset

    def note_on(self, midi_note, velocity=127):
        """Démarre une note."""
        self.fs.noteon(0, midi_note, velocity)

    def note_off(self, midi_note):
        """Arrête une note."""
        self.fs.noteoff(0, midi_note)

    def set_master_volume(self, volume):
        """Définit le volume master (0.0 à 1.0)."""
        self.master_volume = volume
        # Note: FluidSynth gère le gain via son propre réglage si on le souhaite
        # mais on peut aussi l'appliquer lors de la génération.

    def generate_audio(self, num_samples):
        """
        Génère les échantillons audio.
        Compatible avec l'interface attendue par AudioOutput.
        """
        # FluidSynth génère de la stéréo par défaut
        samples = self.fs.get_samples(num_samples)
        
        # pyfluidsynth retourne un tableau d'interleaved samples (L, R, L, R...)
        # Notre AudioOutput attend du mono qu'il convertit lui-même, 
        # ou un signal que nous pouvons adapter.
        
        # Pour rester compatible avec audio_output.py qui attend un array mono:
        # On fait la moyenne des deux canaux pour avoir du mono.
        samples_mono = (samples[0::2] + samples[1::2]) / 2.0
        
        # Appliquer le volume master avec un gain supplémentaire de 8x
        # pour compenser le volume plus faible de FluidSynth
        return (samples_mono / 32768.0) * self.master_volume * 8.0


    def panic(self):
        """Arrête toutes les notes."""
        for i in range(128):
            self.fs.noteoff(0, i)
            
    def close(self):
        """Libère les ressources."""
        self.fs.delete()
