"""
Moteur de synthèse principal avec gestion polyphonique des voix.
"""

import numpy as np
from waveforms import Oscillator, midi_to_frequency
from envelope import ADSREnvelope
from filters import LowPassFilter
from audio_effects import ChorusEffect, ReverbEffect

class Voice:
    """
    Représente une voix du synthétiseur (une note jouée).
    Combine oscillateur, enveloppe et filtre.
    """
    
    def __init__(self, sample_rate=44100):
        """
        Initialise une voix.
        
        Args:
            sample_rate: Taux d'échantillonnage
        """
        self.sample_rate = sample_rate
        
        # Composants de la voix
        self.oscillator = Oscillator(sample_rate)
        self.envelope = ADSREnvelope(sample_rate)
        self.filter = LowPassFilter(sample_rate)
        
        # État
        self.is_active = False
        self.midi_note = 0
        self.velocity = 0.0
    
    def note_on(self, midi_note, velocity=1.0):
        """
        Démarre une note.
        
        Args:
            midi_note: Numéro MIDI de la note (0-127)
            velocity: Vélocité (0.0 à 1.0)
        """
        self.midi_note = midi_note
        self.velocity = velocity
        
        # Configurer l'oscillateur
        frequency = midi_to_frequency(midi_note)
        self.oscillator.set_frequency(frequency)
        self.oscillator.reset_phase()
        
        # Démarrer l'enveloppe
        self.envelope.note_on()
        
        # Réinitialiser le filtre
        self.filter.reset()
        
        self.is_active = True
    
    def note_off(self):
        """Relâche la note (démarre le release)."""
        self.envelope.note_off()
    
    def is_playing(self):
        """Retourne True si la voix est encore active."""
        return self.is_active and self.envelope.is_active()
    
    def generate(self, num_samples):
        """
        Génère les échantillons audio pour cette voix.
        
        Args:
            num_samples: Nombre d'échantillons à générer
        
        Returns:
            numpy array des échantillons générés
        """
        if not self.is_active:
            return np.zeros(num_samples)
        
        # Générer la forme d'onde
        audio = self.oscillator.generate(num_samples)
        
        # Appliquer le filtre
        audio = self.filter.process(audio)
        
        # Appliquer l'enveloppe
        envelope_values = self.envelope.process(num_samples)
        audio = audio * envelope_values
        
        # Appliquer la vélocité
        audio = audio * self.velocity
        
        # Vérifier si la voix est toujours active
        if not self.envelope.is_active():
            self.is_active = False
        
        return audio
    
    def stop(self):
        """Arrête immédiatement la voix."""
        self.is_active = False
        self.envelope.reset()

class VoiceManager:
    """
    Gère l'allocation et la libération des voix pour la polyphonie.
    """
    
    def __init__(self, num_voices=32, sample_rate=44100):
        """
        Initialise le gestionnaire de voix.
        
        Args:
            num_voices: Nombre maximum de voix simultanées
            sample_rate: Taux d'échantillonnage
        """
        self.sample_rate = sample_rate
        self.voices = [Voice(sample_rate) for _ in range(num_voices)]
        self.note_to_voice = {}  # Mapping note MIDI -> index de voix
    
    def note_on(self, midi_note, velocity=1.0):
        """
        Démarre une note.
        
        Args:
            midi_note: Numéro MIDI de la note
            velocity: Vélocité (0.0 à 1.0)
        """
        # Si la note est déjà jouée, l'arrêter d'abord
        if midi_note in self.note_to_voice:
            voice_index = self.note_to_voice[midi_note]
            self.voices[voice_index].note_off()
        
        # Trouver une voix libre
        voice_index = self._find_free_voice()
        
        if voice_index is not None:
            self.voices[voice_index].note_on(midi_note, velocity)
            self.note_to_voice[midi_note] = voice_index
    
    def note_off(self, midi_note):
        """
        Relâche une note.
        
        Args:
            midi_note: Numéro MIDI de la note
        """
        if midi_note in self.note_to_voice:
            voice_index = self.note_to_voice[midi_note]
            self.voices[voice_index].note_off()
            # Ne pas retirer de note_to_voice tant que la voix est en release
    
    def _find_free_voice(self):
        """
        Trouve une voix libre ou en vole une.
        
        Returns:
            Index de la voix, ou None si aucune disponible
        """
        # D'abord chercher une voix inactive
        for i, voice in enumerate(self.voices):
            if not voice.is_playing():
                # Nettoyer le mapping si cette voix était assignée
                for note, v_idx in list(self.note_to_voice.items()):
                    if v_idx == i:
                        del self.note_to_voice[note]
                return i
        
        # Toutes les voix sont actives, voler la plus ancienne (index 0)
        # Dans une implémentation plus sophistiquée, on pourrait voler
        # celle avec l'enveloppe la plus faible
        return 0
    
    def generate(self, num_samples):
        """
        Génère le mix de toutes les voix actives.
        
        Args:
            num_samples: Nombre d'échantillons à générer
        
        Returns:
            numpy array du mix final
        """
        output = np.zeros(num_samples)
        
        # Sommer toutes les voix actives
        active_count = 0
        for voice in self.voices:
            if voice.is_playing():
                output += voice.generate(num_samples)
                active_count += 1
        
        # Normalisation pour éviter le clipping
        if active_count > 0:
            # Réduction proportionnelle au nombre de voix
            output = output / max(1.0, active_count * 0.5)
        
        return output
    
    def all_notes_off(self):
        """Arrête toutes les notes immédiatement."""
        for voice in self.voices:
            voice.note_off()
        self.note_to_voice.clear()

class SynthEngine:
    """
    Moteur de synthèse principal coordonnant tous les composants.
    """
    
    def __init__(self, sample_rate=44100, num_voices=32):
        """
        Initialise le moteur de synthèse.
        
        Args:
            sample_rate: Taux d'échantillonnage
            num_voices: Nombre de voix polyphoniques
        """
        self.sample_rate = sample_rate
        self.voice_manager = VoiceManager(num_voices, sample_rate)
        self.master_volume = 0.5
        
        # Effets audio
        self.chorus = ChorusEffect(sample_rate)
        self.reverb = ReverbEffect(sample_rate)
    
    def set_waveform(self, waveform):
        """
        Définit la forme d'onde pour toutes les voix.
        
        Args:
            waveform: 'sine', 'square', 'sawtooth', ou 'triangle'
        """
        for voice in self.voice_manager.voices:
            voice.oscillator.set_waveform(waveform)
    
    def set_adsr(self, attack, decay, sustain, release):
        """
        Définit les paramètres ADSR pour toutes les voix.
        
        Args:
            attack: Temps d'attaque en secondes
            decay: Temps de decay en secondes
            sustain: Niveau de sustain (0.0 à 1.0)
            release: Temps de release en secondes
        """
        for voice in self.voice_manager.voices:
            voice.envelope.set_attack(attack)
            voice.envelope.set_decay(decay)
            voice.envelope.set_sustain(sustain)
            voice.envelope.set_release(release)
    
    def set_filter(self, cutoff, resonance):
        """
        Définit les paramètres du filtre pour toutes les voix.
        
        Args:
            cutoff: Fréquence de coupure en Hz
            resonance: Résonance (Q factor)
        """
        for voice in self.voice_manager.voices:
            voice.filter.set_cutoff(cutoff)
            voice.filter.set_resonance(resonance)
    
    def set_master_volume(self, volume):
        """
        Définit le volume master.
        
        Args:
            volume: Volume (0.0 à 1.0)
        """
        self.master_volume = np.clip(volume, 0.0, 1.0)
    
    def note_on(self, midi_note, velocity=127):
        """
        Démarre une note.
        
        Args:
            midi_note: Numéro MIDI (0-127)
            velocity: Vélocité MIDI (0-127)
        """
        # Convertir la vélocité MIDI en 0.0-1.0
        velocity_normalized = velocity / 127.0
        self.voice_manager.note_on(midi_note, velocity_normalized)
    
    def note_off(self, midi_note):
        """
        Relâche une note.
        
        Args:
            midi_note: Numéro MIDI (0-127)
        """
        self.voice_manager.note_off(midi_note)
    
    def generate_audio(self, num_samples):
        """
        Génère les échantillons audio.
        
        Args:
            num_samples: Nombre d'échantillons à générer
        
        Returns:
            numpy array des échantillons (stéréo si nécessaire)
        """
        # Générer le mix de toutes les voix
        audio = self.voice_manager.generate(num_samples)
        
        # Appliquer le volume master
        audio = audio * self.master_volume
        
        # Chaîne d'effets
        audio = self.chorus.process(audio)
        audio = self.reverb.process(audio)
        
        # Clipping pour éviter la distorsion
        audio = np.clip(audio, -1.0, 1.0)
        
        return audio
    
    def panic(self):
        """Arrête toutes les notes (bouton de panique)."""
        self.voice_manager.all_notes_off()
    
    def get_active_voice_count(self):
        """Retourne le nombre de voix actuellement actives."""
        return sum(1 for voice in self.voice_manager.voices if voice.is_playing())
